/*  
 * 	hstruct/point.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h>
#include <float.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_hstruct_point.h>

struct _gdl_hstruct_point
{
	size_t n;
	gdl_boolean eligible;
	double x;
	double max;
	double min;
	double tol;
	double tot;
	size_t ntot;
	double (*scale)(double y, double a, double b);
	double (*iscale)(double y, double a, double b);
	double (*prior)(double x);
};

static double
_scale_pos_inf (double y, double a, double b)
{
	return exp(y)+a;
}

static double
_iscale_pos_inf (double y, double a, double b)
{
	return log(y-a);
}

static double
_scale_neg_inf (double y, double a, double b)
{
	return b-exp(y);
}

static double
_iscale_neg_inf (double y, double a, double b)
{
	return log(b-y);
}

static double
_scale_bounded (double y, double a, double b)
{
	return (b*exp(y)+a)/(1+exp(y));
}

static double
_iscale_bounded (double y, double a, double b)
{
	double z = (y-a)/(b-a);
	return log (z/(1-z));
}

gdl_hstruct_point *
gdl_hstruct_point_alloc (double x, double min, double max, double (*prior)(double x))
{
	gdl_hstruct_point * t = GDL_CALLOC (gdl_hstruct_point, 1);
	
	t->n = 0;
	t->eligible = gdl_true;
	t->tol      = 1.e-5;
	t->x = x;
	t->max = max;
	t->min = min;
	t->prior = prior;
	
	if (t->min == GDL_NEGINF && t->max == GDL_POSINF)
	{
		t->scale = NULL;
		t->iscale = NULL;
	}
	else if (t->min != GDL_NEGINF && t->max == GDL_POSINF)
	{
		t->scale = &_scale_pos_inf;
		t->iscale = &_iscale_pos_inf;
	}
	else if (t->min == GDL_NEGINF && t->max != GDL_POSINF)
	{
		t->scale = &_scale_neg_inf;	
		t->iscale = &_iscale_neg_inf;
	}
	else
	{
		t->scale = &_scale_bounded;
		t->iscale = &_iscale_bounded;
	}
	
	return t;
}

void
gdl_hstruct_point_free (gdl_hstruct_point * p)
{
	GDL_FREE (p);	
}

void
gdl_hstruct_point_vfree (void * p)
{
	GDL_FREE (p);	
}

gdl_hstruct_point *
gdl_hstruct_point_clone (const gdl_hstruct_point * t)
{
	gdl_hstruct_point * c = NULL;
	
	if (t)
	{
		c           = gdl_hstruct_point_alloc (t->x, t->min, t->max, t->prior);
		c->n        = t->n;
		c->eligible = t->eligible;
		c->tol      = t->tol;
	}
	
	return c;
}

int
gdl_hstruct_point_copy (gdl_hstruct_point * dest, const gdl_hstruct_point * src)
{
	if (dest && src)
	{
		dest->x      = src->x;
		dest->min    =	src->min;
		dest->max    =	src->max;
		if (src->prior)
		{
			dest->prior  = src->prior;
		}
		dest->scale  = src->scale;
		dest->iscale = src->iscale;
		dest->n      = src->n;
		dest->eligible = src->eligible;
		dest->tol = src->tol;
	}	
}

double
gdl_hstruct_point_value (const gdl_hstruct_point * t)
{
	return t->x;
}

gdl_boolean
gdl_hstruct_point_is_eligible (const gdl_hstruct_point * t)
{
	return t->eligible;
}

void
gdl_hstruct_point_set_eligible (gdl_hstruct_point * t, gdl_boolean b)
{
	t->eligible = b;
}

double
gdl_hstruct_point_rng_value (const gdl_hstruct_point * t, const gdl_rng * rng)
{
	if (t->min == GDL_NEGINF && t->max == GDL_POSINF)
	{
		return log (1.0/gdl_rng_uniform (rng)-1);
	}
	else if (t->min != GDL_NEGINF && t->max == GDL_POSINF)
	{
		return t->min - log (gdl_rng_uniform (rng));
	}
	else if (t->min == GDL_NEGINF && t->max != GDL_POSINF)
	{
		return t->max + log (gdl_rng_uniform (rng));
	}
	else
	{
		return gdl_ran_flat (rng, t->min, t->max);
	}	
}

double
gdl_hstruct_point_update (gdl_hstruct_point * t, double x)
{
	double err = t->x - x;
	t->x = x;
	return err;
}

double
gdl_hstruct_point_update_scale (gdl_hstruct_point * t, double y)
{
	double err;
	
	if (t->scale==0)
	{
		err = t->x - y;
		t->x = y;
	}
	else
	{
		err = t->x;
		t->x = (t->scale)(y, t->min, t->max);
		err -= t->x;
	}
	
	return err;
}

double
gdl_hstruct_point_prior (const gdl_hstruct_point * t)
{
	if (t->prior)
	{
		return (t->prior)(t->x);
	}
	return 1.0;
}

double
gdl_hstruct_point_scale (gdl_hstruct_point * t, double x)
{
	if (t->iscale==0)
	{
		return x;
	}
	else
	{
		return (t->iscale)(x, t->min, t->max);
	}
}

void
gdl_hstruct_point_attach (gdl_hstruct_point * t)
{
	t->n++;
}

void
gdl_hstruct_point_detach (gdl_hstruct_point * t)
{
	if (t->n) t->n--;
}

size_t
gdl_hstruct_point_inuse (const gdl_hstruct_point * t)
{
	if (t)
	{
		return t->n;
	}
	else
	{
		return 0;	
	}
}

void
gdl_hstruct_point_set_golden_tolerance (gdl_hstruct_point * t, double tol)
{
	t->tol = tol;	
}

double
gdl_hstruct_point_get_golden_tolerance (const gdl_hstruct_point * t)
{
	return t->tol;	
}

gdl_hstruct_point *
gdl_hstruct_point_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		double x, min, max;
		gdl_hstruct_point * p;
		
		status = fread (&x, sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&min, sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&max, sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		p = gdl_hstruct_point_alloc (x, min, max, NULL);
		
		status = fread (&(p->tol), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(p->n), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(p->eligible), sizeof (gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);		
		
		return p;
	}
	
	return NULL;
}

int
gdl_hstruct_point_fwrite (FILE * stream, const gdl_hstruct_point * p)
{
	if (stream && p)
	{
		int status;
		
		status = fwrite (&(p->x), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->min), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->max), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->tol), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->n), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->eligible), sizeof (gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
	
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;		
}

gdl_hstruct_point *
gdl_hstruct_point_sscanf (const gdl_string * str)
{
	if (!str)
	{
		return NULL;
	}
	size_t i, j, n;
	gdl_string * buf;
	gdl_hstruct_point * p = 0;

	n   = strlen (str);
	buf = gdl_string_alloc (n);
	
	for (i=0;isspace(str[i]) && i < n;i++);
	if (i==n) {gdl_string_free (buf);GDL_ERROR_VAL (gdl_string_sprintf ("Invalid input string [ %s ]", str), GDL_EINVAL, 0);}
	for (j=i;!isspace(str[j]) && j < n;j++);
	strncpy (buf, &str[i], j-i);
	
	p = gdl_hstruct_point_alloc ((double)atof(buf), 0, GDL_POSINF, 0);
	
	gdl_string_free (buf);
	
	return p;
}

void
gdl_hstruct_point_collect (gdl_hstruct_point * t, double x)
{
	t->tot+=x;
	t->ntot++;
}

#include <gdl/gdl_randist.h>

double
gdl_hstruct_point_hot_prior (double x)
{
	return gdl_ran_gaussian_pdf ((log (x)/log (10) - 1), 0.5);
}

double
gdl_hstruct_point_mut_prior (double x)
{
	return (x < 1.e-5) ? 0.999 : 0.001;
}
