/*  
 * 	hstruct/result.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_hview_wrapper.h>
#include <gdl/gdl_hmap.h>
#include <gdl/gdl_hstruct_point.h>
#include <gdl/gdl_hstruct_config.h>
#include <gdl/gdl_hstruct_model_chromosome.h>
#include <gdl/gdl_hstruct_model_type.h>
#include <gdl/gdl_hstruct_model.h>
#include <gdl/gdl_hstruct_parameters.h>
#include <gdl/gdl_hstruct_block.h>
#include <gdl/gdl_hstruct_result.h>

typedef struct
{
	gdl_hashtable ** mutation;
} gdl_hstruct_mutation_site;

static gdl_hstruct_mutation_site *
gdl_hstruct_mutation_site_alloc (size_t k)
{
	size_t i;
	gdl_hstruct_mutation_site * m;
	
	m = GDL_MALLOC (gdl_hstruct_mutation_site, 1);
	
	m->mutation = GDL_CALLOC (gdl_hashtable *, k);
	
	return m;	
}

static void
gdl_hstruct_mutation_site_free (gdl_hstruct_mutation_site * s, size_t k)
{
	if (s)
	{
		size_t i;
		for (i = 0; i < k; i++)
		{
			 gdl_hashtable_free (s->mutation[i]);
		}
		GDL_FREE (s->mutation);	
		GDL_FREE (s);
	}	
}

static int
gdl_hstruct_mutation_site_add (gdl_hstruct_mutation_site * c, size_t k, size_t allele)
{
	if (!c->mutation[k])
	{
		c->mutation[k] = gdl_hashtable_alloc (gdl_interface_uint, 5);
	}
	
	int status;
	gdl_string * key;
	size_t * ptr = GDL_MALLOC (size_t, 1);
	
	*ptr = allele;
	key  = gdl_string_sprintf ("%d", k);
	
	status = gdl_hashtable_update (c->mutation[k], key, ptr, 1);
	
	gdl_string_free (key);
	
	return status;
}

static gdl_vector_uint *
gdl_hstruct_mutation_site_get (const gdl_hstruct_mutation_site * c, size_t k)
{
	gdl_vector_uint * m = NULL;
	
	if (c->mutation[k])
	{
		size_t i = 0;
		gdl_hashtable_itr * itr;
		
		m = gdl_vector_uint_alloc (gdl_hashtable_size (c->mutation[k]));
		
		itr = gdl_hashtable_iterator (c->mutation[k]);
		
		do
		{
			const size_t * ptr = (size_t *) gdl_hashtable_iterator_value (itr);
			gdl_vector_uint_set (m, i, *ptr);
			i++;
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
	}
	
	return m;
}

typedef struct
{
	size_t n;
	size_t k;
	gdl_hstruct_mutation_site ** sites;
} gdl_hstruct_mutation_pattern;

static gdl_hstruct_mutation_pattern *
gdl_hstruct_mutation_pattern_alloc (size_t n, size_t k)
{
	size_t i;
	gdl_hstruct_mutation_pattern * m;
	
	m = GDL_MALLOC (gdl_hstruct_mutation_pattern, 1);
	
	m->sites = GDL_MALLOC (gdl_hstruct_mutation_site *, n);
	
	for (i = 0; i < n; i++)
	{
		m->sites[i] = gdl_hstruct_mutation_site_alloc (k);
	}
	
	return m;	
}

static void
gdl_hstruct_mutation_pattern_free (gdl_hstruct_mutation_pattern * p)
{
	if (p)
	{
		size_t i;
		for (i = 0; i < p->n; i++)
		{
			 gdl_hstruct_mutation_site_free (p->sites[i], p->k);
		}
		GDL_FREE (p->sites);
		GDL_FREE (p);
	}
}

static int
gdl_hstruct_mutation_pattern_add (gdl_hstruct_mutation_pattern * p, size_t l, size_t k, size_t a)
{
	return gdl_hstruct_mutation_site_add (p->sites[l], k, a);
}

static gdl_vector_uint *
gdl_hstruct_mutation_pattern_get (const gdl_hstruct_mutation_pattern * p, size_t l, size_t k)
{
	return gdl_hstruct_mutation_site_get (p->sites[l], k);
}

struct _gdl_hstruct_partition_result
{
	size_t fl;
	size_t np;
	size_t nl;
	size_t na;
	size_t nc;
	size_t kmax;
	const gdl_chromosome * chrom;
	gdl_mask             * gmask;
	gdl_clustering       * gclust;
	gdl_accession        ** accession;
	gdl_matrix           ** proba;
	gdl_vector_uint      ** best;
	gdl_vector_uint      ** mutation;
	gdl_vector            * recomb;
	gdl_matrix            * pi;
	gdl_gview_collector   * gmiss;
	gdl_hstruct_block_partition * ancestral;
};

struct _gdl_hstruct_chromosome_result
{
	size_t                        size;
	size_t                        kmax;
	const gdl_chromosome          * chrom;
	gdl_hstruct_partition_result ** results;
};

struct _gdl_hstruct_genome_result
{
	size_t                        size;
	size_t                        kmax;
	const gdl_genome              * genome;
	gdl_hstruct_chromosome_result ** results;
};

struct _gdl_hstruct_result
{
	size_t                                np;
	size_t                                kmax;
	double                                loglikelihood;
	gdl_hashtable                         * criterion;
	gdl_gmap                              * gmap;
	gdl_hstruct_static_parameter_registry * registry;
	gdl_hstruct_genome_result             ** results;
};

static void
_gdl_hstruct_result_set_ancestral (const gdl_hstruct_model_chromosome * MC, gdl_hstruct_static_config * s, size_t l)
{
	size_t k;
	
	for (k = 0; k < s->k; k++)
	{
		s->ancestral[k] = gdl_entity_clone (gdl_hstruct_model_chromosome_get_ancestral (MC, k, l));
	}
}



static gdl_hstruct_partition_result *
gdl_hstruct_partition_result_alloc (gdl_hstruct_result * r, const gdl_chromosome * chrom, const gdl_gview * gview, const gdl_hstruct_model_partition * MP)
{
	size_t i, ii, j, k, nc, p, b, nb;
	double * x, w;
	const gdl_hstruct_config  * conf, * pconf;
	gdl_hstruct_static_config * sconf;
	gdl_hstruct_hmm * hmm;
	gdl_hstruct_partition_result * pr;
	
	pr = GDL_CALLOC (gdl_hstruct_partition_result, 1);
	
	pr->chrom  = chrom;
	pr->gmask  = gdl_mask_clone (gdl_gview_wrapper_gmask (gdl_hstruct_model_partition_data (MP)));
	pr->gclust = gdl_clustering_clone (gdl_gview_wrapper_clustering (gdl_hstruct_model_partition_data (MP)));
	pr->gmiss  = gdl_gview_collector_clone (gdl_gview_wrapper_missing_sites (gdl_hstruct_model_partition_data (MP)));
	
	pr->nl = gdl_hstruct_model_partition_size (MP);
	pr->fl = gdl_chromosome_search (chrom, gdl_hstruct_model_partition_get_locus (MP, 0));
	pr->na        = gdl_clustering_size (pr->gclust);
	pr->accession = GDL_MALLOC (gdl_accession *, pr->na); 
	for (i = 0; i < pr->na; i++)
	{
		pr->accession[i] = gdl_entity_clone (GDL_GVIEW_GET_ACCESSION (gview, pr->gmask, i));
		gdl_entity_set_idx (pr->accession[i], i);
	}
	pr->nc = gdl_clustering_nclust (pr->gclust);
	pr->np = gdl_gview_ploidy (gview);
	pr->proba = GDL_MALLOC (gdl_matrix *, pr->nc*pr->np);
	pr->best  = GDL_MALLOC (gdl_vector_uint *, pr->nc*pr->np);
	pr->mutation = GDL_MALLOC (gdl_vector_uint *, pr->nc*pr->np);
	
	hmm = gdl_hstruct_model_partition_get_hmm (MP);
	gdl_hstruct_hmm_store_fb_probs (hmm);
	
	pr->ancestral = gdl_hstruct_block_partition_alloc (MP);
	pr->kmax      = gdl_hstruct_block_partition_kmax (pr->ancestral);
	
	pr->pi = gdl_matrix_calloc (pr->nl+1, pr->kmax);
		
	for (i = 0; i < pr->nc; i++)
	{
		ii = gdl_clustering_clust_idx (pr->gclust, i);
		w  = (double)gdl_clustering_clust_size (pr->gclust, i)/(double)pr->na;
		for (p = 0; p < pr->np; p++)
		{
			gdl_matrix_uint * mosaic      = gdl_hstruct_hmm_get_ancestral_mosaic (hmm, ii, p);
			gdl_vector_uint_view viterbi  = gdl_matrix_uint_column (mosaic, 0);
			gdl_vector_uint_view mutation = gdl_matrix_uint_column (mosaic, 1);
			pr->best[i*pr->np + p] = gdl_vector_uint_alloc (viterbi.vector.size);
			gdl_vector_uint_memcpy (pr->best[i*pr->np + p], &(viterbi.vector));
			pr->mutation[i*pr->np + p] = gdl_vector_uint_alloc (mutation.vector.size);
			gdl_vector_uint_memcpy (pr->mutation[i*pr->np + p], &(mutation.vector));
			
			gdl_vector_uint * path = gdl_hstruct_partition_result_accession_best_block_path (pr, ii, p);
			
			// collect mutation and compute diversity
			for (j = 0; j < pr->nl; j++)
			{
				if (gdl_vector_uint_get (&(mutation.vector), j))
				{
					const gdl_locus * locus = gdl_hstruct_partition_result_locus (pr, j);
					sconf = (gdl_hstruct_static_config *) locus->extra;
					gdl_hstruct_static_config_add_mutation (sconf, gdl_vector_uint_get (&(viterbi.vector), j), gdl_vector_uint_get (&(mutation.vector), j)-1);
					x = gdl_matrix_ptr (pr->pi, j+1, gdl_vector_uint_get (path, j));
					*x += w;
				}
				x = gdl_matrix_ptr (pr->pi, 0, gdl_vector_uint_get (path, j));
				*x += w;
			}
			gdl_vector_uint_free (path);
			// proba
			pr->proba[i*pr->np + p] = gdl_hstruct_hmm_get_ancestral_probs (hmm, ii, p);
			// clean
			gdl_matrix_uint_free (mosaic);
		}
	}
	for (k = 0; k < pr->kmax; k++)
	{
		w = 0;
		for (j = 0; j < pr->nl; j++)
		{
			w += gdl_matrix_get (pr->pi, j+1, k);
		}
		w /= gdl_matrix_get (pr->pi, 0, k);
		gdl_matrix_set (pr->pi, 0, k, w);
	}
	//pr->pi = gdl_hstruct_hmm_get_ancestral_intra_diveristy (hmm, pr->best);
	// Cumul proba of transition between ancestral haplotypes.
	pr->recomb = gdl_vector_calloc (pr->nl);
	for (i = 0; i < pr->nl-1; i++)
	{
		double rec = 0, tot;
		gdl_matrix * tr = gdl_hstruct_hmm_get_transition_matrix (hmm, i);
		for (ii = 0; ii < GDL_MIN (tr->size1, tr->size2); ii++)
		{
			rec += gdl_matrix_get (tr, ii, ii);
		}
		tot = gdl_matrix_sum_all (tr);
		gdl_vector_set (pr->recomb, i, 1.0-rec/tot);
		gdl_matrix_free (tr);
	}
	
	return pr;
}

static void
gdl_hstruct_partition_result_free (gdl_hstruct_partition_result *pr)
{
	if (pr)
	{
		size_t i, j;
		
		gdl_gview_collector_free (pr->gmiss);
		gdl_mask_free (pr->gmask);
		gdl_clustering_free (pr->gclust);
		for (i = 0; i < pr->na; i++)
		{
			if (i < pr->nc)
			{
				for (j = 0; j < pr->np; j++)
				{
					gdl_matrix_free (pr->proba[i*pr->np+j]);
					gdl_vector_uint_free (pr->best[i*pr->np+j]);	
					gdl_vector_uint_free (pr->mutation[i*pr->np+j]);
				}
			}
			gdl_accession_free (pr->accession[i]);	
		}
		gdl_vector_free (pr->recomb);
		gdl_hstruct_block_partition_free (pr->ancestral);
		GDL_FREE (pr->proba);
		GDL_FREE (pr->best);
		GDL_FREE (pr->accession);
		GDL_FREE (pr);
	}	
}

static gdl_hstruct_chromosome_result *
gdl_hstruct_chromosome_result_alloc (gdl_hstruct_result * r, const gdl_chromosome * chrom, const gdl_gview * gview, const gdl_hstruct_model_chromosome * MC)
{
	size_t i;
	gdl_hstruct_chromosome_result * cr;
	
	cr = GDL_MALLOC (gdl_hstruct_chromosome_result, 1);
	
	cr->size  = gdl_hstruct_model_chromosome_size (MC);
	cr->chrom = chrom;
	
	cr->results = GDL_MALLOC (gdl_hstruct_partition_result *, cr->size); 
	
	for (cr->kmax = i = 0; i < cr->size; i++)
	{
		const gdl_hstruct_model_partition * MP = gdl_hstruct_model_chromosome_get_partition (MC, i);
		cr->results[i] = gdl_hstruct_partition_result_alloc (r, chrom, gview, MP);
		if (cr->results[i]->kmax > cr->kmax)
		{
			cr->kmax = cr->results[i]->kmax;
		}
	}
	
	return cr;
}

static void
gdl_hstruct_chromosome_result_free (gdl_hstruct_chromosome_result * cr)
{
	if (cr)
	{
		size_t i;
		for (i = 0; i < cr->size; i++)
		{
			gdl_hstruct_partition_result_free (cr->results[i]);
		}
		GDL_FREE (cr->results);
		GDL_FREE (cr);
	}	
}

static gdl_hstruct_genome_result *
gdl_hstruct_genome_result_alloc (gdl_hstruct_result * r, const gdl_genome * genome, const gdl_gview * gview, const gdl_hstruct_model * M, size_t * s)
{
	size_t i;
	gdl_hstruct_genome_result * gr;
	
	gr = GDL_MALLOC (gdl_hstruct_genome_result, 1);
	
	gr->size   = gdl_genome_size (genome);
	gr->genome = genome;
	
	gr->results = GDL_MALLOC (gdl_hstruct_chromosome_result *, gr->size); 
	
	for (gr->kmax = 0, i = *s; i < *s + gr->size; i++)
	{
		const gdl_hstruct_model_chromosome * MC = gdl_hstruct_model_get_chromosome (M, i);
		const gdl_chromosome * chrom = gdl_genome_get (genome, i-(*s));
		gr->results[i-(*s)] = gdl_hstruct_chromosome_result_alloc (r, chrom, gview, MC);
		if (gr->results[i-(*s)]->kmax > gr->kmax)
		{
			gr->kmax = gr->results[i-(*s)]->kmax;
		}
	}
	
	return gr;
}

static void
gdl_hstruct_genome_result_free (gdl_hstruct_genome_result * gr)
{
	if (gr)
	{
		size_t i;
		for (i = 0; i < gr->size; i++)
		{
			gdl_hstruct_chromosome_result_free (gr->results[i]);
		}
		GDL_FREE (gr->results);
		GDL_FREE (gr);
	}	
}

static int
_gdl_hstruct_result_alloc (gdl_hstruct_result * r, const gdl_hstruct_model * M)
{
	size_t s, c, i, j, k, l, ng, nc, nl, na;
	gdl_hstruct_static_parameters * SP;
	gdl_hstruct_data             * D = gdl_hstruct_model_data (M);
	const gdl_gmap               * G = gdl_hstruct_model_locus_map (M);
	const gdl_hstruct_parameters * P = gdl_hstruct_model_parameters (M);
	const gdl_gview              * GV = gdl_hstruct_data_get_gview (D);
	const gdl_hstruct_model_chromosome * MC;
	const gdl_mask              * GM;
	const gdl_genome             * GENOME;
	
	SP = gdl_hstruct_static_parameters_alloc (P);
	
	r->np         = gdl_gview_ploidy (GV);
	r->registry   = gdl_hstruct_static_parameters_detach_registry (SP);
	r->gmap       = gdl_gmap_clone (G);
	
	ng = gdl_gmap_size (G);
	
	r->results    = GDL_MALLOC (gdl_hstruct_genome_result *, ng);
	
	for (r->kmax = s = c = i = 0; i < ng; i++)
	{
		GENOME = gdl_gmap_get (r->gmap, i);
		
		nc = gdl_gmap_genome_size (G, i);
				
		for (j = 0; j < nc; j++, c++)
		{
			MC = gdl_hstruct_model_get_chromosome (M, c);
			nl = gdl_gmap_chromosome_size (G, i, j);
			
			for (l = 0; l < nl; l++)
			{
				 const gdl_hstruct_config * co 
				    = gdl_hstruct_config_get (gdl_gmap_get_locus (G, i, j, l));
				 gdl_hstruct_static_config * s
				    = gdl_hstruct_static_parameters_new_config (SP, co);
				 gdl_locus * locus
				    = gdl_gmap_get_locus (r->gmap, i, j, l);
				 
				 _gdl_hstruct_result_set_ancestral (MC, s, l);
				 
				 locus->extra = s;				 
			}
		}
		
		r->results[i] = gdl_hstruct_genome_result_alloc (r, GENOME, GV, M, &s);
		
		if (r->results[i]->kmax > r->kmax)
		{
			r->kmax = r->results[i]->kmax;
		}
		
	}
	
	gdl_hstruct_static_parameters_free (SP);
}

static void
_gdl_hstruct_result_criterion (gdl_hstruct_result * r, const gdl_hstruct_workspace * w)
{
	double * value;
	
	r->loglikelihood = gdl_hstruct_workspace_loglikelihood (w);
	r->criterion     = gdl_hashtable_alloc (gdl_interface_double, 2);
	
	value  = GDL_MALLOC (double, 1);
	*value = gdl_hstruct_workspace_criterion (w, gdl_hstruct_criterion_BIC);
	gdl_hashtable_add (r->criterion, gdl_hstruct_criterion_BIC->acronym, value, 1);
	value  = GDL_MALLOC (double, 1);
	*value = gdl_hstruct_workspace_criterion (w, gdl_hstruct_criterion_AIC);
	gdl_hashtable_add (r->criterion, gdl_hstruct_criterion_AIC->acronym, value, 1);
	
}

gdl_hstruct_result *
gdl_hstruct_result_alloc (const gdl_hstruct_workspace * w)
{
	const gdl_hstruct_model * M;
	gdl_hstruct_result * r;
	
	r = GDL_MALLOC (gdl_hstruct_result, 1);
	
	M = gdl_hstruct_workspace_model (w);
	
	_gdl_hstruct_result_alloc (r, M);
	
	_gdl_hstruct_result_criterion (r, w);
		
	return r;	
}

void
gdl_hstruct_result_free (gdl_hstruct_result * r)
{
	if (r)
	{
		size_t i, ng;
		ng = gdl_gmap_size (r->gmap);
		for (i = 0; i < ng; i++)
		{
			gdl_hstruct_genome_result_free (r->results[i]);
		}
		gdl_hstruct_static_parameter_registry_free (r->registry);
		gdl_gmap_free (r->gmap);
		gdl_hashtable_free (r->criterion);
		GDL_FREE (r);
	}	
}

const gdl_gmap *
gdl_hstruct_result_gmap (const gdl_hstruct_result * r)
{
	return r->gmap;	
}

size_t
gdl_hstruct_result_ploidy (const gdl_hstruct_result * r)
{
	return r->np;	
}

size_t
gdl_hstruct_result_kmax (const gdl_hstruct_result * r)
{
	return r->kmax;	
}

size_t
gdl_hstruct_result_size (const gdl_hstruct_result * r)
{
	return gdl_gmap_size (r->gmap);	
}

const gdl_hstruct_genome_result *
gdl_hstruct_result_genome (const gdl_hstruct_result * r, size_t i)
{
	return r->results[i];	
}

const gdl_hstruct_static_parameter_registry *
gdl_hstruct_result_parameter_registry (const gdl_hstruct_result * r)
{
	return r->registry;
}

size_t
gdl_hstruct_genome_result_size (const gdl_hstruct_genome_result * gr)
{
	return gr->size;
}

const gdl_genome *
gdl_hstruct_genome_result_genome (const gdl_hstruct_genome_result * gr)
{
	return gr->genome;	
}

const gdl_hstruct_chromosome_result *
gdl_hstruct_genome_result_chromosome (const gdl_hstruct_genome_result * gr, size_t i)
{
	return gr->results[i];	
}

size_t
gdl_hstruct_chromosome_result_size (const gdl_hstruct_chromosome_result *cr)
{
	return cr->size;	
}

const gdl_chromosome *
gdl_hstruct_chromosome_result_chromosome (const gdl_hstruct_chromosome_result * cr)
{
	return cr->chrom;	
}

const gdl_hstruct_partition_result *
gdl_hstruct_chromosome_result_partition (const gdl_hstruct_chromosome_result *cr, size_t i)
{
	return cr->results[i];	
}

size_t
gdl_hstruct_partition_result_accession_size (const gdl_hstruct_partition_result * pr)
{
	return pr->na;	
}

size_t
gdl_hstruct_partition_result_locus_size (const gdl_hstruct_partition_result * pr)
{
	return pr->nl;	
}

size_t
gdl_hstruct_partition_result_ancestral_size (const gdl_hstruct_partition_result * pr)
{
	return pr->kmax;	
}

static const gdl_hstruct_static_config *
_gdl_hstruct_partition_result_static_config (const gdl_hstruct_partition_result * pr, size_t l)
{
	gdl_locus * locus = gdl_chromosome_get (pr->chrom, pr->fl + l);
	return (gdl_hstruct_static_config *)locus->extra;
}

size_t
gdl_hstruct_partition_result_ancestral_locus_size (const gdl_hstruct_partition_result * pr, size_t l)
{
	const gdl_hstruct_static_config * c = _gdl_hstruct_partition_result_static_config (pr, l);
	return c->k;
}

const gdl_mask * 
gdl_hstruct_partition_result_gmask (const gdl_hstruct_partition_result * pr)
{
	return pr->gmask;	
}

const gdl_clustering *
gdl_hstruct_partition_result_gclust (const gdl_hstruct_partition_result * pr)
{
	return pr->gclust;	
}

const gdl_chromosome *
gdl_hstruct_partition_result_chromosome (const gdl_hstruct_partition_result * pr)
{
	return pr->chrom;	
}

const gdl_locus *
gdl_hstruct_partition_result_locus (const gdl_hstruct_partition_result * pr, size_t i)
{
	return gdl_chromosome_get (pr->chrom, pr->fl + i);	
}

const gdl_accession *
gdl_hstruct_partition_result_accession (const gdl_hstruct_partition_result * pr, size_t i)
{
	return pr->accession[i];	
}

const gdl_accession *
gdl_hstruct_partition_result_search_accession (const gdl_hstruct_partition_result * pr, const gdl_string * name, gdl_hashtable ** search)
{
	gdl_hashtable * table = *search;
	if (!table)
	{
		size_t i;
		table = gdl_hashtable_alloc (gdl_hash_default, pr->na);
		for (i = 0; i < pr->na; i++)
		{
			gdl_hashtable_add (table, gdl_entity_get_name (pr->accession[i]), pr->accession[i], 0);
		}
		*search=table;
	}
	return gdl_hashtable_lookup (table, name);  
}

const gdl_allele *
gdl_hstruct_partition_result_ancestral_allele (const gdl_hstruct_partition_result * pr, size_t k, size_t l)
{
	const gdl_hstruct_static_config * c = _gdl_hstruct_partition_result_static_config (pr, l);
	return c->ancestral[k];	
}

const gdl_gview_collector *
gdl_hstruct_partition_result_gmiss (const gdl_hstruct_partition_result * pr)
{
	return pr->gmiss;	
}

const gdl_hstruct_block_partition *
gdl_hstruct_partition_result_block (const gdl_hstruct_partition_result * pr)
{
	return pr->ancestral;	
}

gdl_gdistance *
gdl_hstruct_partition_result_interval_length (const gdl_hstruct_partition_result * pr, size_t j)
{
	if (j < pr->nl - 1)
	{
		return gdl_chromosome_get_distance (pr->chrom, j, j+1);
	}
	else
	{
		return NULL;	
	}
}

double
gdl_hstruct_partition_result_ancestral_accession_proba (const gdl_hstruct_partition_result * pr, size_t k, size_t a, size_t p, size_t l)
{
	size_t aa = gdl_clustering_cluster (pr->gclust, a);
	return gdl_matrix_get (pr->proba[(pr->np)*aa + p], l, k);
}

size_t
gdl_hstruct_partition_result_ancestral_accession_best (const gdl_hstruct_partition_result * pr, size_t a, size_t p, size_t l)
{
	size_t aa = gdl_clustering_cluster (pr->gclust, a);
	return gdl_vector_uint_get (pr->best[(pr->np)*aa + p], l);
}

size_t
gdl_hstruct_partition_result_ancestral_accession_best_max (const gdl_hstruct_partition_result * pr, size_t a)
{
	if (pr->kmax > 1)
	{
		size_t i, p, k, * t, maxk;
		gdl_vector_int * max;
		
		max = gdl_vector_int_alloc (pr->kmax);
		
		for (i = 0; i < pr->nl; i++)
		{
			for (p = 0; p < pr->np; p++)
			{
				k = gdl_hstruct_partition_result_ancestral_accession_best (pr, a, p, i);
				t = gdl_vector_int_ptr (max, k);
				(*t)++;
			}
		}
		
		maxk = gdl_vector_int_max_index (max);
		
		gdl_vector_int_free (max);
		
		return maxk;
	}
	else
	{
		return 0;	
	}
}

double
gdl_hstruct_partition_result_ancestral_accession_proba_cumul (const gdl_hstruct_partition_result * pr, size_t k, size_t a)
{
	size_t l, p;
	double u = 0;
	
	for (l = 0; l < pr->nl; l++)
	{
		for (p = 0; p < pr->np; p++)
		{
			u += gdl_hstruct_partition_result_ancestral_accession_proba (pr, k, a, p, l);
		}	
	}
	
	return u/(pr->nl*pr->np);
}

double
gdl_hstruct_partition_result_ancestral_accession_best_cumul (const gdl_hstruct_partition_result * pr, size_t k, size_t a)
{
	size_t l, p;
	double u = 0;
	
	for (l = 0; l < pr->nl; l++)
	{
		for (p = 0; p < pr->np; p++)
		{
			if (gdl_hstruct_partition_result_ancestral_accession_best (pr, a, p, l)==k)
			{
				u+=1.0;
			}
		}	
	}
	
	return u/(pr->nl*pr->np);
}

double
gdl_hstruct_partition_result_ancestral_proba (const gdl_hstruct_partition_result * pr, size_t k, size_t l)
{
	if (pr->kmax > 1)
	{
		size_t i, p, nc;
		double u = 0;
		
		for (i = 0; i < pr->nc; i++)
		{
			nc = gdl_clustering_clust_size (pr->gclust, i);
			for (p = 0; p < pr->np; p++)
			{
				u += nc * gdl_matrix_get (pr->proba[(pr->np)*i + p], l, k);
			}
		}
		return u/(pr->na*pr->np);
	}
	else
	{
		return 1.0;	
	}
}

double
gdl_hstruct_partition_result_ancestral_best (const gdl_hstruct_partition_result * pr, size_t k, size_t l)
{
	if (pr->kmax > 1)
	{
		size_t i, p, nc;
		double u = 0;
		
		for (i = 0; i < pr->nc; i++)
		{
			nc = gdl_clustering_clust_size (pr->gclust, i);
			for (p = 0; p < pr->np; p++)
			{
				if (gdl_vector_uint_get (pr->best[(pr->np)*i + p], l) == k)
				{
					u += nc;
				}
			}
		}
		
		return u/(pr->na*pr->np);
	}
	else
	{
		return 1.0;	
	}
}


double
gdl_hstruct_partition_result_mutation_proba (const gdl_hstruct_partition_result * p, size_t k, size_t l)
{
	size_t b, nb, i, n;
	double w, pr = 0.0;
	const gdl_hstruct_block * block;
	
	nb = gdl_hstruct_block_partition_size (p->ancestral);
	
	for (b = 0; b < nb; b++)
	{
		block = gdl_hstruct_block_partition_get (p->ancestral, b);
		if (block->to >= l)
		{
			break;	
		}
	}
	
	for (n = i = 0; i < p->kmax; i++)
	{
		if (gdl_vector_uint_get (block->best, i) == k)
		{
			pr += gdl_matrix_get (p->pi, l+1, i);
			n++;
		}
	}
	
	return pr / n;
}

double
gdl_hstruct_partition_result_average_mutation_proba (const gdl_hstruct_partition_result * p, size_t l)
{
	size_t b, nb, k;
	double w, pr = 0.0;
	const gdl_hstruct_block * block;
	
	nb = gdl_hstruct_block_partition_size (p->ancestral);
	
	for (b = 0; b < nb; b++)
	{
		block = gdl_hstruct_block_partition_get (p->ancestral, b);
		if (block->to >= l)
		{
			break;	
		}
	}
	
	for (k = 0; k < p->kmax; k++)
	{
		pr += gdl_matrix_get (p->pi, l+1, k);
	}
	
	return pr / p->kmax;
}

double
gdl_hstruct_partition_result_ancestral_length (const gdl_hstruct_partition_result * pr, size_t k)
{
	size_t i, j, p, nc;
	double w, s = 0, l = 0, ss;
	
	for (i = 0; i < pr->nc; i++)
	{
		nc = gdl_clustering_clust_size (pr->gclust, i);
		for (p = 0; p < pr->np; p++)
		{
			for (ss = j = 0; j < pr->nl; j++)
			{
				w   = gdl_matrix_get (pr->proba[(pr->np)*i + p], j, k);
				ss += w;
			}
			s  += nc*ss;
			l  += nc*ss*ss;
		}
	}
	
	return (l/s)/(pr->nl);
}

double
gdl_hstruct_partition_result_ancestral_distance (const gdl_hstruct_partition_result * pr, size_t k, size_t kk)
{
	size_t l, b, nb, k1, k2;
	double d = 0;
	const gdl_hstruct_block * block;
	const gdl_hstruct_static_config * c;
	
	nb = gdl_hstruct_block_partition_size (pr->ancestral);
	for (b = 0; b < nb; b++)
	{
		block = gdl_hstruct_block_partition_get (pr->ancestral, b);
		for (l = block->from; l <= block->to; l++)
		{
			c  = gdl_hstruct_partition_result_config (pr, l);
			k1 = gdl_vector_uint_get (block->best, k);
			k2 = gdl_vector_uint_get (block->best, kk);
			if (c->ancestral[k1]->idx!=c->ancestral[k2]->idx)
			{
				d+=1.0;
			}
		}
	}
	
	return d;
}

double
gdl_hstruct_partition_result_ancestral_diversity (const gdl_hstruct_partition_result * pr, size_t l)
{
	size_t k, kk, u;
	double f1, f2, d = 0;
	const gdl_hstruct_static_config * c;
	const gdl_allele * a1, * a2;
	
	c = gdl_hstruct_partition_result_config (pr, l);
	
	for (k = 0; k < c->k; k++)
	{
		f1 = gdl_hstruct_partition_result_ancestral_proba (pr, k, l);
		a1  = gdl_hstruct_partition_result_ancestral_allele (pr, k, l);
		for (kk = 0; kk < c->k; kk++)
		{
			if (kk != k)
			{
				f2 = gdl_hstruct_partition_result_ancestral_proba (pr, kk, l);
				a2 = gdl_hstruct_partition_result_ancestral_allele (pr, kk, l);
				if (gdl_entity_get_idx (a1) != gdl_entity_get_idx (a2))
				{
					d += f1*f2;
				}
			}	
		}
	}
	
	return d;
}

gdl_boolean
gdl_hstruct_partition_result_accession_is_missing (const gdl_hstruct_partition_result * pr, size_t a, size_t p, size_t l)
{
	if (pr->gmiss)
	{
		size_t aa = gdl_clustering_cluster (pr->gclust, a);
		if (gdl_gview_collector_hget_idx (pr->gmiss, aa, l, p))
		{
			return gdl_true;	
		}
	}
	return gdl_false;	
}

gdl_boolean
gdl_hstruct_partition_result_accession_is_mutated (const gdl_hstruct_partition_result * pr, size_t a, size_t p, size_t l)
{
	size_t aa = gdl_clustering_cluster (pr->gclust, a);
	return (gdl_vector_uint_get (pr->mutation[(pr->np)*aa + p], l)) ? gdl_true : gdl_false;
}

gdl_vector_uint *
gdl_hstruct_partition_result_accession_best_block_path (const gdl_hstruct_partition_result * pr, size_t a, size_t p)
{
	int ii;
	size_t i, aa, l, b, nb, k1, k2;
	double d = 0;
	const gdl_hstruct_block * b1, * b2;
	const gdl_hstruct_static_config * c;
	gdl_vector_uint * path;
	
	path = gdl_vector_uint_alloc (pr->nl);
	
	aa = gdl_clustering_cluster (pr->gclust, a);
	
	gdl_vector_uint_memcpy (path, pr->best[(pr->np)*aa + p]);
	
	nb = gdl_hstruct_block_partition_size (pr->ancestral);
	
	for (b = 0; b < nb - 1; b++)
	{
		b1 = gdl_hstruct_block_partition_get (pr->ancestral, b);
		b2 = gdl_hstruct_block_partition_get (pr->ancestral, b+1);
		k2 = gdl_vector_uint_get (path, b2->from);
		k1 = gdl_vector_uint_get (path, b1->to);
		
		if (b1->k < b2->k)
		{
			if (gdl_vector_uint_get (b1->best, k2) == k1)
			{
				for (ii = b1->to; ii >= (int)b1->from; ii--)
				{
					i = ii;
					if (gdl_vector_uint_get (path, i) != k1)
					{
						break;	
					}
					gdl_vector_uint_set (path, i, k2);
				}	
			}
		}
		else if (b1->k > b2->k)
		{
			if (gdl_vector_uint_get (b2->best, k1) == k2)
			{
				for (i = b2->from; i <= b2->to; i++)
				{
					if (gdl_vector_uint_get (path, i) != k2)
					{
						break;	
					}
					gdl_vector_uint_set (path, i, k1);
				}	
			}
		}
	}
	
	return path;
}

gdl_hstruct_mutation_pattern *
gdl_hstruct_partition_result_mutation_pattern (const gdl_hstruct_partition_result * pr)
{
	size_t i, j, k, a, h;
	gdl_vector_uint * path;
	gdl_hstruct_mutation_pattern * mp;
	
	mp = gdl_hstruct_mutation_pattern_alloc (pr->nl, pr->kmax);

	for (i = 0; i < pr->nc; i++)
	{
		for (k = 0; k < pr->np; k++)
		{
			path = gdl_hstruct_partition_result_accession_best_block_path (pr, gdl_clustering_clust_idx (pr->gclust, i), k);
			for (j = 0; j < pr->nl; j++)
			{
				a = gdl_vector_uint_get (pr->mutation[(pr->np)*i + k], j);
				if (a)
				{
					a--;
					h = gdl_vector_uint_get (path, j);
					gdl_hstruct_mutation_pattern_add (mp, j, h, a);
				}
			}
			gdl_vector_uint_free (path);
		}	
	}
	
	return mp;
}

double
gdl_hstruct_result_loglikelihood (const gdl_hstruct_result * r)
{
	return r->loglikelihood;
}

double
gdl_hstruct_result_criterion (const gdl_hstruct_result * r, const gdl_hstruct_criterion_type * T)
{
	double * x = (double *) gdl_hashtable_lookup (r->criterion, T->acronym);
	if (x)
	{
		return *x;	
	}
	return 0;
}

void *
gdl_hstruct_result_parameter (const gdl_hstruct_result * r, const gdl_hstruct_point_type * T, size_t idx)
{
	return gdl_hstruct_static_parameter_registry_get (r->registry, T, idx);
}

const gdl_hstruct_static_config *
gdl_hstruct_partition_result_config (const gdl_hstruct_partition_result * pr, size_t i)
{
	const gdl_locus * locus = gdl_hstruct_partition_result_locus (pr, i);
	return (gdl_hstruct_static_config *) locus->extra;
}

double
gdl_hstruct_partition_result_ancestral_recomb (const gdl_hstruct_partition_result * pr, size_t l)
{
	return gdl_vector_get (pr->recomb, l);
}

size_t *
gdl_hstruct_partition_result_ancestral_accession_sort (const gdl_hstruct_partition_result * p)
{
	size_t i, * ii, j, k, maxk, nk, na, np, ng, nt, * idx, * sidx, * tidx;
	double x, * q;
	gdl_list     * group;
	gdl_list_itr * itr;
	
	na = gdl_hstruct_partition_result_accession_size (p);
	nk = p->kmax;
	nt = 0;
	tidx = GDL_MALLOC (size_t, na);
	
	for (j = 0; j < nk; j++)
	{
		group = gdl_list_alloc (gdl_interface_uint);
		for (i = 0; i < na; i++)
		{
			if (gdl_hstruct_partition_result_ancestral_accession_best_max (p, i) == j)
			{
				ii = GDL_MALLOC (size_t, 1);
				*ii = i;
				gdl_list_push_back (group, ii, 1);
			}
		}
		
		if ((ng=gdl_list_size (group))!=0)
		{
			q    = GDL_MALLOC (double, ng);
			idx  = GDL_MALLOC (size_t, ng);
			sidx = GDL_MALLOC (size_t, ng);
			
			i   = 0;
			itr = gdl_list_iterator_front (group);
			do
			{
				ii     = (size_t *) gdl_list_iterator_value (itr);
				idx[i] = *ii;
				q[i]   = gdl_hstruct_partition_result_ancestral_accession_best_cumul (p, j, idx[i]);
				i++;
			}
			while (gdl_list_iterator_next (itr));
			
			gdl_list_iterator_free (itr);
			
			gdl_sort_index (sidx, q, 1, ng);
			
			for (i = 0; i < ng/2; i++)
			{
				k = sidx[i];
				sidx[i]      = idx[sidx[ng-1-i]];
				sidx[ng-1-i] = idx[k];
			}
			if (ng % 2 == 1)
			{
				sidx[ng/2] = idx[sidx[ng/2]];	
			}
			
			memcpy (&tidx[nt], sidx, sizeof (size_t)*ng);
			
			nt += ng;

			GDL_FREE (q);
			GDL_FREE (idx);
			GDL_FREE (sidx);
		}
		
		gdl_list_free (group);
	}
	
	return tidx;
}

gdl_gmap *
gdl_hstruct_result_clone_gmap (const gdl_hstruct_result * r)
{
	size_t i, j, l, ng, nc, nl;
	gdl_gmap * gmap;	
	
	gmap = gdl_gmap_clone (r->gmap);
	
	ng = gdl_gmap_size (gmap);
	for (i = 0; i < ng; i++)
	{
		nc = gdl_gmap_genome_size (gmap, i);
		for (j = 0; j < nc; j++)
		{
			nl = gdl_gmap_chromosome_size (gmap, i, j);
			for (l = 0; l < nl; l++)
			{
				gdl_locus * l1 = gdl_gmap_get_locus (gmap, i, j, l);
				gdl_locus * l2 = gdl_gmap_get_locus (r->gmap, i, j, l);
				l1->extra = l2->extra;
			}
		}
	}
	
	return gmap;
}

static gdl_hstruct_partition_result *
gdl_hstruct_partition_result_fread (FILE * stream, const gdl_chromosome * chrom)
{
	if (stream && chrom)
	{
		int status;
		size_t i, j;
		gdl_hstruct_partition_result * pr;
		
		pr = GDL_MALLOC (gdl_hstruct_partition_result, 1);
		pr->chrom = chrom;
		
		status = fread (&(pr->fl), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(pr->np), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(pr->nl), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(pr->na), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(pr->nc), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(pr->kmax), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		pr->gmask = gdl_mask_fread (stream);
		GDL_FREAD_STATUS (pr->gmask!=0, 1);
		pr->gclust = gdl_clustering_fread (stream);
		GDL_FREAD_STATUS (pr->gclust!=0, 1);
		
		pr->accession = GDL_MALLOC (gdl_accession *, pr->na);
		pr->proba     = GDL_MALLOC (gdl_matrix *, pr->nc*pr->np);
		pr->best      = GDL_MALLOC (gdl_vector_uint *, pr->nc*pr->np);
		pr->mutation  = GDL_MALLOC (gdl_vector_uint *, pr->nc*pr->np);
		
		for (i = 0; i < pr->na; i++)
		{
			if (i < pr->nc)
			{
				for (j = 0; j < pr->np; j++)
				{
					pr->proba[i*pr->np + j] = gdl_matrix_fread (stream);
					GDL_FREAD_STATUS (pr->proba[i*pr->np + j]!=0, 1);
					pr->best[i*pr->np + j] = gdl_vector_uint_fread (stream);
					GDL_FREAD_STATUS (pr->best[i*pr->np + j]!=0, 1);
					pr->mutation[i*pr->np + j] = gdl_vector_uint_fread (stream);
					GDL_FREAD_STATUS (pr->mutation[i*pr->np + j]!=0, 1);
				}
			}
			pr->accession[i] = gdl_entity_fread (stream);
			GDL_FREAD_STATUS (pr->accession[i]!=0, 1);
		}
		
		pr->pi = gdl_matrix_fread (stream);
		GDL_FREAD_STATUS (pr->pi!=0, 1);
		
		pr->recomb = gdl_vector_fread (stream);
		GDL_FREAD_STATUS (pr->recomb!=0, 1);
		
		pr->gmiss = gdl_gview_collector_fread (stream);
		GDL_FREAD_STATUS (pr->gmiss!=0, 1);
		
		pr->ancestral = gdl_hstruct_block_partition_fread (stream);
		GDL_FREAD_STATUS (pr->ancestral!=0, 1);
		
		return pr;
	}
	
	return NULL;	
}

static gdl_hstruct_chromosome_result *
gdl_hstruct_chromosome_result_fread (FILE * stream, const gdl_chromosome * chrom)
{
	if (stream && chrom)
	{
		int status;
		size_t i;
		gdl_hstruct_chromosome_result * cr;
		
		cr = GDL_MALLOC (gdl_hstruct_chromosome_result, 1);
		cr->chrom = chrom;
		
		status = fread (&(cr->size), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(cr->kmax), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		cr->results = GDL_MALLOC (gdl_hstruct_partition_result *, cr->size);
		
		for (i = 0; i < cr->size; i++)
		{
			cr->results[i] = gdl_hstruct_partition_result_fread (stream, chrom);
			GDL_FREAD_STATUS (cr->results[i]!=0, 1);
		}
		
		// FREAD STATIC CONFIG
		for (i = 0; i < gdl_chromosome_size (chrom); i++)
		{
			gdl_locus * locus = gdl_chromosome_get (chrom, i);
			locus->extra = gdl_hstruct_static_config_fread (stream);	
			GDL_FREAD_STATUS (locus->extra!=0, 1);
		}
		
		return cr;
	}
	return NULL;
}

static gdl_hstruct_genome_result *
gdl_hstruct_genome_result_fread (FILE * stream, const gdl_genome * genome)
{
	if (stream && genome)
	{
		int status;
		size_t i;
		gdl_hstruct_genome_result * gr;
		
		gr = GDL_MALLOC (gdl_hstruct_genome_result, 1);
		gr->genome = genome;
		
		status = fread (&(gr->size), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(gr->kmax), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		gr->results = GDL_MALLOC (gdl_hstruct_chromosome_result *, gr->size);
		
		for (i = 0; i < gr->size; i++)
		{
			gr->results[i] = gdl_hstruct_chromosome_result_fread (stream, gdl_genome_get (genome, i));
			GDL_FREAD_STATUS (gr->results[i]!=0, 1);
		}
		
		return gr;
	}
	return NULL;
}

const gdl_hstruct_model_type *
gdl_hstruct_model_type_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_string * name;
		
		name = gdl_string_fread (stream);
		GDL_FREAD_STATUS (name!=0, 1);
		if (!strcmp (name, gdl_hstruct_model_gmap->name))
		{
			return gdl_hstruct_model_gmap;
		}
		else if (!strcmp (name, gdl_hstruct_model_hmap->name))
		{
			return gdl_hstruct_model_hmap;
		}
		else if (!strcmp (name, gdl_hstruct_model_hmap_unknown->name))
		{
			return gdl_hstruct_model_hmap_unknown;	
		}
	}
	return NULL;	
}

gdl_hstruct_result *
gdl_hstruct_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i, ng;
		gdl_hstruct_result * r;
		
		r = GDL_MALLOC (gdl_hstruct_result, 1);
		
		status  = fread (&(r->np), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status  = fread (&(r->kmax), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		r->gmap = gdl_gmap_fread (stream);
		GDL_FREAD_STATUS (r->gmap!=0, 1);
		r->registry = gdl_hstruct_static_parameter_registry_fread (stream);
		GDL_FREAD_STATUS (r->registry!=0, 1);
		status = fread (&(r->loglikelihood), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		r->criterion  = gdl_hashtable_alloc (gdl_interface_double, 0);
		status = gdl_hashtable_fread (stream, r->criterion);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		ng         = gdl_gmap_size (r->gmap);
		r->results = GDL_MALLOC (gdl_hstruct_genome_result *, ng);
		
		for (i = 0; i < ng; i++)
		{
			r->results[i] = gdl_hstruct_genome_result_fread (stream, gdl_gmap_get (r->gmap, i));
			GDL_FREAD_STATUS (r->results[i]!=0, 1);
			// FREAD THE STATIC CONFIG
		}
		
		return r;
	}
	
	return NULL;
}

static int
gdl_hstruct_partition_result_fwrite (FILE * stream, const gdl_hstruct_partition_result * pr)
{
	if (stream && pr)
	{
		int status;
		size_t i, j;
		
		status = fwrite (&(pr->fl), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(pr->np), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(pr->nl), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(pr->na), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(pr->nc), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(pr->kmax), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		status = gdl_mask_fwrite (stream, pr->gmask);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = gdl_clustering_fwrite (stream, pr->gclust);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		for (i = 0; i < pr->na; i++)
		{
			if (i < pr->nc)
			{
				for (j = 0; j < pr->np; j++)
				{
					status = gdl_matrix_fwrite (stream, pr->proba[i*pr->np + j]);
					GDL_FWRITE_STATUS (status, GDL_SUCCESS);
					status = gdl_vector_uint_fwrite (stream, pr->best[i*pr->np + j]);
					GDL_FWRITE_STATUS (status, GDL_SUCCESS);
					status = gdl_vector_uint_fwrite (stream, pr->mutation[i*pr->np + j]);
					GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				}
			}
			status = gdl_entity_fwrite (stream, pr->accession[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		status = gdl_matrix_fwrite (stream, pr->pi);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = gdl_vector_fwrite (stream, pr->recomb);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = gdl_gview_collector_fwrite (stream, pr->gmiss);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = gdl_hstruct_block_partition_fwrite (stream, pr->ancestral);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

static int
gdl_hstruct_chromosome_result_fwrite (FILE * stream, const gdl_hstruct_chromosome_result * cr)
{
	if (stream && cr)
	{
		int status;
		size_t i;
		
		status = fwrite (&(cr->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(cr->kmax), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < cr->size; i++)
		{
			status = gdl_hstruct_partition_result_fwrite (stream, cr->results[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		// FWRITE STATIC CONFIG
		for (i = 0; i < gdl_chromosome_size (cr->chrom); i++)
		{
			gdl_locus * locus = gdl_chromosome_get (cr->chrom, i);
			status = gdl_hstruct_static_config_fwrite (stream, (gdl_hstruct_static_config *)locus->extra);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

static int
gdl_hstruct_genome_result_fwrite (FILE * stream, const gdl_hstruct_genome_result * gr)
{
	if (stream && gr)
	{
		int status;
		size_t i;
		
		status = fwrite (&(gr->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(gr->kmax), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < gr->size; i++)
		{
			status = gdl_hstruct_chromosome_result_fwrite (stream, gr->results[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

int
gdl_hstruct_model_type_fwrite (FILE * stream, const gdl_hstruct_model_type * T)
{
	if (stream)
	{
		return gdl_string_fwrite (stream, T->name);
	}
	return GDL_EINVAL;
}

int
gdl_hstruct_result_fwrite (FILE * stream, const gdl_hstruct_result * r)
{
	if (stream && r)
	{
		int status;
		size_t i, ng;
		
		status  = fwrite (&(r->np), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status  = fwrite (&(r->kmax), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_gmap_fwrite (stream, r->gmap);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_hstruct_static_parameter_registry_fwrite (stream, r->registry);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = fwrite (&(r->loglikelihood), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_hashtable_fwrite (stream, r->criterion);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		ng = gdl_gmap_size (r->gmap);
		for (i = 0; i < ng; i++)
		{
			status = gdl_hstruct_genome_result_fwrite (stream, r->results[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			// FWRITE THE STATIC CONFIG
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

static void
_gdl_hstruct_result_point_fprintf (FILE * stream, const gdl_hstruct_result * r, const gdl_hstruct_point_type * T, int idx)
{
	if (idx < 0)
	{
		fprintf (stream, "\t-");
		return;
	}
	if (T == gdl_hstruct_point_f)
	{
		size_t k;
		gdl_hstruct_fpoint * fp;
		
		fp = (gdl_hstruct_fpoint *) gdl_hstruct_static_parameter_registry_get (r->registry, T, idx);
		
		for (k = 0; k < gdl_hstruct_fpoint_size (fp); k++)
		{
			fprintf (stream, "\t%1.3f", gdl_hstruct_fpoint_value (fp, k));
		}
	}
	else
	{
		gdl_hstruct_point * p;
		
		p = (gdl_hstruct_point *) gdl_hstruct_static_parameter_registry_get (r->registry, T, idx);
		
		if (T == gdl_hstruct_point_hot)
		{
			fprintf (stream, "\t%.1f", gdl_hstruct_point_value (p));
		}
		else
		{
			fprintf (stream, "\t%g", gdl_hstruct_point_value (p));	
		}
	}	
}

static void
_gdl_hstruct_result_criterion_fprintf (FILE * stream, const gdl_hstruct_result * r)
{
	gdl_hashtable_itr * itr;
	
	itr = gdl_hashtable_iterator (r->criterion);
	
	do
	{
		const gdl_string * acronym = gdl_hashtable_iterator_key (itr);
		const double * value = (double *) gdl_hashtable_iterator_value (itr);
		fprintf (stream, "\t%s = %.1f\n", acronym, *value);
	}
	while (gdl_hashtable_iterator_next (itr));
	
	gdl_hashtable_iterator_free (itr);
}

int
gdl_hstruct_result_fprintf (FILE * stream, const gdl_hstruct_result * r)
{
	if (stream && r)
	{
		size_t i, b, ii, ic, j, jj, k, kk, l, c, ng, nc, nl, np, nb;
			
		fprintf (stream, "Parameters:\n");
		fprintf (stream, "\tLn Proba of Data = %.1f\n", r->loglikelihood);
		_gdl_hstruct_result_criterion_fprintf (stream, r);
		fprintf (stream, "\n--------------------------------------------\n");
		fprintf (stream, "\nParameter Registry\n\n");
		np = gdl_hstruct_static_parameter_registry_size (r->registry, gdl_hstruct_point_rho);
		fprintf (stream, "Rho points (%d)\n", np);
		for (i = 0; i < np; i++)
		{
			fprintf (stream, "\t%d", i+1);
			_gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_rho, i);
			fprintf (stream, "\n");
		}
		np = gdl_hstruct_static_parameter_registry_size (r->registry, gdl_hstruct_point_mu);
		fprintf (stream, "\nMu points (%d)\n", np);
		for (i = 0; i < np; i++)
		{
			fprintf (stream, "\t%d", i+1);
			_gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_mu, i);
			fprintf (stream, "\n");
		}
		np = gdl_hstruct_static_parameter_registry_size (r->registry, gdl_hstruct_point_hot);
		fprintf (stream, "\nHot points (%d)\n", np);
		for (i = 0; i < np; i++)
		{
			fprintf (stream, "\t%d", i+1);
			_gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_hot, i);
			fprintf (stream, "\n");
		}
		np = gdl_hstruct_static_parameter_registry_size (r->registry, gdl_hstruct_point_f);
		fprintf (stream, "\nF points (%d)\n", np);
		for (i = 0; i < np; i++)
		{
			fprintf (stream, "\t%d", i+1);
			_gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_f, i);
			fprintf (stream, "\n");
		}
		fprintf (stream, "\n\n--------------------------------------------\n\n");
		fprintf (stream, "Map of the parameter indexes\n\n");
		fprintf (stream, "Genome\tChromosome\tLocus:ID\tLocus:Name\tPosition\tUnit\tK\tRho\tMu\tHot\tF-->\n");
		ng = gdl_gmap_size (r->gmap);
		for (i = 0; i < ng; i++)
		{
			nc = gdl_gmap_genome_size (r->gmap, i);
			for (j = 0; j < nc; j++)
			{
				nl = gdl_gmap_chromosome_size (r->gmap, i, j);
				for (l = 0; l < nl; l++)
				{
					 gdl_locus * locus       = gdl_gmap_get_locus (r->gmap, i, j, l);
					 const gdl_gdistance * d = gdl_gmap_get_distance (r->gmap, i, j, 0, l);
					 gdl_hstruct_static_config * s = (gdl_hstruct_static_config *) locus->extra;
				 	 fprintf (stream, "%d\t%d\t%d\t%s\t", i+1, j+1, l+1, gdl_entity_get_name (locus));
				 	 gdl_gdistance_fprintf (stream, d);
				 	 fprintf (stream, "\t%d\t%d\t%d\t%d\t%d\n", s->k, s->rho_idx+1, s->mu_idx+1, s->hot_idx+1, s->f_idx+1);					 
				}
			}
		}
		fprintf (stream, "\n\n--------------------------------------------\n\n");
		fprintf (stream, "Map of the parameter values\n\n");
		fprintf (stream, "Genome\tChromosome\tLocus:ID\tLocus:Name\tPosition\tUnit\tK\tRho\tMu\tHot\tF-->\n");
		ng = gdl_gmap_size (r->gmap);
		for (i = 0; i < ng; i++)
		{
			nc = gdl_gmap_genome_size (r->gmap, i);
			for (j = 0; j < nc; j++)
			{
				nl = gdl_gmap_chromosome_size (r->gmap, i, j);
				for (l = 0; l < nl; l++)
				{
					 gdl_locus * locus = gdl_gmap_get_locus (r->gmap, i, j, l);
					 const gdl_gdistance * d = gdl_gmap_get_distance (r->gmap, i, j, 0, l);
					 gdl_hstruct_static_config * s = (gdl_hstruct_static_config *) locus->extra;
					 fprintf (stream, "%d\t%d\t%d\t%s\t", i+1, j+1, l+1, gdl_entity_get_name (locus));
					 gdl_gdistance_fprintf (stream, d);
					 fprintf (stream, "\t%d", s->k);
					 if (l < nl - 1)
					 {
					 	_gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_rho, s->rho_idx);
					 }
					 else
					 {
					 	fprintf (stream, "\t-");
					 }
					 _gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_mu, s->mu_idx);
					 _gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_hot, s->hot_idx);
					 _gdl_hstruct_result_point_fprintf (stream, r, gdl_hstruct_point_f, s->f_idx);
					 fprintf (stream, "\n");
				}
			}
		}
		fprintf (stream, "\n\n--------------------------------------------\n\n");
		fprintf (stream, "Map of local statistics\n\n");
		fprintf (stream, "Genome\tChromosome\tLocus:ID\tLocus:Name\tPosition\tUnit\tK\tPr(rec:interval)\tPr(rec:unit)\tPr(mut)\tPr(ancestral)-->\n");
		ng = gdl_gmap_size (r->gmap);
		for (i = 0; i < ng; i++)
		{
			const gdl_hstruct_genome_result * gr = r->results[i];
			for (j = 0; j < nc; j++)
			{
				const gdl_hstruct_chromosome_result * cr = gr->results[j];
				for (jj = 0; jj < cr->size; jj++)
				{
					const gdl_hstruct_partition_result * pr = cr->results[jj];
					for (l = 0; l < pr->nl; l++)
					{
						const gdl_gdistance * d = gdl_gmap_get_distance (r->gmap, i, j, 0, l+pr->fl);
					 	gdl_locus * locus = gdl_gmap_get_locus (r->gmap, i, j, l+pr->fl);
					 	k = gdl_hstruct_partition_result_ancestral_locus_size (pr, l);
						fprintf (stream, "%d\t%d\t%d\t%s\t", i+1, j+1, l+1, gdl_entity_get_name (locus));
						gdl_gdistance_fprintf (stream, d);
					   fprintf (stream, "\t%d", k);
					   if (l < pr->nl - 1)
						{
							fprintf (stream, "\t%e", gdl_vector_get (pr->recomb, l));
							d = gdl_gmap_get_distance (r->gmap, i, j, l+pr->fl, l+pr->fl+1);
							if (d->value != 0.0)
							{
								fprintf (stream, "\t%e", gdl_vector_get (pr->recomb, l)/(d->value));
							}
							else
							{
								fprintf (stream, "\t-");
							}
						}
						else
						{
							fprintf (stream, "\t-\t-");
						}
						fprintf (stream, "\t%e", gdl_hstruct_partition_result_average_mutation_proba (pr, l));
						for (kk = 0; kk < k; kk++)
						{
							fprintf (stream, "\t%1.2f", gdl_hstruct_partition_result_ancestral_proba (pr, kk, l));
						}
						fprintf (stream, "\n");
					}
				}
			}
		}
		fprintf (stream, "\n\n--------------------------------------------\n\n");
		fprintf (stream, "Genome\tChromosome\tLocus:ID\tLocus:Name\t\tPosition\tUnit\tK\tPi\tPr(mut|K)-->\n");
		ng = gdl_gmap_size (r->gmap);
		for (i = 0; i < ng; i++)
		{
			const gdl_hstruct_genome_result * gr = r->results[i];
			for (j = 0; j < nc; j++)
			{
				const gdl_hstruct_chromosome_result * cr = gr->results[j];
				for (jj = 0; jj < cr->size; jj++)
				{
					const gdl_hstruct_partition_result * pr = cr->results[jj];
					for (l = 0; l < pr->nl; l++)
					{
						const gdl_gdistance * d = gdl_gmap_get_distance (r->gmap, i, j, 0, l+pr->fl);
					 	gdl_locus * locus = gdl_gmap_get_locus (r->gmap, i, j, l+pr->fl);
					 	
					 	k = gdl_hstruct_partition_result_ancestral_locus_size (pr, l);
						fprintf (stream, "%d\t%d\t%d\t%s\t", i+1, j+1, l+1, gdl_entity_get_name (locus));
						gdl_gdistance_fprintf (stream, d);
					   fprintf (stream, "\t%d", k);
					   fprintf (stream, "\t%1.5f", gdl_hstruct_partition_result_ancestral_diversity (pr, l));
					   for (kk = 0; kk < k; kk++)
						{
							fprintf (stream, "\t%e", gdl_hstruct_partition_result_mutation_proba (pr, kk, l));
						}
						fprintf (stream, "\n");
					}
				}
			}
		}
		fprintf (stream, "\n\n--------------------------------------------\n\n");
		fprintf (stream, "Map of the ancestral haplotypes\n\n");
		ng = gdl_gmap_size (r->gmap);
		for (i = 0; i < ng; i++)
		{
			const gdl_hstruct_genome_result * gr = r->results[i];
			for (j = 0; j < gr->size; j++)
			{
				const gdl_hstruct_chromosome_result * cr = gr->results[j];
				for (jj = 0; jj < cr->size; jj++)
				{
					const gdl_hstruct_partition_result * pr = cr->results[jj];
					fprintf (stream, "Genome %d Chromosome %d Partition %d\tkmax = %d\n", i+1, j+1, jj+1, pr->kmax);
					fprintf (stream, "--\nAncestral Haplotype Distance Matrix\n--\n");
					for (k = 0; k < pr->kmax; k++)
					{
						fprintf (stream, "%d", k+1);
						for (kk = 0; kk < pr->kmax; kk++)
						{
							if (k != kk)
							{
								fprintf (stream, "\t%.2f", gdl_hstruct_partition_result_ancestral_distance (pr, k, kk));
							}
							else
							{
								fprintf (stream, "\t-");	
							}
						}
						fprintf (stream, "\n");
					}
					fprintf (stream, "--\nAncestral Haplotype\tAverage Length\tIntra Diversity\n--\n");
					for (k = 0; k < pr->kmax; k++)
					{
						fprintf (stream, "%d\t%.1f\t%1.5f\n", k+1, gdl_hstruct_partition_result_ancestral_length (pr, k), gdl_matrix_get (pr->pi, 0, k));
					}
					fprintf (stream, "--\n");
					fprintf (stream, "Ancestral Haplotype\tSequence-->\n--\n");
					nb = gdl_hstruct_block_partition_size (pr->ancestral);
					for (k = 0; k < pr->kmax; k++)
					{
						fprintf (stream, "%d", k+1);
						for (b = 0; b < nb; b++)
						{
							const gdl_hstruct_block * block = gdl_hstruct_block_partition_get (pr->ancestral, b);
							for (l = block->from; l <= block->to; l++)
							{
								 const gdl_hstruct_static_config * s = _gdl_hstruct_partition_result_static_config (pr, l);
								 fprintf (stream, "\t%s", gdl_entity_get_name (s->ancestral[gdl_vector_uint_get (block->best, k)]));
							}	
						}						
						fprintf (stream, "\n");
					}
					fprintf (stream, "--\n");
					fprintf (stream, "Ancestral Haplotype\tMutation-->\n--\n");
					gdl_hstruct_mutation_pattern * mp = gdl_hstruct_partition_result_mutation_pattern (pr);
					for (k = 0; k < pr->kmax; k++)
					{
						fprintf (stream, "%d", k+1);
						for (l = 0; l < pr->nl; l++)
						{
							 gdl_vector_uint * mutations = gdl_hstruct_mutation_pattern_get (mp, l, k);
							 if (mutations)
							 {
								 	const gdl_locus * locus = gdl_gmap_get_locus (r->gmap, i, j, l+pr->fl);
					 				fprintf (stream, "\t");
								 	for (kk = 0; kk < mutations->size; kk++)
								 	{
								 		fprintf (stream, "%s", gdl_entity_get_name (gdl_locus_get_allele (locus, gdl_vector_uint_get (mutations, kk))));
								 		if (kk < mutations->size - 1)
								 		{
								 			fprintf (stream, ","); 	
								 		}
								 	}
								 	gdl_vector_uint_free (mutations);
							  }
							  else
							  {
								 	fprintf (stream, "\t.");
							  }
						}						
						fprintf (stream, "\n");
					}
					//gdl_hstruct_mutation_pattern_free (mp);
					fprintf (stream, "\n--\n");
					fprintf (stream, "Accession\tName\tPhase\t--> Ancestral Best Path\n--\n");
					for (ii = 0; ii < pr->na; ii++)
					{
						ic = gdl_clustering_cluster (pr->gclust, ii);
						for (k = 0; k < pr->np; k++)
						{
							fprintf (stream, "%d\t%s\t%d", ii+1, gdl_entity_get_name (pr->accession[ii]), k+1);
							for (l = 0; l < pr->nl; l++)
							{
								fprintf (stream, "\t%d", gdl_vector_uint_get (pr->best[ic*(pr->np)+k], l)+1);	
								//fprintf (stream, "\t%1.2f", gdl_matrix_get (pr->proba[ic*(pr->np)+k], l, 0));
							}
							fprintf (stream, "\n");
						}
					}
					fprintf (stream, "\n--\n");
					fprintf (stream, "Accession\tName\tPhase\t--> Ancestral Best Block Path\n--\n");
					for (ii = 0; ii < pr->na; ii++)
					{
						for (k = 0; k < pr->np; k++)
						{
							gdl_vector_uint * path = gdl_hstruct_partition_result_accession_best_block_path (pr, ii, k);
							fprintf (stream, "%d\t%s\t%d", ii+1, gdl_entity_get_name (pr->accession[ii]), k+1);
							for (l = 0; l < pr->nl; l++)
							{
								fprintf (stream, "\t%d", gdl_vector_uint_get (path, l)+1);	
							}
							fprintf (stream, "\n");
							gdl_vector_uint_free (path);
						}
					}
					fprintf (stream, "\n--\n");
					fprintf (stream, "Accession\tName\tPhase\t--> Mutation\n--\n");
					for (ii = 0; ii < pr->na; ii++)
					{
						ic = gdl_clustering_cluster (pr->gclust, ii);
						for (k = 0; k < pr->np; k++)
						{
							fprintf (stream, "%d\t%s\t%d", ii+1, gdl_entity_get_name (pr->accession[ii]), k+1);
							for (l = 0; l < pr->nl; l++)
							{
								const gdl_locus * locus = gdl_gmap_get_locus (r->gmap, i, j, l+pr->fl);
					 			if (gdl_vector_uint_get (pr->mutation[ic*(pr->np)+k], l))
								{
									fprintf (stream, "\t%s", gdl_entity_get_name (gdl_locus_get_allele (locus, gdl_vector_uint_get (pr->mutation[ic*(pr->np)+k], l)-1)));	
								}
								else
								{
								   fprintf (stream, "\t.");
								}
							}
							fprintf (stream, "\n");
						}
					}
				}
			}
		}
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

static int
gdl_hstruct_partition_result_block_extend (gdl_hstruct_result * r, gdl_hstruct_partition_result * pr)
{
	size_t i, k, b, nl, nb, kmax, fidx;
	gdl_hstruct_fpoint * f, * f1;
	gdl_hstruct_static_config * c, * uc;
	gdl_locus * locus;
	
	nl = pr->nl; 
	nb = gdl_hstruct_block_partition_size (pr->ancestral);
	
	if (nb > 1)
	{
		kmax = gdl_hstruct_block_partition_kmax (pr->ancestral);
		for (i = 0; i < nl; i++)
		{
			locus = gdl_chromosome_get (pr->chrom, pr->fl + i);
			c = (gdl_hstruct_static_config *) locus->extra;
			if (c->k==kmax)
			{
				break;
			}
		}
		fidx = c->f_idx;
		// get the fpoint and update it...
		f = (gdl_hstruct_fpoint *) gdl_hstruct_static_parameter_registry_get (r->registry, gdl_hstruct_point_f, fidx);
		for (b = 0; b < nb; b++)
		{
			const gdl_hstruct_block * block = gdl_hstruct_block_partition_get (pr->ancestral, b);
			for (i = block->from; i <= block->to; i++)
			{
				locus = gdl_chromosome_get (pr->chrom, pr->fl + i);
				c  = (gdl_hstruct_static_config *) locus->extra;
				if (c->f_idx >= 0)
				{
					f1 = (gdl_hstruct_fpoint *) gdl_hstruct_static_parameter_registry_get (r->registry, gdl_hstruct_point_f, c->f_idx);
					for (k = 0; k < c->k; k++)
					{
						gdl_hstruct_fpoint_collect (f, k, gdl_hstruct_fpoint_value (f1, k));
					}
				}
				if (c->k != kmax)
				{
					uc = gdl_hstruct_static_config_extend (c, kmax, fidx);
					for (k = c->k; k < uc->k; k++)
					{
						uc->ancestral[k] = gdl_entity_clone (c->ancestral[gdl_vector_uint_get (block->best, k)]);
					}
					gdl_hstruct_static_config_free (c);
					locus->extra = uc;
				}
				else
				{
					c->f_idx = fidx;
				}		
			} 
		}
		gdl_hstruct_fpoint_update (f, 0, 0);
	}
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_result_block_extend (gdl_hstruct_result * r)
{
	if (!r)
	{
		return GDL_EINVAL;	
	}
	
	size_t i, j, k, l, ng, nc, np;
	gdl_hstruct_genome_result * gr;
	gdl_hstruct_chromosome_result * cr;
	gdl_hstruct_partition_result * pr;
	
	ng = gdl_gmap_size (r->gmap);
	for (i = 0; i < ng; i++)
	{
		gr = r->results[i];
		nc = gr->size;
		for (j = 0; j < nc; j++)
		{
			cr = gr->results[j];
			np = cr->size;
			for (k = 0; k < np; k++)
			{
				pr = cr->results[k];
				gdl_hstruct_partition_result_block_extend (r, pr);	
			}	
		}
	}
	
	return GDL_SUCCESS;
}

