/*  
 * 	hstruct/test.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_test.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview_reader.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_gmap_reader.h>
#include <gdl/gdl_hmap.h>
#include <gdl/gdl_view.h>
#include <gdl/gdl_runtime.h>
#include <gdl/gdl_hstruct.h>


gdl_gview_reader_type *
_reader_type (size_t t)
{
	gdl_gview_reader_type * type =
    	gdl_gview_reader_type_new (gdl_gview_reader_standard);
    
    switch (t)
    {
    	case 1 :
    		gdl_gview_reader_type_missing (type, "missing");
    		return type;
    	case 2 :
		    gdl_gview_reader_type_missing (type, "0");
		    gdl_gview_reader_type_nfsep (type, " ");
		    gdl_gview_reader_type_gfsep (type, " ");
		    gdl_gview_reader_type_ploidy (type, 2);
		    gdl_gview_reader_type_locus_name (type, gdl_true);
		    gdl_gview_reader_type_is_phased (type, gdl_false);
    		return type;
    	case 3 :
		    gdl_gview_reader_type_missing (type, "0");
		    gdl_gview_reader_type_nfsep (type, " ");
		    gdl_gview_reader_type_gfsep (type, " ");
		    gdl_gview_reader_type_ploidy (type, 2);
		    gdl_gview_reader_type_locus_name (type, gdl_true);
		    gdl_gview_reader_type_is_phased (type, gdl_false);
    		return type;
    	case 4 :
    		gdl_gview_reader_type_missing (type, "N");
		    gdl_gview_reader_type_nfsep (type, " ");
		    gdl_gview_reader_type_gfsep (type, "");
		    gdl_gview_reader_type_ploidy (type, 1);
		    gdl_gview_reader_type_locus_name (type, gdl_true);
    		return type;
    	case 5 :
    		gdl_gview_reader_type_missing (type, "?");
		    gdl_gview_reader_type_nfsep (type, " ");
		    gdl_gview_reader_type_gfsep (type, " ");
		    gdl_gview_reader_type_ploidy (type, 1);
		    gdl_gview_reader_type_locus_name (type, gdl_true);
		    return type;
    }
}

gdl_gview *
_read_data (size_t t)
{
	size_t i, j;
	FILE * stream;
	gdl_string * file;
	gdl_gview_reader_type * type;
	gdl_gview_reader      * reader;
	gdl_gview * v;
	
	type = _reader_type (t);
	
	reader = gdl_gview_reader_alloc (type);
	
	file = gdl_string_sprintf ("examples/data%d.txt", t);
	
	gdl_gview_reader_open (reader, file);
	
	v = gdl_gview_reader_parse (reader);
	
	gdl_gview_reader_close (reader);
	
	gdl_gview_reader_free (reader);
	
	gdl_string_free (file);
	
	return v;	
}

gdl_gmap *
_read_map (size_t t)
{
	size_t i, j;
	FILE * stream;
	gdl_string * file;
	gdl_gmap_reader * reader;
	gdl_gmap * map;
	
	reader = gdl_gmap_reader_alloc (gdl_gmap_reader_absolute);
	
	file = gdl_string_sprintf ("examples/map%d.txt", t);
	
	gdl_gmap_reader_open (reader, file);
	
	map = gdl_gmap_reader_parse (reader);
	
	if (map==0)
	{
		fprintf (stderr, "[ ERROR ] %s", reader->error);
		exit (1);	
	}
	
	gdl_gmap_reader_close (reader);
	
	gdl_gmap_reader_free (reader);
	
	gdl_string_free (file);
	
	return map;	
}

static gdl_view *
test_view ()
{
	gdl_view * v;
	
	v = gdl_view_alloc ();
	
	gdl_view_add_gview (v, "data", _read_data (5));
    
    gdl_view_set_gmap (v, _read_map (5));
    
	return v;
}

static gdl_hstruct_data *
test_data ()
{
	gdl_view * v;
	gdl_gview * g;
	gdl_mask * u, * m = NULL;
	gdl_hstruct_data * D;
	
	v = test_view ();
	
	g = gdl_view_get_gview (v);
	
	u = gdl_gview_gmask_uninformative (g, NULL, GDL_LOCUS);
	m = gdl_gview_gmask_uninformative (g, u, GDL_ACCESSION);
	gdl_mask_free (u);
	
	D = gdl_hstruct_data_alloc (gdl_hstruct_model_gmap);
	D->view  = v;
	D->gmask = m;
	D->rng   = gdl_rng_alloc (gdl_rng_default);
	
	return D;
}

static gdl_mask **
test_partition (const gdl_hstruct_data * D,
                const gdl_chromosome * chrom,
                void * extra,
                size_t * np)
{

#define PART 40
	
	size_t i, j, l, nl;
	const gdl_gview * gview;
	gdl_entity_mask * plocus;
	gdl_mask * cmask, ** pmask, * tmp;
	
	gview = D->gview;
	
	cmask = gdl_mask_alloc ();
	gdl_mask_set (cmask, GDL_LOCUS, gdl_chromosome_get_mask (chrom, gdl_mask_get (D->gmask, GDL_LOCUS), 1);
	gdl_mask_set (cmask, GDL_ACCESSION, gdl_mask_get_clone (D->gmask, GDL_ACCESSION), 1);
	
	nl  = GDL_GVIEW_LOCUS_SIZE (gview, cmask);
	
	*np = nl / PART;
	
	if (nl % PART >= 10)
	{
		(*np)++;
	}
	
	pmask = GDL_MALLOC (gdl_mask *, *np);
	
	for (l = i = 0; i < *np; i++)
	{
		pmask[i] = gdl_mask_alloc ();
		plocus   = gdl_entity_mask_alloc (PART);
		
		for (j = 0; j < PART && l < nl; j++, l++)
		{
			gdl_locus * loc = GDL_GVIEW_GET_LOCUS (gview, cmask, l);
			gdl_entity_mask_add (plocus, loc);
		}
		
		gdl_mask_set (pmask[i], GDL_LOCUS, plocus, 1);
		gdl_mask_set (pmask[i], GDL_ACCESSION, gdl_mask_get_clone (D->gmask, GDL_ACCESSION), 0);
		
		tmp = gdl_gview_gmask_uninformative (gview, pmask[i], GDL_ACCESSION);
		
		gdl_mask_free (pmask[i]);
		
		pmask[i] = tmp;
	}
	
	gdl_mask_free (cmask);
	
	return pmask;
}

int
hmm_maximize (gdl_hstruct_workspace * wf)
{
	size_t iter=0;
	
	do
	{
		iter++;
		//printf ("GOLDEN RHO\n");
		gdl_hstruct_workspace_hmm_iterate (wf, gdl_hstruct_point_rho);
		//printf ("GOLDEN HOT\n");
		gdl_hstruct_workspace_hmm_iterate (wf, gdl_hstruct_point_hot);
		//printf ("GOLDEN MU\n");
		gdl_hstruct_workspace_hmm_iterate (wf, gdl_hstruct_point_mu);
		//printf ("GOLDEN F\n");
		gdl_hstruct_workspace_hmm_iterate (wf, gdl_hstruct_point_f);
		printf ("HMM %d %g %g\n", iter, gdl_hstruct_workspace_residual_sq (wf), gdl_hstruct_workspace_loglikelihood (wf));
	} while (iter < 10);
	
	printf ("CRITERION AIC %g\n", gdl_hstruct_workspace_criterion (wf, gdl_hstruct_criterion_AIC));
	printf ("CRITERION BIC %g\n", gdl_hstruct_workspace_criterion (wf, gdl_hstruct_criterion_BIC));
	
	return GDL_SUCCESS;
}

int
test_create (size_t K)
{
	int status;
	size_t i, j, nc, np, iter;
	gdl_hstruct_data * D;
	gdl_hstruct_model * M;
	gdl_hstruct_workspace * wf;
	FILE * stream;
    
    D = test_data ();
    
    M = gdl_hstruct_model_alloc (D);
    
    gdl_hstruct_model_init (M, gdl_hstruct_model_homogeneous, K);
    
    //gdl_hstruct_model_init_pl (M, gdl_hstruct_model_heterogeneous, K, &test_partition, NULL, gdl_gligation_hierarchical);
   
	wf = gdl_hstruct_workspace_alloc (M);
	
	nc = gdl_hstruct_workspace_chromosome_size (wf);
	
	for (i = 0; i < nc; i++)
	{
		
		np = gdl_hstruct_workspace_partition_size (wf, i);
		
		for (j = 0; j < np; j++)
		{
			printf ("CHROMOSOME %d PARTITION %d\n", i, j);
			iter=0;
			do
			{	iter++;
				gdl_hstruct_workspace_fuzzy_iterate (wf, i, j);
				printf ("FUZZY %d %g\n", iter, gdl_hstruct_workspace_residual_sq (wf));
			} while (iter < 500);
		}
	}
	
	gdl_hstruct_workspace_fuzzy2hmm (wf);
	
	do
	{
		
		hmm_maximize (wf);
		
		for (i = 0; i < nc; i++)
		{
			np = gdl_hstruct_workspace_partition_size (wf, i);
			for (j = 0; j < np; j++)
			{
				printf ("HOTREC::CHROMOSOME %d PARTITION %d\n", i, j);
				gdl_hstruct_workspace_hotrec_search (wf, gdl_hstruct_criterion_BIC, i, j);
			}
		}
		
		for (i = 0; i < nc; i++)
		{
			np = gdl_hstruct_workspace_partition_size (wf, i);
			for (j = 0; j < np; j++)
			{
				printf ("COLLAPSE::CHROMOSOME %d PARTITION %d\n", i, j);
				gdl_hstruct_workspace_collapse_ancestral (wf, gdl_hstruct_criterion_BIC, i, j, 1.e-6);
			}
		}
		
	} while (gdl_hstruct_workspace_ligation (wf) == GDL_CONTINUE);
	
	gdl_hstruct_workspace_free (wf);
	
	return GDL_SUCCESS;
}

int
main (int argc, char * argv[])
{
	if (argc)
	{
	    test_create (atoi(argv[1]));
		exit (gdl_test_summary());
	}
}
