/*  
 * 	hview/hview.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_permutation.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_collector.h>
#include <gdl/gdl_hview.h>

#include "private.c"

gdl_hview *
gdl_hview_alloc (const gdl_gview * data, const gdl_mask * mask)
{
	gdl_hview * h;
	
	h = GDL_CALLOC (gdl_hview, 1);
	
	h->data  = data;
	h->mask  = mask;
	
	return h;
}

int
gdl_hview_create (gdl_hview * h)
{
	size_t i;
	int status;
	_gdl_hview * _h = _gdl_hview_alloc (h);
	
	h->guniq = gdl_gview_accession_clustering (h->data, h->mask);
	
	status = _collect_ambiguous (_h);
	
	if (status != GDL_SUCCESS)
	{
		GDL_ERROR_VAL (_h->error,
		               status,
		               0);
	}
	
	status = _resolve_ambiguous (_h, 1.0);
	
	if (status != GDL_SUCCESS)
	{
		GDL_ERROR_VAL (_h->error,
		               status,
		               0);
	}
	
	_gdl_hview_free (_h);
	
	return GDL_SUCCESS;
}

static void
_gdl_hview_clean (gdl_hview * h)
{
	size_t i, j, n;
	
	n = GDL_MAX(h->na, h->nh);
	
	for (i = 0; i < n; i++)
	{
		if (i < h->na)
		{
			for (j = 0; j < h->nhc[i]; j++)
			{
				gdl_hconfig_free (h->hconfigs[i][j]);
			}
			GDL_FREE (h->hconfigs[i]);
		}
		if (i < h->nh)
		{
			GDL_FREE (h->haplotypes[i]);
		}
	}
	GDL_FREE (h->hconfigs);
	GDL_FREE (h->haplotypes);
	GDL_FREE (h->mult);
	gdl_clustering_free (h->guniq);
}

void
gdl_hview_free (gdl_hview * h)
{
	if (h)
	{
		_gdl_hview_clean (h);
		GDL_FREE (h);
	}	
}

int
gdl_hview_copy (gdl_hview * dest, const gdl_hview * src)
{
	if (dest && src)
	{
		size_t i, j, n;
		
		if (dest->data != src->data)
		{
			return GDL_EINVAL;
		}
		if (dest->mask != src->mask)
		{
			return GDL_EINVAL;
		}
		
		_gdl_hview_clean (dest);
		
		dest->na = src->na;
		dest->nl = src->nl;
		dest->nh = src->nh;
		
		dest->mult = GDL_MALLOC (double, src->nh);
		
		memcpy (dest->mult, src->mult, sizeof(double)*src->nh);
		
		dest->haplotypes = GDL_MALLOC (gdl_haplotype, src->nh);
		
		for (i = 0; i < src->nh; i++)
		{
			dest->haplotypes[i] = gdl_haplotype_clone (&(src->haplotypes[i]), src->nl);
		}
		
		dest->guniq = gdl_clustering_clone (src->guniq);
		
		dest->nhc = GDL_MALLOC (size_t, src->na);
		
		memcpy (dest->nhc, src->nhc, sizeof(size_t)*src->na);
		
		dest->hconfigs = GDL_MALLOC (gdl_hconfig_ptr *, src->na);
		
		for (i = 0; i < src->na; i++)
		{
			dest->hconfigs[i] = GDL_MALLOC (gdl_hconfig_ptr, dest->nhc[i]);
			
			for (j = 0; j < dest->nhc[i]; j++)
			{
				dest->hconfigs[i][j] = gdl_hconfig_clone (src->hconfigs[i][j], gdl_gview_ploidy(src->data));
			}
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

gdl_hview *
gdl_hview_clone (const gdl_hview * h)
{
	if (h)
	{
		gdl_hview * c;
		
		c = GDL_CALLOC (gdl_hview, 1);
		
		c->data = h->data;
		c->mask = h->mask;
		
		gdl_hview_copy (c, h);
		
		return c;
	}
	
	return NULL;	
}

size_t
gdl_hview_ploidy (const gdl_hview * h)
{
	return gdl_gview_ploidy (h->data);
}

size_t
gdl_hview_accession_size (const gdl_hview * h)
{
	return GDL_GVIEW_ACCESSION_SIZE (h->data, h->mask);
}

size_t
gdl_hview_accession_size_c (const gdl_hview * h)
{
	return gdl_clustering_nclust (h->guniq);
}

size_t
gdl_hview_accession_mult_c (const gdl_hview * h, size_t i)
{
	return gdl_clustering_clust_size (h->guniq, i);
}

size_t
gdl_hview_locus_size (const gdl_hview * h)
{
	return GDL_GVIEW_LOCUS_SIZE (h->data, h->mask);	
}

size_t
gdl_hview_locus_allele_size (const gdl_hview * h, size_t i)
{
	return gdl_locus_allele (GDL_GVIEW_GET_LOCUS (h->data, h->mask, i));
}

size_t
gdl_hview_haplotype_size (const gdl_hview * h)
{
	return h->nh;	
}

gdl_locus *
gdl_hview_get_locus (const gdl_hview * h, size_t i)
{
	return GDL_GVIEW_GET_LOCUS (h->data, h->mask, i);
}

gdl_accession *
gdl_hview_get_accession (const gdl_hview * h, size_t i)
{
	return GDL_GVIEW_GET_ACCESSION (h->data, h->mask, i);	
}

gdl_haplotype *
gdl_hview_get_haplotype (const gdl_hview * h, size_t i)
{
	return h->haplotypes + i;
}

double
gdl_hview_get_haplotype_freq (const gdl_hview * h, size_t i)
{
	return h->mult[i];
}

void
gdl_hview_set_haplotype_freq (gdl_hview * h, size_t i, double x)
{
	h->mult[i] = x;
}

const gdl_gview *
gdl_hview_get_gview (const gdl_hview * h)
{
	return h->data;	
}

const gdl_mask *
gdl_hview_get_gmask (const gdl_hview * h)
{
	return h->mask;	
}

const gdl_clustering *
gdl_hview_get_clustering (const gdl_hview * h)
{
	return h->guniq;	
}

gdl_gvalues_get *
gdl_hview_get_new (const gdl_hview * h)
{
	return GDL_GVIEW_GET_NEW (h->data, h->mask);	
}

const gdl_gvalues *
gdl_hview_get_gallele_c (const gdl_hview * h, size_t ac, size_t l, size_t p, gdl_gvalues_get * gb)
{
	size_t a = gdl_clustering_clust_idx (h->guniq, ac);
	return gdl_hview_get_gallele (h, a, l, p, gb);
}

const gdl_gvalues *
gdl_hview_get_gallele (const gdl_hview * h, size_t a, size_t l, size_t p, gdl_gvalues_get * gb)
{
	gdl_gvalues * gv;
	
	GDL_GVIEW_GET_ALLELE (h->data, h->mask, a, l, p, gb);
	
	gv = gb->x;
	
	if (gv == 0)
	{
		size_t i, ih, nc;
		double * f;
		gdl_locus  * locus;
		gdl_allele * allele;
		gdl_string * key;
		gdl_list * tmp;
		gdl_hashtable * hash;
		gdl_hashtable_itr * itr;
		
		locus = GDL_GVIEW_GET_LOCUS (h->data, h->mask, l);
		
		tmp  = gdl_list_alloc (gdl_list_default);
		hash = gdl_hashtable_alloc (gdl_hash_default, gdl_locus_allele (locus));
		
		nc = gdl_hview_hconfig_size (h, a);
		
		for (i = 0; i < nc; i++)
		{
			gdl_hconfig * hc = gdl_hview_get_hconfig (h, a, i);
			ih = gdl_hconfig_get_haplotype (hc, p);
			allele = gdl_hview_get_hallele (h, gdl_hview_get_haplotype (h, ih), l);
			f = (double *) gdl_hashtable_lookup (hash, allele->name);
			if (f == 0)
			{
				f = GDL_CALLOC (double, 1);
				gdl_list_push_front (tmp, f, 1);
				gdl_hashtable_add (hash, allele->name, f, 0);
			}
			*f += 1.0;
		}
		
		itr = gdl_hashtable_iterator (hash);
		i = 0;
		
		do
		{
			key = gdl_hashtable_iterator_key (itr);
			f = (double *) gdl_hashtable_iterator_value (itr);
			allele = gdl_locus_search_allele (locus, key);
			gv->values[i]->idx   = allele->idx;
			gv->values[i]->value = *f/(double)nc;
			i++;
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gv->size = i;
		gb->na   = gdl_false;
		
		gdl_hashtable_iterator_free (itr);
		gdl_hashtable_free (hash);
		gdl_list_free (tmp);
	}
	
	return gv;
}

gdl_allele *
gdl_hview_get_hallele (const gdl_hview * h, gdl_haplotype * haplo, size_t j)
{
	gdl_locus * l = GDL_GVIEW_GET_LOCUS (h->data, h->mask, j);
	return gdl_locus_get_allele (l, (*haplo)[j]);
}

size_t
gdl_hview_hconfig_size (const gdl_hview * h, size_t i)
{
	size_t ic = gdl_clustering_cluster (h->guniq, i);
	return h->nhc[ic];
}

gdl_hconfig *
gdl_hview_get_hconfig (const gdl_hview * h, size_t i, size_t j)
{
	size_t ic = gdl_clustering_cluster (h->guniq, i);
	return h->hconfigs[ic][j];
}

size_t
gdl_hview_hconfig_size_c (const gdl_hview * h, size_t ic)
{
	return h->nhc[ic];	
}

gdl_hconfig * 
gdl_hview_get_hconfig_c (const gdl_hview * h, size_t ic, size_t j)
{
	return h->hconfigs[ic][j];
}

size_t
gdl_hconfig_get_haplotype (const gdl_hconfig * hconf, size_t i)
{
	return hconf->idx[i];
}

double 
gdl_hconfig_get_proba (const gdl_hconfig * hconf)
{
	return hconf->pr;
}

void
gdl_hconfig_set_proba (gdl_hconfig * hconf, double pr)
{
	hconf->pr = pr;	
}

gdl_haplotype
gdl_haplotype_clone (const gdl_haplotype * h, size_t n)
{
	if (h)
	{
		gdl_haplotype c = GDL_MALLOC (size_t, n);
		memcpy (c, *h, sizeof(size_t)*n);
		return c;
	}
	return NULL;
}

int
gdl_hview_fprintf (FILE * stream, const gdl_hview * h)
{
	size_t i, ii, j, k;
	size_t np = gdl_hview_ploidy (h);
	size_t na = gdl_hview_accession_size (h);
	size_t nl = gdl_hview_locus_size (h);
	size_t nh = gdl_hview_haplotype_size (h);
	double tot = 0;
	const gdl_clustering * gclust = gdl_hview_get_clustering (h);
	
	for (i = 0; i < nh; i++)
	{
		gdl_haplotype * haplo = gdl_hview_get_haplotype (h, i);
		double x = gdl_hview_get_haplotype_freq (h, i);
		printf ("H(%d,%g)", i, x);
		tot += x;
		for (j = 0; j < nl; j++)
		{
			gdl_allele * allele = gdl_hview_get_hallele (h, haplo, j);
			printf (" %s", allele->name);
		}
		printf ("\n");
	}
	printf ("TOT = %g\n", tot);
	
	na = gdl_clustering_nclust (gclust);
	
	for (ii = 0; ii < na; ii++)
	{
		i = gdl_clustering_clust_idx (gclust, ii);
		
		nh = gdl_hview_hconfig_size (h, i);
		
		printf (">CLUSTER %d (%d) %d\n", ii, gdl_clustering_clust_size (gclust, ii), nh);
		
		for (j = 0; j < nh; j++)
		{
			gdl_hconfig	* hc = gdl_hview_get_hconfig (h, i, j);
			printf ("H(%d) %g ", i, gdl_hconfig_get_proba (hc));
			for (k = 0; k < np; k++)
			{
				size_t idx = gdl_hconfig_get_haplotype (hc, k);
				printf ("%d", idx);
				if (k < np - 1) printf ("/");
			}
			printf (" ");
		}
		printf ("\n");
	}
	
	return GDL_SUCCESS;
}

gdl_gview_collector *
gdl_hview_get_missing_collector (const gdl_hview * h)
{
	size_t i, ic, j, k, l, na, nh;
	const size_t * lidx;
	gdl_gvalues * x;
	gdl_gvalues_get * gb;
	gdl_gview_collector * missing;
	
	missing = gdl_gview_collector_alloc (gdl_gview_collector_missing);
	
	gdl_gview_collector_perform (missing, h->data, h->mask, h->guniq);
	
	gb = gdl_hview_get_new (h);
	
	na = gdl_gview_collector_accession_size (missing);
	for (i = 0; i < na; i++)
	{
		ic = gdl_gview_collector_accession_idx (missing, i);
		for (j = 0; j < gdl_hview_ploidy (h); j++)
		{
			nh   = gdl_gview_collector_hlocus_size (missing, i, j);
			lidx = gdl_gview_collector_hlocus_idx (missing, i, j);
			for (l = 0; l < nh; l++)
			{
				const gdl_gvalues * y = gdl_hview_get_gallele_c (h, ic, lidx[l], j, gb);
				x = gdl_gvalues_clone (y); 
				gdl_gview_collector_hset (missing, i, j, l, x);
			}
		}
	}
	
	gdl_gvalues_get_free (gb);
	
	return missing;
}

#include "kbest.c"
#include "hcrm.c"
#include "ligation.c"
#include "extract.c"
#include "fread.c"
