/*  
 *  hview/wrapper.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_hview_wrapper.h>

struct _gdl_hview_wrapper
{
	size_t np;
	size_t na;
	size_t nl;
	gdl_mask       * mask;
	gdl_hview      * hview;
	gdl_locus     ** locus;
	gdl_accession ** accession;
};

gdl_hview_wrapper *
gdl_hview_wrapper_alloc (const gdl_hview * hview)
{
	size_t i;
	gdl_hview_wrapper * w;
	
	w = GDL_CALLOC (gdl_hview_wrapper, 1);
	
	w->hview = gdl_hview_clone (hview);
	w->mask  = gdl_mask_clone (gdl_hview_get_gmask (hview));
	w->np    = gdl_hview_ploidy (hview);
	w->nl    = gdl_hview_locus_size (hview);
	w->na    = gdl_hview_accession_size (hview);
	w->locus = GDL_MALLOC (gdl_locus *, w->nl);
	w->accession = GDL_MALLOC (gdl_accession *, w->na);
	for (i = 0; i < w->nl; i++)
	{
		w->locus[i] = gdl_entity_clone (gdl_hview_get_locus (hview, i));	
	}
	for (i = 0; i < w->na; i++)
	{
		w->accession[i] = gdl_entity_clone (gdl_hview_get_accession (hview, i));	
	}
	
	return w;
}

void
gdl_hview_wrapper_free (gdl_hview_wrapper * h)
{
	if (h)
	{
		size_t i;
		gdl_hview_free (h->hview);
		gdl_mask_free (h->mask);
		for (i = 0; i < h->nl; i++)
		{
			gdl_locus_free (h->locus[i]);	
		}
		for (i = 0; i < h->na; i++)
		{
			gdl_accession_free (h->accession[i]);	
		}
		GDL_FREE (h->locus);
		GDL_FREE (h->accession);
		GDL_FREE (h);	
	}	
}

gdl_hview_wrapper *
gdl_hview_wrapper_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_hview_wrapper * w;
		
		w = GDL_CALLOC (gdl_hview_wrapper, 1);
		
		w->hview = gdl_hview_alloc (0, 0);
		status = gdl_hview_fread (stream, w->hview);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		w->mask = gdl_mask_fread (stream);
		GDL_FREAD_STATUS (w->mask!=0, 1);
		
		status = fread (&(w->np), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(w->na), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(w->nl), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		w->locus = GDL_MALLOC (gdl_locus *, w->nl);
		w->accession = GDL_MALLOC (gdl_accession *, w->na);
		
		for (i = 0; i < w->nl; i++)
		{
			w->locus[i] = gdl_entity_fread (stream);
			GDL_FREAD_STATUS (w->locus[i]!=0, 1);
		}
		for (i = 0; i < w->na; i++)
		{
			w->accession[i] = gdl_entity_fread (stream);
			GDL_FREAD_STATUS (w->accession[i]!=0, 1);
		}
		
		return w;
	}
	
	return NULL;	
}

int
gdl_hview_wrapper_fwrite (FILE * stream, const gdl_hview_wrapper * w)
{
	if (stream && w)
	{
		int status;
		size_t i;
		
		status = gdl_hview_fwrite (stream, w->hview);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_mask_fwrite (stream, w->mask);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = fwrite (&(w->np), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(w->na), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(w->nl), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < w->nl; i++)
		{
			status = gdl_entity_fwrite (stream, w->locus[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		for (i = 0; i < w->na; i++)
		{
			status = gdl_entity_fwrite (stream, w->accession[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

size_t
gdl_hview_wrapper_ploidy (const gdl_hview_wrapper * w)
{
	return w->np;
}

size_t
gdl_hview_wrapper_accession_size (const gdl_hview_wrapper * w)
{
	return w->na;	
}

size_t
gdl_hview_wrapper_accession_size_c (const gdl_hview_wrapper * w)
{
	return gdl_hview_accession_size_c (w->hview);	
}

size_t
gdl_hview_wrapper_accession_mult_c (const gdl_hview_wrapper * w, size_t i)
{
	return gdl_hview_accession_mult_c (w->hview, i);
}

size_t
gdl_hview_wrapper_locus_size (const gdl_hview_wrapper * w)
{
	return w->nl;	
}

size_t
gdl_hview_wrapper_locus_allele_size (const gdl_hview_wrapper * w, size_t i)
{
	return gdl_locus_allele (w->locus[i]);
}

size_t
gdl_hview_wrapper_haplotype_size (const gdl_hview_wrapper * w)
{
	return gdl_hview_haplotype_size (w->hview);	
}

gdl_locus *
gdl_hview_wrapper_get_locus (const gdl_hview_wrapper * w, size_t i)
{
	return w->locus[i];	
}

gdl_accession *
gdl_hview_wrapper_get_accession (const gdl_hview_wrapper * w, size_t i)
{
	return w->accession[i];
}

gdl_haplotype *
gdl_hview_wrapper_get_haplotype (const gdl_hview_wrapper * w, size_t i)
{
	return gdl_hview_get_haplotype (w->hview, i);	
}

double
gdl_hview_wrapper_get_haplotype_freq (const gdl_hview_wrapper * w, size_t i)
{
	return gdl_hview_get_haplotype_freq (w->hview, i);
}

void
gdl_hview_wrapper_set_haplotype_freq (gdl_hview_wrapper * w, size_t i, double x)
{
	gdl_hview_set_haplotype_freq (w->hview, i, x);
}

const gdl_allele *
gdl_hview_wrapper_get_allele (const gdl_hview_wrapper * w, gdl_haplotype * haplo, size_t i)
{
	return gdl_locus_get_allele (w->locus[i], (*haplo)[i]);
}

const gdl_mask *
gdl_hview_wrapper_get_mask (const gdl_hview_wrapper * w)
{
	return w->mask;	
}

const gdl_clustering *
gdl_hview_wrapper_get_clustering (const gdl_hview_wrapper * w)
{
	return gdl_hview_get_clustering (w->hview);	
}

size_t
gdl_hview_wrapper_hconfig_size (const gdl_hview_wrapper * w, size_t i)
{
	return gdl_hview_hconfig_size (w->hview, i);
}

gdl_hconfig *
gdl_hview_wrapper_get_hconfig (const gdl_hview_wrapper * w, size_t i, size_t j)
{
	return gdl_hview_get_hconfig (w->hview, i, j);
}

size_t
gdl_hview_wrapper_hconfig_size_c (const gdl_hview_wrapper * w, size_t ic)
{
	return gdl_hview_hconfig_size_c (w->hview, ic);	
}

gdl_hconfig *
gdl_hview_wrapper_get_hconfig_c (const gdl_hview_wrapper * w, size_t ic, size_t j)
{
	return gdl_hview_get_hconfig_c (w->hview, ic, j);	
}

gdl_hview *
gdl_hview_wrapper_unwrap (const gdl_hview_wrapper * w, const gdl_gview * g)
{
	gdl_hview * h;
	
	h = gdl_hview_alloc (g, w->mask);
	
	gdl_hview_copy (h, w->hview);
	
	return h;
}
