/*  
 *  flabels/flabels.c
 *   
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:54 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_wrapper.h>
#include <gdl/gdl_flabels.h>

struct _gdl_flabels
{
	size_t n;
	size_t l;
	size_t a;
	int * aidx;
	size_t ** cidx;
	gdl_string ** factor;
	gdl_string ** accession;
	gdl_string ** level;
	gdl_hashtable * _haccession;
	gdl_hashtable * _hfactor;
	gdl_factor     ** _factor;
	gdl_accession ** _accession;
};

static gdl_flabels *
_gdl_flabels_alloc (size_t n, size_t l, size_t a)
{
	gdl_flabels * g;
	
	g = GDL_MALLOC (gdl_flabels, 1);
	
	g->n = n;
	g->l = l;
	g->a = a;
	
	g->factor     = GDL_CALLOC (gdl_string *, l);
	g->level    = GDL_CALLOC (gdl_string *, a);
	g->accession = GDL_CALLOC (gdl_string *, n);
	g->cidx      = GDL_CALLOC (size_t * , a);
	
	return g;
}

static int
_build_reverse_search (gdl_flabels * gl)
{
	 size_t i, j, na;
	 
	 gl->_accession  = GDL_MALLOC (gdl_accession *, gl->n);
	 gl->_haccession = gdl_hashtable_alloc (gdl_hash_default, gl->n);
	 for (i = 0; i < gl->n; i++)
	 {
	 	 gl->_accession[i] = gdl_accession_alloc ();
	 	 gdl_entity_set_idx (gl->_accession[i], i);
	 	 gdl_hashtable_add (gl->_haccession, gl->accession[i], gl->_accession[i], 0); 	 
	 }
	 gl->_factor  = GDL_MALLOC (gdl_factor *, gl->l);
	 gl->_hfactor = gdl_hashtable_alloc (gdl_hash_default, gl->l);
	 for (i = 0; i < gl->l; i++)
	 {
	 	 gl->_factor[i] = gdl_factor_alloc (gdl_factor_categorial);
	 	 gdl_entity_set_idx (gl->_factor[i], i);
	 	 gdl_hashtable_add (gl->_hfactor, gl->factor[i], gl->_factor[i], 0);
	 	 na = gdl_flabels_factor_level_size (gl, i);
	 	 if (na)
	 	 {
		 	 for (j = 0; j < na; j++)
		 	 {
		 	 	 gdl_factor_level * level = gdl_factor_level_new (gdl_flabels_level (gl, i, j));
			 	 gdl_factor_add (gl->_factor[i], &level, 1);
		 	 }
	 	 }
	 }
}

static int
_clean_reverse_search (gdl_flabels * gl)
{
	if (gl)
	{
		size_t i;
		for (i = 0; i < gl->n; i++)
		{
			gdl_entity_free (gl->_accession[i]);	
		}
		for (i = 0; i < gl->l; i++)
		{
			gdl_entity_free (gl->_factor[i]);	
		}
		gdl_hashtable_free (gl->_haccession);
		gdl_hashtable_free (gl->_hfactor);
	}
}

gdl_flabels *
gdl_flabels_alloc (const gdl_fview * fview, const gdl_mask  * mask, gdl_boolean last_level)
{
	size_t i, j, n, l, a, ca, na;
	int * aidx;
	gdl_flabels * g;
	
	n = GDL_FVIEW_ACCESSION_SIZE (fview, mask);
	
	l = GDL_FVIEW_FACTOR_SIZE (fview, mask);
	
	aidx = GDL_MALLOC (int, l);
	
	for (a = i = 0; i < l; i++)
	{
		gdl_factor * factor = GDL_FVIEW_GET_FACTOR (fview, mask, i);
		
		if (gdl_factor_get_type (factor) != gdl_factor_continuous)
		{
			na = gdl_factor_size (factor);
			aidx[i] = a;
			if (last_level || na == 1)
			{
				a += na;
			}
			else
			{
				a += na-1;
			}
		}
		else
		{
			aidx[i] = -1;	
		}
		
	}
	
	g       = _gdl_flabels_alloc (n, l, a);
	g->aidx = aidx;
	
	for (i = 0; i < n; i++)
	{
		gdl_accession * acc = GDL_FVIEW_GET_ACCESSION (fview, mask, i);
		g->accession[i] = gdl_string_clone (gdl_entity_get_name (acc));
	}
	
	for (a = i = 0; i < l; i++)
	{
		gdl_factor * factor = GDL_FVIEW_GET_FACTOR (fview, mask, i);
		g->factor[i]        = gdl_string_clone (gdl_entity_get_name (factor));
		if (gdl_factor_get_type (factor)!= gdl_factor_continuous)
		{
			na = gdl_factor_size (factor);
			for (j = 0; j < na; j++)
			{
				if (j < na - 1 || last_level)
				{
					gdl_factor_level * all = gdl_factor_get (factor, j);
					g->level[a]            = gdl_string_clone (gdl_entity_get_name (all));
					g->cidx[a]             = GDL_MALLOC (size_t, 2);
					g->cidx[a][0] = i;
					g->cidx[a][1] = j;
					a++;
				}
	  		}
		}
	}
	
	_build_reverse_search (g);
	
	return g;
	
}

gdl_flabels *
gdl_flabels_wrapper_alloc (const gdl_fview_wrapper * fwrap, gdl_boolean last_level)
{
	size_t i, j, n, l, a, ca, na;
	int * aidx;
	gdl_flabels * g;
	
	n = gdl_fview_wrapper_accession_size (fwrap);
	
	l = gdl_fview_wrapper_factor_size (fwrap);
	
	aidx = GDL_MALLOC (int, l);
	
	for (a = i = 0; i < l; i++)
	{
		gdl_factor * factor = gdl_fview_wrapper_get_factor (fwrap, i);
		if (gdl_factor_get_type (factor) != gdl_factor_continuous)
		{
			na = gdl_factor_size (factor);
			aidx[i] = a;
			if (last_level || na == 1)
			{
				a += na;
			}
			else
			{
				a += na-1;
			}
		}
		else
		{
			aidx[i] = -1;	
		}
	}
	
	g       = _gdl_flabels_alloc (n, l, a);
	g->aidx = aidx;
	
	for (i = 0; i < n; i++)
	{
		gdl_accession * acc = gdl_fview_wrapper_get_accession (fwrap, i);
		g->accession[i]     = gdl_string_clone (gdl_entity_get_name (acc));
	}
	
	for (a = i = 0; i < l; i++)
	{
		gdl_factor * factor = gdl_fview_wrapper_get_factor (fwrap, i);
		g->factor[i]        = gdl_string_clone (gdl_entity_get_name (factor));
		if (gdl_factor_get_type (factor) != gdl_factor_continuous)
		{
			na = gdl_factor_size (factor);
			for (j = 0; j < na; j++)
			{
				if (j < na - 1 || last_level)
				{
					gdl_factor_level * all = gdl_factor_get (factor, j);
					g->level[a]      = gdl_string_clone (gdl_entity_get_name (all));
					g->cidx[a]       = GDL_MALLOC (size_t, 2);
					g->cidx[a][0] = i;
					g->cidx[a][1] = j;
					a++;
				}
	  		}
		}
	}
	
	_build_reverse_search (g);
	
	return g;
}

gdl_flabels *
gdl_flabels_wrapper_mask_alloc (const gdl_fview_wrapper * fwrap, const gdl_mask * mask, gdl_boolean last_level)
{
	size_t i, j, n, l, a, ca, na;
	int * aidx;
	gdl_flabels * g;
	
	n = gdl_mask_size (mask, GDL_ACCESSION);
	
	if (!n)
	{
		GDL_ERROR_VAL ("No accession in mask", GDL_FAILURE, 0);	
	}
	
	l = gdl_mask_size (mask, GDL_FACTOR);
	
	if (!l)
	{
		GDL_ERROR_VAL ("No factor in mask", GDL_FAILURE, 0);	
	}
	
	aidx = GDL_MALLOC (int, l);
	
	for (a = i = 0; i < l; i++)
	{
		gdl_factor * factor = gdl_fview_wrapper_get_factor (fwrap, gdl_mask_get_idx (mask, GDL_FACTOR, i));
		if (gdl_factor_get_type (factor) != gdl_factor_continuous)
		{
			na = gdl_factor_size (factor);
			aidx[i] = a;
			if (last_level || na == 1)
			{
				a += na;
			}
			else
			{
				a += na-1;
			}
		}
		else
		{
			aidx[i] = -1;	
		}
	}
	
	g       = _gdl_flabels_alloc (n, l, a);
	g->aidx = aidx;
	
	for (i = 0; i < n; i++)
	{
		gdl_accession * acc = gdl_fview_wrapper_get_accession (fwrap, gdl_mask_get_idx (mask, GDL_ACCESSION, i));
		g->accession[i]     = gdl_string_clone (gdl_entity_get_name (acc));
	}
	
	for (a = i = 0; i < l; i++)
	{
		gdl_factor * factor = gdl_fview_wrapper_get_factor (fwrap, gdl_mask_get_idx (mask, GDL_FACTOR, i));
		g->factor[i]        = gdl_string_clone (gdl_entity_get_name (factor));
		if (gdl_factor_get_type (factor) != gdl_factor_continuous)
		{
			na = gdl_factor_size (factor);
			for (j = 0; j < na; j++)
			{
				if (j < na - 1 || last_level)
				{
					gdl_factor_level * all = gdl_factor_get (factor, j);
					g->level[a]      = gdl_string_clone (gdl_entity_get_name (all));
					g->cidx[a]       = GDL_MALLOC (size_t, 2);
					g->cidx[a][0] = i;
					g->cidx[a][1] = j;
					a++;
				}
	  		}
		}
	}
	
	_build_reverse_search (g);
	
	return g;
	
}

static
_gdl_flabels_clean (gdl_string ** s, size_t n)
{
	size_t i;
	
	for (i = 0; i < n; i++)
	{
		gdl_string_free (s[i]);
	}	
}

void
gdl_flabels_free (gdl_flabels * gl)
{
	if (gl)
	{
		size_t i;
		_clean_reverse_search (gl);
		_gdl_flabels_clean (gl->factor,  gl->l);
		_gdl_flabels_clean (gl->level, gl->a);
		_gdl_flabels_clean (gl->accession, gl->n);
		GDL_FREE (gl->factor);
		GDL_FREE (gl->level);
		GDL_FREE (gl->accession);
		GDL_FREE (gl->aidx);
		for (i = 0; i < gl->a; i++)
		{
			GDL_FREE (gl->cidx[i]);
		}
		GDL_FREE (gl->cidx);
		GDL_FREE (gl);
	}	
}

size_t
gdl_flabels_accession_size (const gdl_flabels * gl)
{
	return gl->n;	
}

size_t
gdl_flabels_factor_size (const gdl_flabels * gl)
{
	return gl->l;	
}

size_t
gdl_flabels_factor_level_size (const gdl_flabels * gl, size_t l)
{
	if (gl->aidx[l]==-1)
	{
		return 0;	
	}
	else if (l + 1 < gl->l)
	{
		return gl->aidx[l+1]-gl->aidx[l];
	}
	else
	{
		return gl->a-gl->aidx[l];
	}
}

gdl_string *
gdl_flabels_accession (const gdl_flabels * gl, size_t a)
{
	return gl->accession[a];	
}

gdl_string *
gdl_flabels_factor (const gdl_flabels * gl, size_t l)
{
	return gl->factor[l];
}

gdl_string *
gdl_flabels_level (const gdl_flabels * gl, size_t l, size_t a)
{
	if (gl->aidx[l]==-1)
	{
		return NULL;
	}
	else
	{
		return gl->level[gl->aidx[l]+a];
	}
}

int
gdl_flabels_level_cumul_index (const gdl_flabels * gl, size_t a, size_t * lidx, size_t * aidx)
{
	*lidx = gl->cidx[a][0];
	*aidx = gl->cidx[a][1];
	return GDL_SUCCESS;
}

int
gdl_flabels_search_accession (const gdl_flabels * gl, const gdl_string * name)
{
	gdl_accession * accession = (gdl_accession *) gdl_hashtable_lookup (gl->_haccession, name);
	if (accession)
	{
		return gdl_entity_get_idx (accession);
	}
	else
	{
		return -1;	
	}
}

int
gdl_flabels_search_factor (const gdl_flabels * gl, const gdl_string * name)
{
	gdl_factor * factor = (gdl_factor *) gdl_hashtable_lookup (gl->_hfactor, name);
	if (factor)
	{
		return gdl_entity_get_idx (factor);
	}
	else
	{
		return -1;	
	}
}

int
gdl_flabels_search_level (const gdl_flabels * gl, size_t l, const gdl_string * name)
{
	gdl_factor       * factor    = gl->_factor[l];
	gdl_factor_level * level     = gdl_factor_search (factor, name);
	if (level)
	{
		return level->idx;
	}
	else
	{
		return -1;
	} 
}

static int
_gdl_flabels_fread (FILE * stream, gdl_string ** s, size_t n)
{
	size_t i;
	
	for (i = 0; i < n; i++)
	{
		s[i] = gdl_string_fread (stream);
		GDL_FWRITE_STATUS (s[i]!=0, 1);
	}
	
	return GDL_SUCCESS;
} 

gdl_flabels *
gdl_flabels_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i, n, l, a, ** cidx;
		int * aidx;
		gdl_flabels * gl;
		
		
		status = fread (&n, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&l, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&a, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		aidx   = GDL_MALLOC (int, l);
		status = fread (aidx, sizeof (int), l, stream);
		GDL_FREAD_STATUS (status, l);
		cidx   = GDL_MALLOC (size_t *, a);
		for (i = 0; i < a; i++)
		{
			cidx[i] = GDL_MALLOC (size_t, 2);
			status  = fread (cidx[i], sizeof (size_t), 2, stream);
			GDL_FREAD_STATUS (status, 2);
		}
		
		gl = _gdl_flabels_alloc (n, l, a);
		gl->aidx = aidx;
		gl->cidx = cidx;
		
		status = _gdl_flabels_fread (stream, gl->accession, gl->n);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		status =  _gdl_flabels_fread (stream, gl->factor, gl->l);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		status = _gdl_flabels_fread (stream, gl->level, gl->a);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		_build_reverse_search (gl);
		
		return gl;
	}
	return 0;
}

static int
_gdl_flabels_fwrite (FILE * stream, gdl_string ** s, size_t n)
{
	int status;
	size_t i;
	
	for (i = 0; i < n; i++)
	{
		status = gdl_string_fwrite (stream, s[i]);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	}
	
	return GDL_SUCCESS;
} 

int
gdl_flabels_fwrite (FILE * stream, const gdl_flabels * gl)
{
	if (stream && gl)
	{
		size_t i;
		int status;
		
		status = fwrite (&(gl->n), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(gl->l), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(gl->a), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (gl->aidx, sizeof (int), gl->l, stream);
		GDL_FWRITE_STATUS (status, gl->l);
		for (i = 0; i < gl->a; i++)
		{
			status = fwrite (gl->cidx[i], sizeof (size_t), 2, stream);
			GDL_FWRITE_STATUS (status, 2);
		}
		status = _gdl_flabels_fwrite (stream, gl->accession, gl->n);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = _gdl_flabels_fwrite (stream, gl->factor, gl->l);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = _gdl_flabels_fwrite (stream, gl->level, gl->a);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}
