/*  
 *  glabels/glabels.c
 *   
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:54 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_glabels.h>

struct _gdl_glabels
{
	size_t n;
	size_t l;
	gdl_hashtable * _accession;
	gdl_hashtable * _locus;
	gdl_locus     ** locus;
	gdl_accession ** accession;
	gdl_boolean   last_level;
};

static gdl_glabels *
_gdl_glabels_alloc (size_t n, size_t l, gdl_boolean last_level)
{
	gdl_glabels * gl;	
	
	gl = GDL_MALLOC (gdl_glabels, 1);
	
	gl->n = n;
	gl->l = l;
	
	gl->accession  = GDL_MALLOC (gdl_accession *, n);
	gl->locus      = GDL_MALLOC (gdl_locus *, l);
	gl->_accession = gdl_hashtable_alloc (gdl_hash_default, n);
	gl->_locus     = gdl_hashtable_alloc (gdl_hash_default, l);
	
	gl->last_level = last_level;
	
	return gl;
}

gdl_glabels *
gdl_glabels_alloc (const gdl_gview * gview,
                   const gdl_mask  * mask,
                   const gdl_clustering * clustering,
                   gdl_boolean last_level)
{
	size_t i, j, n, l, a, g, ca, na, ng, * aidx;
	gdl_glabels * gl;
	
	if (clustering)
	{
		n = gdl_clustering_nclust (clustering);
	}
	else
	{
		n = GDL_GVIEW_ACCESSION_SIZE (gview, mask);
	}
	
	l = GDL_GVIEW_LOCUS_SIZE (gview, mask);
	
	gl = _gdl_glabels_alloc (n, l, last_level);
	
	for (i = 0; i < n; i++)
	{
		gdl_accession * acc;
		if (clustering)
		{
			acc = GDL_GVIEW_GET_ACCESSION (gview, mask, gdl_clustering_clust_idx (clustering, i));
		}
		else
		{
			acc = GDL_GVIEW_GET_ACCESSION (gview, mask, i);
		}
		gl->accession[i] = gdl_entity_clone (acc);
		gl->accession[i]->idx = i;
		gdl_hashtable_add (gl->_accession, gdl_entity_get_name (gl->accession[i]), gl->accession[i], 0);
	}
	
	for (i = 0; i < l; i++)
	{
		gdl_locus * locus  = GDL_GVIEW_GET_LOCUS (gview, mask, i);
		gl->locus[i]       = gdl_entity_clone (locus);
		gl->locus[i]->idx  = i;
		gdl_hashtable_add (gl->_locus, gdl_entity_get_name (gl->locus[i]), gl->locus[i], 0);
	}
	
	return gl;
	
}

gdl_glabels *
gdl_glabels_wrapper_alloc (const gdl_gview_wrapper * gwrap,
                        gdl_boolean accession_cluster,
                        gdl_boolean last_level)
{
	size_t i, j, n, l, a, g, ca, na, ng, * aidx;
	gdl_glabels * gl;
	
	if (accession_cluster)
	{
		n = gdl_gview_wrapper_accession_size_c (gwrap);
	}
	else
	{
		n = gdl_gview_wrapper_accession_size (gwrap);
	}
	
	l = gdl_gview_wrapper_locus_size (gwrap);
	
	gl = _gdl_glabels_alloc (n, l, last_level);
	
	for (i = 0; i < n; i++)
	{
		const gdl_accession * acc;
		if (accession_cluster)
		{
			acc = gdl_gview_wrapper_get_accession_c (gwrap, i);
		}
		else
		{
			acc = gdl_gview_wrapper_get_accession (gwrap, i);
		}
		gl->accession[i] = gdl_entity_clone (acc);
		gl->accession[i]->idx = i;
		gdl_hashtable_add (gl->_accession, gdl_entity_get_name (gl->accession[i]), gl->accession[i], 0);
	}
	for (i = 0; i < l; i++)
	{
		const gdl_locus * locus  = gdl_gview_wrapper_get_locus (gwrap, i);
		gl->locus[i]       = gdl_entity_clone (locus);
		gl->locus[i]->idx  = i;
		gdl_hashtable_add (gl->_locus, gdl_entity_get_name (gl->locus[i]), gl->locus[i], 0);
	}
	
	return gl;
}

gdl_glabels *
gdl_glabels_wrapper_mask_alloc (const gdl_gview_wrapper * gwrap, const gdl_mask * mask, gdl_boolean last_level)
{
	size_t i, j, n, l, a, g, ca, na, ng, * aidx;
	gdl_glabels * gl;
	
	n = gdl_mask_size (mask, GDL_ACCESSION);
	if (!n)
	{
		n = gdl_gview_wrapper_accession_size (gwrap);
	}
	l = gdl_mask_size (mask, GDL_LOCUS);
	if (!l)
	{
		l = gdl_gview_wrapper_locus_size (gwrap);
	}
	
	gl = _gdl_glabels_alloc (n, l, last_level);
	
	for (i = 0; i < n; i++)
	{
	   gdl_accession * acc = gdl_gview_wrapper_get_accession (gwrap, gdl_mask_get_idx (mask, GDL_ACCESSION, i));
		gl->accession[i] = gdl_entity_clone (acc);
		gl->accession[i]->idx = i;
		gdl_hashtable_add (gl->_accession, gdl_entity_get_name (gl->accession[i]), gl->accession[i], 0);
	}
	for (i = 0; i < l; i++)
	{
		gdl_locus * locus  = gdl_gview_wrapper_get_locus (gwrap, gdl_mask_get_idx (mask, GDL_LOCUS, i));
		gl->locus[i]       = gdl_entity_clone (locus);
		gl->locus[i]->idx  = i;
		gdl_hashtable_add (gl->_locus, gdl_entity_get_name (gl->locus[i]), gl->locus[i], 0);
	}
	
	return gl;
	
}

gdl_glabels *
gdl_hlabels_alloc (const gdl_hview * hview, gdl_boolean accession_cluster, gdl_boolean last_level)
{
	if (accession_cluster)
	{
		return gdl_glabels_alloc (gdl_hview_get_gview (hview), gdl_hview_get_gmask (hview), gdl_hview_get_clustering (hview), last_level);
	}
	else
	{
		return gdl_glabels_alloc (gdl_hview_get_gview (hview), gdl_hview_get_gmask (hview), 0, last_level);
	}
}

void
gdl_glabels_free (gdl_glabels * gl)
{
	if (gl)
	{
		size_t i;
		gdl_hashtable_free (gl->_accession);
		gdl_hashtable_free (gl->_locus);
		for (i = 0; i < gl->n; i++)
		{
			gdl_entity_free (gl->accession[i]);	
		}
		for (i = 0; i < gl->l; i++)
		{
			gdl_entity_free (gl->locus[i]);	
		}
		GDL_FREE (gl->locus);
		GDL_FREE (gl->accession);
		GDL_FREE (gl);
	}	
}

size_t
gdl_glabels_accession_size (const gdl_glabels * gl)
{
	return gl->n;	
}

size_t
gdl_glabels_locus_size (const gdl_glabels * gl)
{
	return gl->l;	
}

size_t
gdl_glabels_locus_allele_size (const gdl_glabels * gl, size_t l)
{
	if (gl->last_level)
	{
		return gdl_locus_allele (gl->locus[l]);
	}
	else
	{
		return gdl_locus_allele (gl->locus[l])-1;
	}
}

size_t
gdl_glabels_locus_genotype_size (const gdl_glabels * gl, size_t l)
{
	if (gl->last_level)
	{
		return gdl_locus_genotype (gl->locus[l]);
	}
	else
	{
		return gdl_locus_genotype (gl->locus[l])-1;
	}
}

const gdl_string *
gdl_glabels_accession (const gdl_glabels * gl, size_t a)
{
	return gdl_entity_get_name (gl->accession[a]);
}

const gdl_string *
gdl_glabels_locus (const gdl_glabels * gl, size_t l)
{
	return gdl_entity_get_name (gl->locus[l]);
}

const gdl_string *
gdl_glabels_allele (const gdl_glabels * gl, size_t l, size_t a)
{
	return gdl_entity_get_name (gdl_locus_get_allele (gl->locus[l], a));
}

const gdl_string *
gdl_glabels_genotype (const gdl_glabels * gl, size_t l, size_t a)
{
	return gdl_entity_get_name (gdl_locus_get_genotype (gl->locus[l], a));
}

int
gdl_glabels_allele_cumul_index (const gdl_glabels * gl, size_t a, size_t * lidx, size_t * aidx)
{
	size_t i, j, na;
	for (na = i = 0; i < gl->l; i++)
	{
		na += gdl_glabels_locus_allele_size (gl, i);
		if (na > a)
		{
			*lidx = i;
			*aidx = gdl_glabels_locus_allele_size (gl, i)-(na-a)+1;
			return GDL_SUCCESS;
		}
	}
	return GDL_FAILURE;
}

int
gdl_glabels_genotype_cumul_index (const gdl_glabels * gl, size_t a, size_t * lidx, size_t * aidx)
{
	size_t i, j, na;
	for (na = i = 0; i < gl->l; i++)
	{
		na += gdl_glabels_locus_genotype_size (gl, i);
		if (na > a)
		{
			*lidx = i;
			*aidx = gdl_glabels_locus_genotype_size (gl, i)-(na-a)+1;
			return GDL_SUCCESS;
		}
	}
	return GDL_FAILURE;
}

int
gdl_glabels_search_accession (const gdl_glabels * gl, const gdl_string * name)
{
	gdl_accession * accession = (gdl_accession *) gdl_hashtable_lookup (gl->_accession, name);
	if (accession)
	{
		return gdl_entity_get_idx (accession);
	}
	else
	{
		return -1;	
	}
}

int
gdl_glabels_search_locus (const gdl_glabels * gl, const gdl_string * name)
{
	gdl_locus * locus = (gdl_locus *) gdl_hashtable_lookup (gl->_locus, name);
	if (locus)
	{
		return gdl_entity_get_idx (locus);
	}
	else
	{
		return -1;	
	}
}

int
gdl_glabels_search_allele (const gdl_glabels * gl, size_t l, const gdl_string * name)
{
	gdl_allele * allele = gdl_locus_search_allele (gl->locus[l], name);
	if (allele)
	{
		return allele->idx;
	}
	else
	{
		return -1;
	} 
}

int
gdl_glabels_search_genotype (const gdl_glabels * gl, size_t l, const gdl_string * name)
{
	gdl_genotype * genotype = gdl_locus_search_genotype (gl->locus[l], name);
	if (genotype)
	{
		return genotype->idx;
	}
	else
	{
		return -1;
	} 
}

gdl_glabels *
gdl_glabels_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i, n, l;
		gdl_boolean last;
		gdl_glabels * gl;
		
		status = fread (&n, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&l, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&last, sizeof (gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		gl = _gdl_glabels_alloc (n, l, last);
		
		for (i = 0; i < n; i++)
		{
			gl->accession[i] = gdl_entity_fread (stream);
			GDL_FREAD_STATUS (gl->accession[i]!=0, 1);
			gdl_hashtable_add (gl->_accession, gdl_entity_get_name (gl->accession[i]), gl->accession[i], 0);
		}
		
		for (i = 0; i < l; i++)
		{
			gl->locus[i] = gdl_entity_fread (stream);
			GDL_FREAD_STATUS (gl->locus[i]!=0, 1);
			gdl_hashtable_add (gl->_locus, gdl_entity_get_name (gl->locus[i]), gl->locus[i], 0);
		}
		
		return gl;
	}
	return 0;
}

int
gdl_glabels_fwrite (FILE * stream, const gdl_glabels * gl)
{
	if (stream && gl)
	{
		size_t i;
		int status;
		
		status = fwrite (&(gl->n), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(gl->l), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(gl->last_level), sizeof (gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < gl->n; i++)
		{
			status = gdl_entity_fwrite (stream, gl->accession[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		for (i = 0; i < gl->l; i++)
		{
			status = gdl_entity_fwrite (stream, gl->locus[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}
