/*  
 * 	lasso/lasso.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:08 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2008  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_mode.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_permutation.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_statistics_double.h>

#include <gdl/gdl_lasso.h>

gdl_lasso_workspace *
gdl_lasso_workspace_alloc (const size_t n, const size_t m)
{
	gdl_lasso_workspace * w;
	
	w = GDL_CALLOC (gdl_lasso_workspace, 1);
	
	w->N         = n;
	w->M         = m;
	w->mux       = GDL_CALLOC (double, m);
	w->model     = gdl_lasso_model_alloc (m);
	w->newx      = GDL_CALLOC (size_t, m);
	w->normx     = GDL_CALLOC (double, m);
	w->Sign      = gdl_vector_alloc (m);
	w->Cvec      = gdl_vector_alloc (m);
	w->C         = gdl_vector_alloc (m);
	w->W         = gdl_vector_alloc (m);
	w->Gi1       = gdl_vector_alloc (m);
	w->av        = gdl_vector_alloc (m);
	w->beta      = gdl_vector_alloc (m);
	w->beta_new  = gdl_vector_alloc (m);
	w->residuals = gdl_vector_alloc (n);
	w->U         = gdl_vector_alloc (n);
	
	return w;
}

void
gdl_lasso_workspace_free (gdl_lasso_workspace * w)
{
	if (w)
	{
		size_t i;
		
		gdl_lasso_model_free (w->model);
		GDL_FREE (w->newx);
		GDL_FREE (w->normx);
		w->Sign->size = w->M; 
		w->Cvec->size = w->M; 
		w->C->size    = w->M;
		w->W->size    = w->M;
		w->Gi1->size  = w->M;
		w->av->size   = w->M;
		w->beta->size = w->M;
		w->beta_new->size = w->M;
		w->residuals->size = w->N;
		w->U->size = w->N;
		gdl_vector_free (w->Sign);
		gdl_vector_free (w->Cvec);
		gdl_vector_free (w->C);
		gdl_vector_free (w->W);
		gdl_vector_free (w->Gi1);
		gdl_vector_free (w->av);
		gdl_vector_free (w->beta);
		gdl_vector_free (w->beta_new);
		gdl_vector_free (w->residuals);
		gdl_vector_free (w->U);
		gdl_vector_free (w->lambda);
		if (w->beta_pure)
		{
			for(i = 0; i < w->max_steps; i++)
    	 		GDL_FREE (w->beta_pure[i]);	
    	 	GDL_FREE (w->beta_pure);
	    	GDL_FREE (w->beta_pure_idx);
	    	GDL_FREE (w->beta_pure_size);
		}
		GDL_FREE (w->df);
		GDL_FREE (w->RSS);
  		GDL_FREE (w);
	}
}

struct _gdl_lasso_algorithm
{
	gdl_string * name;	
};
 
static gdl_lasso_algorithm _gdl_lasso_algorithm_lasso =
{
	"lasso"
};

const gdl_lasso_algorithm * gdl_lasso_algorithm_lasso = &_gdl_lasso_algorithm_lasso;


static gdl_lasso_algorithm _gdl_lasso_algorithm_lar =
{
	"lar"
};

const gdl_lasso_algorithm * gdl_lasso_algorithm_lar = &_gdl_lasso_algorithm_lar;

static gdl_lasso_algorithm _gdl_lasso_algorithm_frwstpw =
{
	"frwstpw"
};

const gdl_lasso_algorithm * gdl_lasso_algorithm_frwstpw = &_gdl_lasso_algorithm_frwstpw;

static gdl_lasso_algorithm _gdl_lasso_algorithm_stpw =
{
	"stpw"
};

const gdl_lasso_algorithm * gdl_lasso_algorithm_stpw = &_gdl_lasso_algorithm_stpw; 
 
static gdl_lasso_algorithm _gdl_lasso_algorithm_adaptive =
{
	"adaptive"
};

const gdl_lasso_algorithm * gdl_lasso_algorithm_adaptive = &_gdl_lasso_algorithm_adaptive;
