/* linalg/apply_givens.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2001 Gerard Jungman, Brian Gough
 * Copyright (C) 2004 Joerg Wensch, modifications for LQ.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

inline static void
apply_givens_qr (size_t M, size_t N, gdl_matrix * Q, gdl_matrix * R,
                 size_t i, size_t j, double c, double s)
{
  size_t k;

  /* Apply rotation to matrix Q,  Q' = Q G */

  for (k = 0; k < M; k++)
    {
      double qki = gdl_matrix_get (Q, k, i);
      double qkj = gdl_matrix_get (Q, k, j);
      gdl_matrix_set (Q, k, i, qki * c - qkj * s);
      gdl_matrix_set (Q, k, j, qki * s + qkj * c);
    }

  /* Apply rotation to matrix R, R' = G^T R (note: upper triangular so
     zero for column < row) */

  for (k = GDL_MIN (i, j); k < N; k++)
    {
      double rik = gdl_matrix_get (R, i, k);
      double rjk = gdl_matrix_get (R, j, k);
      gdl_matrix_set (R, i, k, c * rik - s * rjk);
      gdl_matrix_set (R, j, k, s * rik + c * rjk);
    }
}

inline static void
apply_givens_lq (size_t M, size_t N, gdl_matrix * Q, gdl_matrix * L,
                 size_t i, size_t j, double c, double s)
{
  size_t k;

  /* Apply rotation to matrix Q,  Q' = G Q */

  for (k = 0; k < M; k++)
    {
      double qik = gdl_matrix_get (Q, i, k);
      double qjk = gdl_matrix_get (Q, j, k);
      gdl_matrix_set (Q, i, k, qik * c - qjk * s);
      gdl_matrix_set (Q, j, k, qik * s + qjk * c);
    }

  /* Apply rotation to matrix L, L' = L G^T (note: lower triangular so
     zero for column > row) */

  for (k = GDL_MIN (i, j); k < N; k++)
    {
      double lki = gdl_matrix_get (L, k, i);
      double lkj = gdl_matrix_get (L, k, j);
      gdl_matrix_set (L, k, i, c * lki - s * lkj);
      gdl_matrix_set (L, k, j, s * lki + c * lkj);
    }
}

inline static void
apply_givens_vec (gdl_vector * v, size_t i, size_t j, double c, double s)
{
  /* Apply rotation to vector v' = G^T v */

  double vi = gdl_vector_get (v, i);
  double vj = gdl_vector_get (v, j);
  gdl_vector_set (v, i, c * vi - s * vj);
  gdl_vector_set (v, j, s * vi + c * vj);
}

