/* Cholesky Decomposition
 *
 * Copyright (C) 2000  Thomas Walter
 *
 * 3 May 2000: Modified for GDL by Brian Gough
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This source is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

/*
 * Cholesky decomposition of a symmetrix positive definite matrix.
 * This is useful to solve the matrix arising in
 *    periodic cubic splines
 *    approximating splines
 *
 * This algorthm does:
 *   A = L * L'
 * with
 *   L  := lower left triangle matrix
 *   L' := the transposed form of L.
 *
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_linalg.h>

int
gdl_linalg_cholesky_decomp (gdl_matrix * A)
{
  const size_t M = A->size1;
  const size_t N = A->size2;

  if (M != N)
    {
      GDL_ERROR("cholesky decomposition requires square matrix", GDL_ENOTSQR);
    }
  else
    {
      size_t i,j,k;
      int status = 0;

      /* Do the first 2 rows explicitly.  It is simple, and faster.  And
       * one can return if the matrix has only 1 or 2 rows.  
       */

      double A_00 = gdl_matrix_get (A, 0, 0);
      
      double L_00 = sqrt(A_00);
      
      if (A_00 <= 0)
        {
          status = GDL_EDOM ;
        }

      gdl_matrix_set (A, 0, 0, L_00);
  
      if (M > 1)
        {
          double A_10 = gdl_matrix_get (A, 1, 0);
          double A_11 = gdl_matrix_get (A, 1, 1);
          
          double L_10 = A_10 / L_00;
          double diag = A_11 - L_10 * L_10;
          double L_11 = sqrt(diag);
          
          if (diag <= 0)
            {
              status = GDL_EDOM;
            }

          gdl_matrix_set (A, 1, 0, L_10);        
          gdl_matrix_set (A, 1, 1, L_11);
        }
      
      for (k = 2; k < M; k++)
        {
          double A_kk = gdl_matrix_get (A, k, k);
          
          for (i = 0; i < k; i++)
            {
              double sum = 0;

              double A_ki = gdl_matrix_get (A, k, i);
              double A_ii = gdl_matrix_get (A, i, i);

              gdl_vector_view ci = gdl_matrix_row (A, i);
              gdl_vector_view ck = gdl_matrix_row (A, k);

              if (i > 0) {
                gdl_vector_view di = gdl_vector_subvector(&ci.vector, 0, i);
                gdl_vector_view dk = gdl_vector_subvector(&ck.vector, 0, i);
                
                gdl_blas_ddot (&di.vector, &dk.vector, &sum);
              }

              A_ki = (A_ki - sum) / A_ii;
              gdl_matrix_set (A, k, i, A_ki);
            } 

          {
            gdl_vector_view ck = gdl_matrix_row (A, k);
            gdl_vector_view dk = gdl_vector_subvector (&ck.vector, 0, k);
            
            double sum = gdl_blas_dnrm2 (&dk.vector);
            double diag = A_kk - sum * sum;

            double L_kk = sqrt(diag);
            
            if (diag <= 0)
              {
                status = GDL_EDOM;
              }
            
            gdl_matrix_set (A, k, k, L_kk);
          }
        }

      /* Now copy the transposed lower triangle to the upper triangle,
       * the diagonal is common.  
       */
      
      for (i = 1; i < M; i++)
        {
          for (j = 0; j < i; j++)
            {
              double A_ij = gdl_matrix_get (A, i, j);
              gdl_matrix_set (A, j, i, A_ij);
            }
        } 
      
      if (status == GDL_EDOM)
        {
          GDL_ERROR ("matrix must be positive definite", GDL_EDOM);
        }
      
      return GDL_SUCCESS;
    }
}


int
gdl_linalg_cholesky_solve (const gdl_matrix * LLT,
                           const gdl_vector * b,
                           gdl_vector * x)
{
  if (LLT->size1 != LLT->size2)
    {
      GDL_ERROR ("cholesky matrix must be square", GDL_ENOTSQR);
    }
  else if (LLT->size1 != b->size)
    {
      GDL_ERROR ("matrix size must match b size", GDL_EBADLEN);
    }
  else if (LLT->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else
    {
      /* Copy x <- b */

      gdl_vector_memcpy (x, b);

      /* Solve for c using forward-substitution, L c = b */

      gdl_blas_dtrsv (CblasLower, CblasNoTrans, CblasNonUnit, LLT, x);

      /* Perform back-substitution, U x = c */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, LLT, x);


      return GDL_SUCCESS;
    }
}

int
gdl_linalg_cholesky_svx (const gdl_matrix * LLT,
                         gdl_vector * x)
{
  if (LLT->size1 != LLT->size2)
    {
      GDL_ERROR ("cholesky matrix must be square", GDL_ENOTSQR);
    }
  else if (LLT->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else
    {
      /* Solve for c using forward-substitution, L c = b */

      gdl_blas_dtrsv (CblasLower, CblasNoTrans, CblasNonUnit, LLT, x);

      /* Perform back-substitution, U x = c */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, LLT, x);

      return GDL_SUCCESS;
    }
}



