/* linalg/hh.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Author:  G. Jungman */

#include <stdlib.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_linalg.h>

#define REAL double

/* [Engeln-Mullges + Uhlig, Alg. 4.42]
 */

int
gdl_linalg_HH_solve (gdl_matrix * A, const gdl_vector * b, gdl_vector * x)
{
  if (A->size1 > A->size2)
    {
      /* System is underdetermined. */

      GDL_ERROR ("System is underdetermined", GDL_EINVAL);
    }
  else if (A->size2 != x->size)
    {
      GDL_ERROR ("matrix and vector sizes must be equal", GDL_EBADLEN);
    }
  else
    {
      int status ;

      gdl_vector_memcpy (x, b);

      status = gdl_linalg_HH_svx (A, x);
      
      return status ;
    }
}

int
gdl_linalg_HH_svx (gdl_matrix * A, gdl_vector * x)
{
  if (A->size1 > A->size2)
    {
      /* System is underdetermined. */

      GDL_ERROR ("System is underdetermined", GDL_EINVAL);
    }
  else if (A->size2 != x->size)
    {
      GDL_ERROR ("matrix and vector sizes must be equal", GDL_EBADLEN);
    }
  else
    {
      const size_t N = A->size1;
      const size_t M = A->size2;
      size_t i, j, k;
      REAL *d = (REAL *) malloc (N * sizeof (REAL));

      if (d == 0)
        {
          GDL_ERROR ("could not allocate memory for workspace", GDL_ENOMEM);
        }

      /* Perform Householder transformation. */

      for (i = 0; i < N; i++)
        {
          const REAL aii = gdl_matrix_get (A, i, i);
          REAL alpha;
          REAL f;
          REAL ak;
          REAL max_norm = 0.0;
          REAL r = 0.0;

          for (k = i; k < M; k++)
            {
              REAL aki = gdl_matrix_get (A, k, i);
              r += aki * aki;
            }

          if (r == 0.0)
            {
              /* Rank of matrix is less than size1. */
              free (d);
              GDL_ERROR ("matrix is rank deficient", GDL_ESING);
            }

          alpha = sqrt (r) * GDL_SIGN (aii);

          ak = 1.0 / (r + alpha * aii);
          gdl_matrix_set (A, i, i, aii + alpha);

          d[i] = -alpha;

          for (k = i + 1; k < N; k++)
            {
              REAL norm = 0.0;
              f = 0.0;
              for (j = i; j < M; j++)
                {
                  REAL ajk = gdl_matrix_get (A, j, k);
                  REAL aji = gdl_matrix_get (A, j, i);
                  norm += ajk * ajk;
                  f += ajk * aji;
                }
              max_norm = GDL_MAX (max_norm, norm);

              f *= ak;

              for (j = i; j < M; j++)
                {
                  REAL ajk = gdl_matrix_get (A, j, k);
                  REAL aji = gdl_matrix_get (A, j, i);
                  gdl_matrix_set (A, j, k, ajk - f * aji);
                }
            }

          if (fabs (alpha) < 2.0 * GDL_DBL_EPSILON * sqrt (max_norm))
            {
              /* Apparent singularity. */
              free (d);
              GDL_ERROR("apparent singularity detected", GDL_ESING);
            }

          /* Perform update of RHS. */

          f = 0.0;
          for (j = i; j < M; j++)
            {
              f += gdl_vector_get (x, j) * gdl_matrix_get (A, j, i);
            }
          f *= ak;
          for (j = i; j < M; j++)
            {
              REAL xj = gdl_vector_get (x, j);
              REAL aji = gdl_matrix_get (A, j, i);
              gdl_vector_set (x, j, xj - f * aji);
            }
        }

      /* Perform back-substitution. */

      for (i = N; i > 0 && i--;)
        {
          REAL xi = gdl_vector_get (x, i);
          REAL sum = 0.0;
          for (k = i + 1; k < N; k++)
            {
              sum += gdl_matrix_get (A, i, k) * gdl_vector_get (x, k);
            }

          gdl_vector_set (x, i, (xi - sum) / d[i]);
        }

      free (d);
      return GDL_SUCCESS;
    }
}

