/* linalg/householder.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2004 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_blas.h>

#include <gdl/gdl_linalg.h>

double
gdl_linalg_householder_transform (gdl_vector * v)
{
  /* replace v[0:n-1] with a householder vector (v[0:n-1]) and
     coefficient tau that annihilate v[1:n-1] */

  const size_t n = v->size ;

  if (n == 1)
    {
      return 0.0; /* tau = 0 */
    }
  else
    { 
      double alpha, beta, tau ;
      
      gdl_vector_view x = gdl_vector_subvector (v, 1, n - 1) ; 
      
      double xnorm = gdl_blas_dnrm2 (&x.vector);
      
      if (xnorm == 0) 
        {
          return 0.0; /* tau = 0 */
        }
      
      alpha = gdl_vector_get (v, 0) ;
      beta = - (alpha >= 0.0 ? +1.0 : -1.0) * hypot(alpha, xnorm) ;
      tau = (beta - alpha) / beta ;
      
      gdl_blas_dscal (1.0 / (alpha - beta), &x.vector);
      gdl_vector_set (v, 0, beta) ;
      
      return tau;
    }
}

int
gdl_linalg_householder_hm (double tau, const gdl_vector * v, gdl_matrix * A)
{
  /* applies a householder transformation v,tau to matrix m */

  size_t i, j;

  if (tau == 0.0)
    {
      return GDL_SUCCESS;
    }

#ifdef USE_BLAS
  {
    gdl_vector_const_view v1 = gdl_vector_const_subvector (v, 1, v->size - 1);
    gdl_matrix_view A1 = gdl_matrix_submatrix (A, 1, 0, A->size1 - 1, A->size2);
    
    for (j = 0; j < A->size2; j++)
      {
        double wj = 0.0;
        gdl_vector_view A1j = gdl_matrix_column(&A1.matrix, j);
        gdl_blas_ddot (&A1j.vector, &v1.vector, &wj);
        wj += gdl_matrix_get(A,0,j);

        {
          double A0j = gdl_matrix_get (A, 0, j);
          gdl_matrix_set (A, 0, j, A0j - tau *  wj);
        }

        gdl_blas_daxpy (-tau * wj, &v1.vector, &A1j.vector);
      }
  }
#else
  for (j = 0; j < A->size2; j++)
    {
      /* Compute wj = Akj vk */

      double wj = gdl_matrix_get(A,0,j);  

      for (i = 1; i < A->size1; i++)  /* note, computed for v(0) = 1 above */
        {
          wj += gdl_matrix_get(A,i,j) * gdl_vector_get(v,i);
        }

      /* Aij = Aij - tau vi wj */

      /* i = 0 */
      {
        double A0j = gdl_matrix_get (A, 0, j);
        gdl_matrix_set (A, 0, j, A0j - tau *  wj);
      }

      /* i = 1 .. M-1 */

      for (i = 1; i < A->size1; i++)
        {
          double Aij = gdl_matrix_get (A, i, j);
          double vi = gdl_vector_get (v, i);
          gdl_matrix_set (A, i, j, Aij - tau * vi * wj);
        }
    }
#endif

  return GDL_SUCCESS;
}

int
gdl_linalg_householder_mh (double tau, const gdl_vector * v, gdl_matrix * A)
{
  /* applies a householder transformation v,tau to matrix m from the
     right hand side in order to zero out rows */

  size_t i, j;

  if (tau == 0)
    return GDL_SUCCESS;

  /* A = A - tau w v' */

#ifdef USE_BLAS
  {
    gdl_vector_const_view v1 = gdl_vector_const_subvector (v, 1, v->size - 1);
    gdl_matrix_view A1 = gdl_matrix_submatrix (A, 0, 1, A->size1, A->size2-1);

    for (i = 0; i < A->size1; i++)
      {
        double wi = 0.0;
        gdl_vector_view A1i = gdl_matrix_row(&A1.matrix, i);
        gdl_blas_ddot (&A1i.vector, &v1.vector, &wi);
        wi += gdl_matrix_get(A,i,0);  
        
        {
          double Ai0 = gdl_matrix_get (A, i, 0);
          gdl_matrix_set (A, i, 0, Ai0 - tau *  wi);
        }
        
        gdl_blas_daxpy(-tau * wi, &v1.vector, &A1i.vector);
      }
  }
#else
  for (i = 0; i < A->size1; i++)
    {
      double wi = gdl_matrix_get(A,i,0);  

      for (j = 1; j < A->size2; j++)  /* note, computed for v(0) = 1 above */
        {
          wi += gdl_matrix_get(A,i,j) * gdl_vector_get(v,j);
        }
      
      /* j = 0 */
      
      {
        double Ai0 = gdl_matrix_get (A, i, 0);
        gdl_matrix_set (A, i, 0, Ai0 - tau *  wi);
      }

      /* j = 1 .. N-1 */
      
      for (j = 1; j < A->size2; j++) 
        {
          double vj = gdl_vector_get (v, j);
          double Aij = gdl_matrix_get (A, i, j);
          gdl_matrix_set (A, i, j, Aij - tau * wi * vj);
        }
    }
#endif

  return GDL_SUCCESS;
}

int
gdl_linalg_householder_hv (double tau, const gdl_vector * v, gdl_vector * w)
{
  /* applies a householder transformation v to vector w */
  const size_t N = v->size;
 
  if (tau == 0)
    return GDL_SUCCESS ;

  {
    /* compute d = v'w */

    double d0 = gdl_vector_get(w,0);
    double d1, d;

    gdl_vector_const_view v1 = gdl_vector_const_subvector(v, 1, N-1);
    gdl_vector_view w1 = gdl_vector_subvector(w, 1, N-1);

    gdl_blas_ddot (&v1.vector, &w1.vector, &d1);
    
    d = d0 + d1;

    /* compute w = w - tau (v) (v'w) */
  
    {
      double w0 = gdl_vector_get (w,0);
      gdl_vector_set (w, 0, w0 - tau * d);
    }
    
    gdl_blas_daxpy (-tau * d, &v1.vector, &w1.vector);
  }
  
  return GDL_SUCCESS;
}


int
gdl_linalg_householder_hm1 (double tau, gdl_matrix * A)
{
  /* applies a householder transformation v,tau to a matrix being
     build up from the identity matrix, using the first column of A as
     a householder vector */

  size_t i, j;

  if (tau == 0)
    {
      gdl_matrix_set (A, 0, 0, 1.0);
      
      for (j = 1; j < A->size2; j++)
        {
          gdl_matrix_set (A, 0, j, 0.0);
        }

      for (i = 1; i < A->size1; i++)
        {
          gdl_matrix_set (A, i, 0, 0.0);
        }

      return GDL_SUCCESS;
    }

  /* w = A' v */

#ifdef USE_BLAS
  {
    gdl_matrix_view A1 = gdl_matrix_submatrix (A, 1, 0, A->size1 - 1, A->size2);
    gdl_vector_view v1 = gdl_matrix_column (&A1.matrix, 0);

    for (j = 1; j < A->size2; j++)
      {
        double wj = 0.0;   /* A0j * v0 */
        
        gdl_vector_view A1j = gdl_matrix_column(&A1.matrix, j);
        gdl_blas_ddot (&A1j.vector, &v1.vector, &wj);

        /* A = A - tau v w' */
        
        gdl_matrix_set (A, 0, j, - tau *  wj);
        
        gdl_blas_daxpy(-tau*wj, &v1.vector, &A1j.vector);
      }

    gdl_blas_dscal(-tau, &v1.vector);
    
    gdl_matrix_set (A, 0, 0, 1.0 - tau);
  }
#else
  for (j = 1; j < A->size2; j++)
    {
      double wj = 0.0;   /* A0j * v0 */

      for (i = 1; i < A->size1; i++)
        {
          double vi = gdl_matrix_get(A, i, 0);
          wj += gdl_matrix_get(A,i,j) * vi;
        }

      /* A = A - tau v w' */

      gdl_matrix_set (A, 0, j, - tau *  wj);
      
      for (i = 1; i < A->size1; i++)
        {
          double vi = gdl_matrix_get (A, i, 0);
          double Aij = gdl_matrix_get (A, i, j);
          gdl_matrix_set (A, i, j, Aij - tau * vi * wj);
        }
    }

  for (i = 1; i < A->size1; i++)
    {
      double vi = gdl_matrix_get(A, i, 0);
      gdl_matrix_set(A, i, 0, -tau * vi);
    }

  gdl_matrix_set (A, 0, 0, 1.0 - tau);
#endif

  return GDL_SUCCESS;
}
