/* linalg/qrpt.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_blas.h>

#include <gdl/gdl_linalg.h>

#define REAL double

#include "givens.c"
#include "apply_givens.c"

/* Factorise a general M x N matrix A into
 *
 *   A P = Q R
 *
 * where Q is orthogonal (M x M) and R is upper triangular (M x N).
 * When A is rank deficient, r = rank(A) < n, then the permutation is
 * used to ensure that the lower n - r rows of R are zero and the first
 * r columns of Q form an orthonormal basis for A.
 *
 * Q is stored as a packed set of Householder transformations in the
 * strict lower triangular part of the input matrix.
 *
 * R is stored in the diagonal and upper triangle of the input matrix.
 *
 * P: column j of P is column k of the identity matrix, where k =
 * permutation->data[j]
 *
 * The full matrix for Q can be obtained as the product
 *
 *       Q = Q_k .. Q_2 Q_1
 *
 * where k = MIN(M,N) and
 *
 *       Q_i = (I - tau_i * v_i * v_i')
 *
 * and where v_i is a Householder vector
 *
 *       v_i = [1, m(i+1,i), m(i+2,i), ... , m(M,i)]
 *
 * This storage scheme is the same as in LAPACK.  See LAPACK's
 * dgeqpf.f for details.
 * 
 */

int
gdl_linalg_QRPT_decomp (gdl_matrix * A, gdl_vector * tau, gdl_permutation * p, int *signum, gdl_vector * norm)
{
  const size_t M = A->size1;
  const size_t N = A->size2;

  if (tau->size != GDL_MIN (M, N))
    {
      GDL_ERROR ("size of tau must be MIN(M,N)", GDL_EBADLEN);
    }
  else if (p->size != N)
    {
      GDL_ERROR ("permutation size must be N", GDL_EBADLEN);
    }
  else if (norm->size != N)
    {
      GDL_ERROR ("norm size must be N", GDL_EBADLEN);
    }
  else
    {
      size_t i;

      *signum = 1;

      gdl_permutation_init (p); /* set to identity */

      /* Compute column norms and store in workspace */

      for (i = 0; i < N; i++)
        {
          gdl_vector_view c = gdl_matrix_column (A, i);
          double x = gdl_blas_dnrm2 (&c.vector);
          gdl_vector_set (norm, i, x);
        }

      for (i = 0; i < GDL_MIN (M, N); i++)
        {
          /* Bring the column of largest norm into the pivot position */

          double max_norm = gdl_vector_get(norm, i);
          size_t j, kmax = i;

          for (j = i + 1; j < N; j++)
            {
              double x = gdl_vector_get (norm, j);

              if (x > max_norm)
                {
                  max_norm = x;
                  kmax = j;
                }
            }

          if (kmax != i)
            {
              gdl_matrix_swap_columns (A, i, kmax);
              gdl_permutation_swap (p, i, kmax);
              gdl_vector_swap_elements(norm,i,kmax);

              (*signum) = -(*signum);
            }

          /* Compute the Householder transformation to reduce the j-th
             column of the matrix to a multiple of the j-th unit vector */

          {
            gdl_vector_view c_full = gdl_matrix_column (A, i);
            gdl_vector_view c = gdl_vector_subvector (&c_full.vector, 
                                                      i, M - i);
            double tau_i = gdl_linalg_householder_transform (&c.vector);

            gdl_vector_set (tau, i, tau_i);

            /* Apply the transformation to the remaining columns */

            if (i + 1 < N)
              {
                gdl_matrix_view m = gdl_matrix_submatrix (A, i, i + 1, M - i, N - (i+1));

                gdl_linalg_householder_hm (tau_i, &c.vector, &m.matrix);
              }
          }

          /* Update the norms of the remaining columns too */

          if (i + 1 < M) 
            {
              for (j = i + 1; j < N; j++)
                {
                  double x = gdl_vector_get (norm, j);

                  if (x > 0.0)
                    {
                      double y = 0;
                      double temp= gdl_matrix_get (A, i, j) / x;
                  
                      if (fabs (temp) >= 1)
                        y = 0.0;
                      else
                        y = x * sqrt (1 - temp * temp);
                      
                      /* recompute norm to prevent loss of accuracy */

                      if (fabs (y / x) < sqrt (20.0) * GDL_SQRT_DBL_EPSILON)
                        {
                          gdl_vector_view c_full = gdl_matrix_column (A, j);
                          gdl_vector_view c = 
                            gdl_vector_subvector(&c_full.vector,
                                                 i+1, M - (i+1));
                          y = gdl_blas_dnrm2 (&c.vector);
                        }
                  
                      gdl_vector_set (norm, j, y);
                    }
                }
            }
        }

      return GDL_SUCCESS;
    }
}

int
gdl_linalg_QRPT_decomp2 (const gdl_matrix * A, gdl_matrix * q, gdl_matrix * r, gdl_vector * tau, gdl_permutation * p, int *signum, gdl_vector * norm)
{
  const size_t M = A->size1;
  const size_t N = A->size2;

  if (q->size1 != M || q->size2 !=M) 
    {
      GDL_ERROR ("q must be M x M", GDL_EBADLEN);
    }
  else if (r->size1 != M || r->size2 !=N)
    {
      GDL_ERROR ("r must be M x N", GDL_EBADLEN);
    }
  else if (tau->size != GDL_MIN (M, N))
    {
      GDL_ERROR ("size of tau must be MIN(M,N)", GDL_EBADLEN);
    }
  else if (p->size != N)
    {
      GDL_ERROR ("permutation size must be N", GDL_EBADLEN);
    }
  else if (norm->size != N)
    {
      GDL_ERROR ("norm size must be N", GDL_EBADLEN);
    }

  gdl_matrix_memcpy (r, A);

  gdl_linalg_QRPT_decomp (r, tau, p, signum, norm);

  /* FIXME:  aliased arguments depends on behavior of unpack routine! */

  gdl_linalg_QR_unpack (r, tau, q, r);

  return GDL_SUCCESS;
}


/* Solves the system A x = b using the Q R P^T factorisation,

   R z = Q^T b

   x = P z;

   to obtain x. Based on SLATEC code. */

int
gdl_linalg_QRPT_solve (const gdl_matrix * QR,
                       const gdl_vector * tau,
                       const gdl_permutation * p,
                       const gdl_vector * b,
                       gdl_vector * x)
{
  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size1 != p->size)
    {
      GDL_ERROR ("matrix size must match permutation size", GDL_EBADLEN);
    }
  else if (QR->size1 != b->size)
    {
      GDL_ERROR ("matrix size must match b size", GDL_EBADLEN);
    }
  else if (QR->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else
    {
      gdl_vector_memcpy (x, b);

      gdl_linalg_QRPT_svx (QR, tau, p, x);
      
      return GDL_SUCCESS;
    }
}

int
gdl_linalg_QRPT_svx (const gdl_matrix * QR,
                     const gdl_vector * tau,
                     const gdl_permutation * p,
                     gdl_vector * x)
{
  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size1 != p->size)
    {
      GDL_ERROR ("matrix size must match permutation size", GDL_EBADLEN);
    }
  else if (QR->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match solution size", GDL_EBADLEN);
    }
  else
    {
      /* compute sol = Q^T b */

      gdl_linalg_QR_QTvec (QR, tau, x);

      /* Solve R x = sol, storing x inplace in sol */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, QR, x);

      gdl_permute_vector_inverse (p, x);

      return GDL_SUCCESS;
    }
}


int
gdl_linalg_QRPT_QRsolve (const gdl_matrix * Q, const gdl_matrix * R,
                         const gdl_permutation * p,
                         const gdl_vector * b,
                         gdl_vector * x)
{
  if (Q->size1 != Q->size2 || R->size1 != R->size2)
    {
      return GDL_ENOTSQR;
    }
  else if (Q->size1 != p->size || Q->size1 != R->size1
           || Q->size1 != b->size)
    {
      return GDL_EBADLEN;
    }
  else
    {
      /* compute b' = Q^T b */

      gdl_blas_dgemv (CblasTrans, 1.0, Q, b, 0.0, x);

      /* Solve R x = b', storing x inplace */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, R, x);

      /* Apply permutation to solution in place */

      gdl_permute_vector_inverse (p, x);

      return GDL_SUCCESS;
    }
}

int
gdl_linalg_QRPT_Rsolve (const gdl_matrix * QR,
                        const gdl_permutation * p,
                        const gdl_vector * b,
                        gdl_vector * x)
{
  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size1 != b->size)
    {
      GDL_ERROR ("matrix size must match b size", GDL_EBADLEN);
    }
  else if (QR->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match x size", GDL_EBADLEN);
    }
  else if (p->size != x->size)
    {
      GDL_ERROR ("permutation size must match x size", GDL_EBADLEN);
    }
  else
    {
      /* Copy x <- b */

      gdl_vector_memcpy (x, b);

      /* Solve R x = b, storing x inplace */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, QR, x);

      gdl_permute_vector_inverse (p, x);

      return GDL_SUCCESS;
    }
}


int
gdl_linalg_QRPT_Rsvx (const gdl_matrix * QR,
                      const gdl_permutation * p,
                      gdl_vector * x)
{
  if (QR->size1 != QR->size2)
    {
      GDL_ERROR ("QR matrix must be square", GDL_ENOTSQR);
    }
  else if (QR->size2 != x->size)
    {
      GDL_ERROR ("matrix size must match x size", GDL_EBADLEN);
    }
  else if (p->size != x->size)
    {
      GDL_ERROR ("permutation size must match x size", GDL_EBADLEN);
    }
  else
    {
      /* Solve R x = b, storing x inplace */

      gdl_blas_dtrsv (CblasUpper, CblasNoTrans, CblasNonUnit, QR, x);

      gdl_permute_vector_inverse (p, x);

      return GDL_SUCCESS;
    }
}



/* Update a Q R P^T factorisation for A P= Q R ,  A' = A + u v^T,

   Q' R' P^-1 = QR P^-1 + u v^T
   = Q (R + Q^T u v^T P ) P^-1
   = Q (R + w v^T P) P^-1

   where w = Q^T u.

   Algorithm from Golub and Van Loan, "Matrix Computations", Section
   12.5 (Updating Matrix Factorizations, Rank-One Changes)  */

int
gdl_linalg_QRPT_update (gdl_matrix * Q, gdl_matrix * R,
                        const gdl_permutation * p,
                        gdl_vector * w, const gdl_vector * v)
{
  if (Q->size1 != Q->size2 || R->size1 != R->size2)
    {
      return GDL_ENOTSQR;
    }
  else if (R->size1 != Q->size2 || v->size != Q->size2 || w->size != Q->size2)
    {
      return GDL_EBADLEN;
    }
  else
    {
      size_t j, k;
      const size_t M = Q->size1;
      const size_t N = Q->size2;
      double w0;

      /* Apply Given's rotations to reduce w to (|w|, 0, 0, ... , 0) 

         J_1^T .... J_(n-1)^T w = +/- |w| e_1

         simultaneously applied to R,  H = J_1^T ... J^T_(n-1) R
         so that H is upper Hessenberg.  (12.5.2) */

      for (k = N - 1; k > 0; k--)
        {
          double c, s;
          double wk = gdl_vector_get (w, k);
          double wkm1 = gdl_vector_get (w, k - 1);

          create_givens (wkm1, wk, &c, &s);
          apply_givens_vec (w, k - 1, k, c, s);
          apply_givens_qr (M, N, Q, R, k - 1, k, c, s);
        }

      w0 = gdl_vector_get (w, 0);

      /* Add in w v^T  (Equation 12.5.3) */

      for (j = 0; j < N; j++)
        {
          double r0j = gdl_matrix_get (R, 0, j);
          size_t p_j = gdl_permutation_get (p, j);
          double vj = gdl_vector_get (v, p_j);
          gdl_matrix_set (R, 0, j, r0j + w0 * vj);
        }

      /* Apply Givens transformations R' = G_(n-1)^T ... G_1^T H  
         Equation 12.5.4 */

      for (k = 1; k < N; k++)
        {
          double c, s;
          double diag = gdl_matrix_get (R, k - 1, k - 1);
          double offdiag = gdl_matrix_get (R, k, k - 1);

          create_givens (diag, offdiag, &c, &s);
          apply_givens_qr (M, N, Q, R, k - 1, k, c, s);
        }

      return GDL_SUCCESS;
    }
}
