/*  
 *  linalg/svd_sis2.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:08 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 *  
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h> 

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_linalg_svd.h>

typedef struct 
{
	size_t M;
	size_t N;
	int t;				/* transpose mode 									*/
	long em;            /* Number of eigenpairs of B approximated           */
	long em2;           /* Number of desired eigenpairs                     */
	long numextra;      /* Number of extra vectors to carry                 */
	long km;            /* Maximum number of iterations                     */
	long imem;          /* The number of bytes needed for this invocation   */ 
	long p;             /* Initial subspace dimension = em2 + numextra      */
	double eps;         /* Tolerance for egeinpairs residuals               */
	double *d;          /* array of eigenvalues of B                        */
	double *f;          /* temporary storage array (p)                      */
	double *cx;         /* temporary storage array (ncol)                   */
	double *u;          /* work array                                       */
	double **x;         /* 2-dimensional array of iteration vectors         */
} sis2_t;

#include "ritzit.c"
/**
 * 
 * 
 */
static int 
sis2_decomp (sis2_t * sis2, gdl_matrix * A, gdl_matrix * V, gdl_vector * S)
{
	long j,i,M, N;
	double tmp1, tmp2, xnorm;
	
	M = A->size1;
	N = A->size2;
	
	 /* Handle the case of N = 1 (SVD of a column vector) */

  if (N == 1)
    {
      gdl_vector_view column = gdl_matrix_column (A, 0);
      double norm = gdl_blas_dnrm2 (&column.vector);

      gdl_vector_set (S, 0, norm); 
      gdl_matrix_set (V, 0, 0, 1.0);
      
      if (norm != 0.0)
        {
          gdl_blas_dscal (1.0/norm, &column.vector);
        }

      return GDL_SUCCESS;
    }
	
	{
		
	sis2_ritzit( N,
			sis2->p,
			sis2->km,
			sis2->eps,
			&sis2_opb,
			sis2->em,
			sis2->x,
			sis2->d,
			sis2->f,
			sis2->cx,
			sis2->u,
			&(sis2->imem),
			A,
			M,
			N,
			sis2->t
		  );
	
	for (j=0;j<sis2->p;j++) {
      sis2_opb (A, M, N, &(sis2->x[j][0]), sis2->cx, sis2->t);
      tmp1=sis2_ddot(N, &(sis2->x[j][0]), 1, sis2->cx, 1);
      sis2_daxpy(N, -tmp1, &(sis2->x[j][0]), 1, sis2->cx, 1);
      tmp1=sqrt(tmp1);
      xnorm=sqrt(sis2_ddot(N, sis2->cx, 1, sis2->cx, 1));
      /* multiply by matrix A to get (scaled) left S-vector */
      sis2_opa(A, M, N, &(sis2->x[j][0]), sis2->u, sis2->t);
      tmp2=1.0/tmp1;
      sis2_dscal(M, tmp2, sis2->u, 1);
      xnorm=xnorm*tmp2;
      sis2->f[j]=xnorm;
      sis2->d[j]=tmp1;
      for(i=0;i<N;i++)
      	gdl_matrix_set (V, i, j, sis2->x[j][i]);
      for(i=0;i<M;i++)
	    gdl_matrix_set (A, i, j, sis2->u[i]);
      gdl_vector_set (S, j, sis2->d[j]);
    }
    
    return GDL_SUCCESS;
    
	}
}

static int
_alloc (void * vstate, size_t M, size_t N)
{
	if (vstate)
	{
		sis2_t * sis2 = (sis2_t *) vstate;
		sis2->M  = M;
		sis2->N  = N;
		sis2->x  = GDL_MATRIX_ALLOC (double, N, M);
		sis2->d  = GDL_CALLOC (double, GDL_MIN (N, M));
		sis2->f  = GDL_CALLOC (double, GDL_MIN (N, M));
		sis2->cx = GDL_CALLOC (double, M);
		sis2->u  = GDL_CALLOC (double, M);
		sis2->t  = 0;
	}
	return GDL_EINVAL;
}

static int 
_free (void * vstate)
{
	if (vstate)
	{
		sis2_t * sis2 = (sis2_t *) vstate;
		GDL_MATRIX_FREE (sis2->x, sis2->N);
		GDL_FREE (sis2->d);
		GDL_FREE (sis2->f);
		GDL_FREE (sis2->cx);
		GDL_FREE (sis2->u);
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

static int
_perform (void * vstate, gdl_matrix * A, gdl_matrix * V, gdl_vector * S, size_t ns)
{
	if (vstate)
	{
		sis2_t * sis2 = (sis2_t *) vstate;
		sis2->em       = sis2->p = ns;
		sis2->numextra = 0;
		sis2->eps      = 1.e-20;
		sis2->km       = 1000;
		return sis2_decomp (sis2, A, V, S);
	}
	return GDL_EINVAL;
}

static const gdl_linalg_svd_workspace_type _sis2 =
{
	"gdl_linalg_svd_sis2",
	sizeof (sis2_t),
	&_alloc,
	&_free,
	&_perform
};

const gdl_linalg_svd_workspace_type * gdl_linalg_svd_sis2 = &_sis2;
