/* linalg/test.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2004 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Author:  G. Jungman
 */
#include <stdlib.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_test.h>
#include <gdl/gdl_math.h>
//#include <gdl/gdl_permute_vector.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_linalg.h>

int check (double x, double actual, double eps);
gdl_matrix * create_hilbert_matrix(size_t size);
gdl_matrix * create_general_matrix(size_t size1, size_t size2);
gdl_matrix * create_vandermonde_matrix(size_t size);
gdl_matrix * create_moler_matrix(size_t size);
gdl_matrix * create_row_matrix(size_t size1, size_t size2);
gdl_matrix * create_2x2_matrix(double a11, double a12, double a21, double a22);
gdl_matrix * create_diagonal_matrix(double a[], size_t size);

int test_matmult(void);
int test_matmult_mod(void);
int test_LU_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_LU_solve(void);
int test_QR_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_QR_solve(void);
int test_QR_QRsolve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_QR_QRsolve(void);
int test_QR_lssolve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_QR_lssolve(void);
int test_QR_decomp_dim(const gdl_matrix * m, double eps);
int test_QR_decomp(void);
int test_QRPT_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_QRPT_solve(void);
int test_QRPT_QRsolve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_QRPT_QRsolve(void);
int test_QRPT_decomp_dim(const gdl_matrix * m, double eps);
int test_QRPT_decomp(void);
int test_QR_update_dim(const gdl_matrix * m, double eps);
int test_QR_update(void);

int test_LQ_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_LQ_solve(void);
int test_LQ_LQsolve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_LQ_LQsolve(void);
int test_LQ_lssolve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_LQ_lssolve(void);
int test_LQ_decomp_dim(const gdl_matrix * m, double eps);
int test_LQ_decomp(void);
int test_PTLQ_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_PTLQ_solve(void);
int test_PTLQ_LQsolve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_PTLQ_LQsolve(void);
int test_PTLQ_decomp_dim(const gdl_matrix * m, double eps);
int test_PTLQ_decomp(void);
int test_LQ_update_dim(const gdl_matrix * m, double eps);
int test_LQ_update(void);

int test_SV_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_SV_solve(void);
int test_SV_decomp_dim(const gdl_matrix * m, double eps);
int test_SV_decomp(void);
int test_SV_decomp_mod_dim(const gdl_matrix * m, double eps);
int test_SV_decomp_mod(void);
int test_cholesky_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_cholesky_solve(void);
int test_cholesky_decomp_dim(const gdl_matrix * m, double eps);
int test_cholesky_decomp(void);
int test_HH_solve_dim(const gdl_matrix * m, const double * actual, double eps);
int test_HH_solve(void);
int test_TDS_solve_dim(size_t dim, double d, double od, const double * actual, double eps);
int test_TDS_solve(void);
int test_TDN_solve_dim(size_t dim, double d, double a, double b, const double * actual, double eps);
int test_TDN_solve(void);
int test_TDS_cyc_solve_one(const size_t dim, const double * d, const double * od, const double * r,
                          const double * actual, double eps);
int test_TDS_cyc_solve(void);
int test_TDN_cyc_solve_dim(size_t dim, double d, double a, double b, const double * actual, double eps);
int test_TDN_cyc_solve(void);
int test_bidiag_decomp_dim(const gdl_matrix * m, double eps);
int test_bidiag_decomp(void);

int 
check (double x, double actual, double eps)
{
  if (x == actual)
    {
      return 0;
    }
  else if (actual == 0)
    {
      return fabs(x) > eps;
    }
  else
    {
      return (fabs(x - actual)/fabs(actual)) > eps;
    }
}

gdl_matrix *
create_hilbert_matrix(size_t size)
{
  size_t i, j;
  gdl_matrix * m = gdl_matrix_alloc(size, size);
  for(i=0; i<size; i++) {
    for(j=0; j<size; j++) {
      gdl_matrix_set(m, i, j, 1.0/(i+j+1.0));
    }
  }
  return m;
}

gdl_matrix *
create_general_matrix(size_t size1, size_t size2)
{
  size_t i, j;
  gdl_matrix * m = gdl_matrix_alloc(size1, size2);
  for(i=0; i<size1; i++) {
    for(j=0; j<size2; j++) {
      gdl_matrix_set(m, i, j, 1.0/(i+j+1.0));
    }
  }
  return m;
}

gdl_matrix *
create_singular_matrix(size_t size1, size_t size2)
{
  size_t i, j;
  gdl_matrix * m = gdl_matrix_alloc(size1, size2);
  for(i=0; i<size1; i++) {
    for(j=0; j<size2; j++) {
      gdl_matrix_set(m, i, j, 1.0/(i+j+1.0));
    }
  }

  /* zero the first column */
  for(j = 0; j <size2; j++)
    gdl_matrix_set(m,0,j,0.0);

  return m;
}


gdl_matrix *
create_vandermonde_matrix(size_t size)
{
  size_t i, j;
  gdl_matrix * m = gdl_matrix_alloc(size, size);
  for(i=0; i<size; i++) {
    for(j=0; j<size; j++) {
      gdl_matrix_set(m, i, j, pow(i + 1.0, size - j - 1.0));
    }
  }
  return m;
}

gdl_matrix *
create_moler_matrix(size_t size)
{
  size_t i, j;
  gdl_matrix * m = gdl_matrix_alloc(size, size);
  for(i=0; i<size; i++) {
    for(j=0; j<size; j++) {
      gdl_matrix_set(m, i, j, GDL_MIN(i+1,j+1)-2.0);
    }
  }
  return m;
}

gdl_matrix *
create_row_matrix(size_t size1, size_t size2)
{
  size_t i;
  gdl_matrix * m = gdl_matrix_calloc(size1, size2);
  for(i=0; i<size1; i++) {
      gdl_matrix_set(m, i, 0, 1.0/(i+1.0));
  }

  return m;
}

gdl_matrix *
create_2x2_matrix(double a11, double a12, double a21, double a22)
{
  gdl_matrix * m = gdl_matrix_alloc(2, 2);
  gdl_matrix_set(m, 0, 0, a11);
  gdl_matrix_set(m, 0, 1, a12);
  gdl_matrix_set(m, 1, 0, a21);
  gdl_matrix_set(m, 1, 1, a22);
  return m;
}

gdl_matrix *
create_diagonal_matrix(double a[], size_t size)
{
  size_t i;
  gdl_matrix * m = gdl_matrix_calloc(size, size);
  for(i=0; i<size; i++) {
      gdl_matrix_set(m, i, i, a[i]);
  }

  return m;
}

gdl_matrix * m11;
gdl_matrix * m51;

gdl_matrix * m35;
gdl_matrix * m53;
gdl_matrix * m97;

gdl_matrix * s35;
gdl_matrix * s53;

gdl_matrix * hilb2;
gdl_matrix * hilb3;
gdl_matrix * hilb4;
gdl_matrix * hilb12;

gdl_matrix * row3;
gdl_matrix * row5;
gdl_matrix * row12;

gdl_matrix * A22;
gdl_matrix * A33;
gdl_matrix * A44;

gdl_matrix * inf5; double inf5_data[] = {1.0, 0.0, -3.0, 0.0, -5.0};
gdl_matrix * nan5;

double m53_lssolution[] = {52.5992295702070, -337.7263113752073, 
                           351.8823436427604};
double hilb2_solution[] = {-8.0, 18.0} ;
double hilb3_solution[] = {27.0, -192.0, 210.0};
double hilb4_solution[] = {-64.0, 900.0, -2520.0, 1820.0};
double hilb12_solution[] = {-1728.0, 245388.0, -8528520.0, 
                            127026900.0, -1009008000.0, 4768571808.0, 
                            -14202796608.0, 27336497760.0, -33921201600.0,
                            26189163000.0, -11437874448.0, 2157916488.0 };

double c7_solution[] = { 2.40717272023734e+01, -9.84612797621247e+00,
                         -2.69338853034031e+02, 8.75455232472528e+01,
                         2.96661356736296e+03, -1.02624473923993e+03,
                         -1.82073812124749e+04, 5.67384473042410e+03,
                         5.57693879019068e+04, -1.61540963210502e+04,
                         -7.88941207561151e+04, 1.95053812987858e+04,
                         3.95548551241728e+04, -7.76593696255317e+03 };

gdl_matrix * vander2;
gdl_matrix * vander3;
gdl_matrix * vander4;
gdl_matrix * vander12;

double vander2_solution[] = {1.0, 0.0}; 
double vander3_solution[] = {0.0, 1.0, 0.0}; 
double vander4_solution[] = {0.0, 0.0, 1.0, 0.0}; 
double vander12_solution[] = {0.0, 0.0, 0.0, 0.0,
                            0.0, 0.0, 0.0, 0.0, 
                            0.0, 0.0, 1.0, 0.0}; 

gdl_matrix * moler10;

//
///* matmult now obsolete */
//#ifdef MATMULT
//int
//test_matmult(void)
//{
//  int s = 0;
//
//  gdl_matrix * A = gdl_matrix_calloc(2, 2);
//  gdl_matrix * B = gdl_matrix_calloc(2, 3);
//  gdl_matrix * C = gdl_matrix_calloc(2, 3);
//
//  gdl_matrix_set(A, 0, 0, 10.0);
//  gdl_matrix_set(A, 0, 1,  5.0);
//  gdl_matrix_set(A, 1, 0,  1.0);
//  gdl_matrix_set(A, 1, 1, 20.0);
//
//  gdl_matrix_set(B, 0, 0, 10.0);
//  gdl_matrix_set(B, 0, 1,  5.0);
//  gdl_matrix_set(B, 0, 2,  2.0);
//  gdl_matrix_set(B, 1, 0,  1.0);
//  gdl_matrix_set(B, 1, 1,  3.0);
//  gdl_matrix_set(B, 1, 2,  2.0);
//
//  gdl_linalg_matmult(A, B, C);
//
//  s += ( fabs(gdl_matrix_get(C, 0, 0) - 105.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 1) -  65.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 2) -  30.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 0) -  30.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 1) -  65.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 2) -  42.0) > GDL_DBL_EPSILON );
//
//  gdl_matrix_free(A);
//  gdl_matrix_free(B);
//  gdl_matrix_free(C);
//
//  return s;
//}
//
//
//int
//test_matmult_mod(void)
//{
//  int s = 0;
//
//  gdl_matrix * A = gdl_matrix_calloc(3, 3);
//  gdl_matrix * B = gdl_matrix_calloc(3, 3);
//  gdl_matrix * C = gdl_matrix_calloc(3, 3);
//  gdl_matrix * D = gdl_matrix_calloc(2, 3);
//  gdl_matrix * E = gdl_matrix_calloc(2, 3);
//  gdl_matrix * F = gdl_matrix_calloc(2, 2);
//
//  gdl_matrix_set(A, 0, 0, 10.0);
//  gdl_matrix_set(A, 0, 1,  5.0);
//  gdl_matrix_set(A, 0, 2,  1.0);
//  gdl_matrix_set(A, 1, 0,  1.0);
//  gdl_matrix_set(A, 1, 1, 20.0);
//  gdl_matrix_set(A, 1, 2,  5.0);
//  gdl_matrix_set(A, 2, 0,  1.0);
//  gdl_matrix_set(A, 2, 1,  3.0);
//  gdl_matrix_set(A, 2, 2,  7.0);
//
//  gdl_matrix_set(B, 0, 0, 10.0);
//  gdl_matrix_set(B, 0, 1,  5.0);
//  gdl_matrix_set(B, 0, 2,  2.0);
//  gdl_matrix_set(B, 1, 0,  1.0);
//  gdl_matrix_set(B, 1, 1,  3.0);
//  gdl_matrix_set(B, 1, 2,  2.0);
//  gdl_matrix_set(B, 2, 0,  1.0);
//  gdl_matrix_set(B, 2, 1,  3.0);
//  gdl_matrix_set(B, 2, 2,  2.0);
//
//  gdl_matrix_set(D, 0, 0, 10.0);
//  gdl_matrix_set(D, 0, 1,  5.0);
//  gdl_matrix_set(D, 0, 2,  1.0);
//  gdl_matrix_set(D, 1, 0,  1.0);
//  gdl_matrix_set(D, 1, 1, 20.0);
//  gdl_matrix_set(D, 1, 2,  5.0);
//
//  gdl_matrix_set(E, 0, 0, 10.0);
//  gdl_matrix_set(E, 0, 1,  5.0);
//  gdl_matrix_set(E, 0, 2,  2.0);
//  gdl_matrix_set(E, 1, 0,  1.0);
//  gdl_matrix_set(E, 1, 1,  3.0);
//  gdl_matrix_set(E, 1, 2,  2.0);
//
//  gdl_linalg_matmult_mod(A, GDL_LINALG_MOD_NONE, B, GDL_LINALG_MOD_NONE, C);
//  s += ( fabs(gdl_matrix_get(C, 0, 0) - 106.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 1) -  68.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 2) -  32.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 0) -  35.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 1) -  80.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 2) -  52.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 0) -  20.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 1) -  35.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 2) -  22.0) > GDL_DBL_EPSILON );
//
//  gdl_linalg_matmult_mod(A, GDL_LINALG_MOD_TRANSPOSE, B, GDL_LINALG_MOD_NONE, C);
//  s += ( fabs(gdl_matrix_get(C, 0, 0) - 102.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 1) -  56.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 2) -  24.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 0) -  73.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 1) -  94.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 2) -  56.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 0) -  22.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 1) -  41.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 2) -  26.0) > GDL_DBL_EPSILON );
//
//  gdl_linalg_matmult_mod(A, GDL_LINALG_MOD_NONE, B, GDL_LINALG_MOD_TRANSPOSE, C);
//  s += ( fabs(gdl_matrix_get(C, 0, 0) - 127.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 1) -  27.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 2) -  27.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 0) - 120.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 1) -  71.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 2) -  71.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 0) -  39.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 1) -  24.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 2) -  24.0) > GDL_DBL_EPSILON );
//
//  gdl_linalg_matmult_mod(A, GDL_LINALG_MOD_TRANSPOSE, B, GDL_LINALG_MOD_TRANSPOSE, C);
//  s += ( fabs(gdl_matrix_get(C, 0, 0) - 107.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 1) -  15.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 2) -  15.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 0) - 156.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 1) -  71.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 2) -  71.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 0) -  49.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 1) -  30.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 2) -  30.0) > GDL_DBL_EPSILON );
//
//  /* now try for non-symmetric matrices */
//  gdl_linalg_matmult_mod(D, GDL_LINALG_MOD_TRANSPOSE, E, GDL_LINALG_MOD_NONE, C);
//  s += ( fabs(gdl_matrix_get(C, 0, 0) - 101.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 1) -  53.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 0, 2) -  22.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 0) -  70.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 1) -  85.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 1, 2) -  50.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 0) -  15.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 1) -  20.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(C, 2, 2) -  12.0) > GDL_DBL_EPSILON );
//
//
//  gdl_linalg_matmult_mod(D, GDL_LINALG_MOD_NONE, E, GDL_LINALG_MOD_TRANSPOSE, F);
//  s += ( fabs(gdl_matrix_get(F, 0, 0) - 127.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(F, 0, 1) -  27.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(F, 1, 0) - 120.0) > GDL_DBL_EPSILON );
//  s += ( fabs(gdl_matrix_get(F, 1, 1) -  71.0) > GDL_DBL_EPSILON );
//
//
//  gdl_matrix_free(A);
//  gdl_matrix_free(B);
//  gdl_matrix_free(C);
//  gdl_matrix_free(D);
//  gdl_matrix_free(E);
//  gdl_matrix_free(F);
//
//  return s;
//}
//#endif
//
//int
//test_LU_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  int signum;
//  size_t i, dim = m->size1;
//
//  gdl_permutation * perm = gdl_permutation_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * lu  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//  gdl_vector * residual = gdl_vector_alloc(dim);
//  gdl_matrix_memcpy(lu,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_LU_decomp(lu, perm, &signum);
//  s += gdl_linalg_LU_solve(lu, perm, rhs, x);
//
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i),actual[i],eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  s += gdl_linalg_LU_refine(m, lu, perm, rhs, x, residual);
//
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i),actual[i],eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g (improved)\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(residual);
//  gdl_vector_free(x);
//  gdl_matrix_free(lu);
//  gdl_vector_free(rhs);
//  gdl_permutation_free(perm);
//
//  return s;
//}
//
//
//int test_LU_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_LU_solve_dim(hilb2, hilb2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LU_solve hilbert(2)");
//  s += f;
//
//  f = test_LU_solve_dim(hilb3, hilb3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LU_solve hilbert(3)");
//  s += f;
//
//  f = test_LU_solve_dim(hilb4, hilb4_solution, 2048.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LU_solve hilbert(4)");
//  s += f;
//
//  f = test_LU_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  LU_solve hilbert(12)");
//  s += f;
//
//  f = test_LU_solve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LU_solve vander(2)");
//  s += f;
//
//  f = test_LU_solve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LU_solve vander(3)");
//  s += f;
//
//  f = test_LU_solve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LU_solve vander(4)");
//  s += f;
//
//  f = test_LU_solve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  LU_solve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_LUc_solve_dim(const gdl_matrix_complex * m, const double * actual, double eps)
//{
//  int s = 0;
//  int signum;
//  size_t i, dim = m->size1;
//
//  gdl_permutation * perm = gdl_permutation_alloc(dim);
//  gdl_vector_complex * rhs = gdl_vector_complex_alloc(dim);
//  gdl_matrix_complex * lu  = gdl_matrix_complex_alloc(dim,dim);
//  gdl_vector_complex * x = gdl_vector_complex_alloc(dim);
//  gdl_vector_complex * residual = gdl_vector_complex_alloc(dim);
//  gdl_matrix_complex_memcpy(lu,m);
//  for(i=0; i<dim; i++) 
//    {
//      gdl_complex z = gdl_complex_rect (2.0*i+1.0, 2.0*i+2.0);
//      gdl_vector_complex_set(rhs, i, z);
//    }
//  s += gdl_linalg_complex_LU_decomp(lu, perm, &signum);
//  s += gdl_linalg_complex_LU_solve(lu, perm, rhs, x);
//
//  for(i=0; i<dim; i++) {
//    gdl_complex z = gdl_vector_complex_get(x, i);
//    int foo_r = check(GDL_REAL(z),actual[2*i],eps);
//    int foo_i = check(GDL_IMAG(z),actual[2*i+1],eps);
//    if(foo_r || foo_i) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, GDL_REAL(z), actual[2*i]);
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, GDL_IMAG(z), actual[2*i+1]);
//    }
//    s += foo_r + foo_i;
//  }
//
//  s += gdl_linalg_complex_LU_refine(m, lu, perm, rhs, x, residual);
//
//  for(i=0; i<dim; i++) {
//    gdl_complex z = gdl_vector_complex_get(x, i);
//    int foo_r = check(GDL_REAL(z),actual[2*i],eps);
//    int foo_i = check(GDL_IMAG(z),actual[2*i+1],eps);
//    if(foo_r || foo_i) {
//      printf("%3d[%d]: %22.18g   %22.18g (improved)\n", dim, i, GDL_REAL(z), actual[2*i]);
//      printf("%3d[%d]: %22.18g   %22.18g (improved)\n", dim, i, GDL_IMAG(z), actual[2*i+1]);
//    }
//    s += foo_r + foo_i;
//  }
//
//  gdl_vector_complex_free(residual);
//  gdl_vector_complex_free(x);
//  gdl_matrix_complex_free(lu);
//  gdl_vector_complex_free(rhs);
//  gdl_permutation_free(perm);
//
//  return s;
//}
//
//
//int test_LUc_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_LUc_solve_dim(c7, c7_solution, 1024.0 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  complex_LU_solve complex(7)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_QR_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, dim = m->size1;
//
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * qr  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  gdl_matrix_memcpy(qr,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_QR_decomp(qr, d);
//  s += gdl_linalg_QR_solve(qr, d, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(qr);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_QR_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QR_solve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_solve hilbert(2)");
//  s += f;
//
//  f = test_QR_solve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_solve hilbert(3)");
//  s += f;
//
//  f = test_QR_solve_dim(hilb4, hilb4_solution, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_solve hilbert(4)");
//  s += f;
//
//  f = test_QR_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  QR_solve hilbert(12)");
//  s += f;
//
//  f = test_QR_solve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_solve vander(2)");
//  s += f;
//
//  f = test_QR_solve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_solve vander(3)");
//  s += f;
//
//  f = test_QR_solve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_solve vander(4)");
//  s += f;
//
//  f = test_QR_solve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  QR_solve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_QR_QRsolve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, dim = m->size1;
//
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * qr  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * q  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * r  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  gdl_matrix_memcpy(qr,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_QR_decomp(qr, d);
//  s += gdl_linalg_QR_unpack(qr, d, q, r);
//  s += gdl_linalg_QR_QRsolve(q, r, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(qr);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_QR_QRsolve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QR_QRsolve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_QRsolve hilbert(2)");
//  s += f;
//
//  f = test_QR_QRsolve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_QRsolve hilbert(3)");
//  s += f;
//
//  f = test_QR_QRsolve_dim(hilb4, hilb4_solution, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_QRsolve hilbert(4)");
//  s += f;
//
//  f = test_QR_QRsolve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  QR_QRsolve hilbert(12)");
//  s += f;
//
//  f = test_QR_QRsolve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_QRsolve vander(2)");
//  s += f;
//
//  f = test_QR_QRsolve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_QRsolve vander(3)");
//  s += f;
//
//  f = test_QR_QRsolve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_QRsolve vander(4)");
//  s += f;
//
//  f = test_QR_QRsolve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  QR_QRsolve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_QR_lssolve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, M = m->size1, N = m->size2;
//
//  gdl_vector * rhs = gdl_vector_alloc(M);
//  gdl_matrix * qr  = gdl_matrix_alloc(M,N);
//  gdl_vector * d = gdl_vector_alloc(N);
//  gdl_vector * x = gdl_vector_alloc(N);
//  gdl_vector * r = gdl_vector_alloc(M);
//  gdl_vector * res = gdl_vector_alloc(M);
//
//  gdl_matrix_memcpy(qr,m);
//  for(i=0; i<M; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_QR_decomp(qr, d);
//  s += gdl_linalg_QR_lssolve(qr, d, rhs, x, res);
//
//  for(i=0; i<N; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("(%3d,%3d)[%d]: %22.18g   %22.18g\n", M, N, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  /* compute residual r = b - m x */
//  if (M == N) {
//    gdl_vector_set_zero(r);
//  } else {
//    gdl_vector_memcpy(r, rhs);
//    gdl_blas_dgemv(CblasNoTrans, -1.0, m, x, 1.0, r);
//  };
//
//  for(i=0; i<N; i++) {
//    int foo = check(gdl_vector_get(res, i), gdl_vector_get(r,i), sqrt(eps));
//    if(foo) {
//      printf("(%3d,%3d)[%d]: %22.18g   %22.18g\n", M, N, i, gdl_vector_get(res, i), gdl_vector_get(r,i));
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(r);
//  gdl_vector_free(res);
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(qr);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_QR_lssolve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QR_lssolve_dim(m53, m53_lssolution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_lssolve m(5,3)");
//  s += f;
//
//  f = test_QR_lssolve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_lssolve hilbert(2)");
//  s += f;
//
//  f = test_QR_lssolve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_lssolve hilbert(3)");
//  s += f;
//
//  f = test_QR_lssolve_dim(hilb4, hilb4_solution, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_lssolve hilbert(4)");
//  s += f;
//
//  f = test_QR_lssolve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  QR_lssolve hilbert(12)");
//  s += f;
//
//  f = test_QR_lssolve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_lssolve vander(2)");
//  s += f;
//
//  f = test_QR_lssolve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_lssolve vander(3)");
//  s += f;
//
//  f = test_QR_lssolve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_lssolve vander(4)");
//  s += f;
//
//  f = test_QR_lssolve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  QR_lssolve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_QR_decomp_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0;
//  size_t i,j, M = m->size1, N = m->size2;
//
//  gdl_matrix * qr = gdl_matrix_alloc(M,N);
//  gdl_matrix * a  = gdl_matrix_alloc(M,N);
//  gdl_matrix * q  = gdl_matrix_alloc(M,M);
//  gdl_matrix * r  = gdl_matrix_alloc(M,N);
//  gdl_vector * d = gdl_vector_alloc(GDL_MIN(M,N));
//
//  gdl_matrix_memcpy(qr,m);
//
//  s += gdl_linalg_QR_decomp(qr, d);
//  s += gdl_linalg_QR_unpack(qr, d, q, r);
//  
//  /* compute a = q r */
//  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, q, r, 0.0, a);
//
//  for(i=0; i<M; i++) {
//    for(j=0; j<N; j++) {
//      double aij = gdl_matrix_get(a, i, j);
//      double mij = gdl_matrix_get(m, i, j);
//      int foo = check(aij, mij, eps);
//      if(foo) {
//        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_vector_free(d);
//  gdl_matrix_free(qr);
//  gdl_matrix_free(a);
//  gdl_matrix_free(q);
//  gdl_matrix_free(r);
//
//  return s;
//}
//
//int test_QR_decomp(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QR_decomp_dim(m35, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp m(3,5)");
//  s += f;
//
//  f = test_QR_decomp_dim(m53, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp m(5,3)");
//  s += f;
//
//  f = test_QR_decomp_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp hilbert(2)");
//  s += f;
//
//  f = test_QR_decomp_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp hilbert(3)");
//  s += f;
//
//  f = test_QR_decomp_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp hilbert(4)");
//  s += f;
//
//  f = test_QR_decomp_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp hilbert(12)");
//  s += f;
//
//  f = test_QR_decomp_dim(vander2, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp vander(2)");
//  s += f;
//
//  f = test_QR_decomp_dim(vander3, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp vander(3)");
//  s += f;
//
//  f = test_QR_decomp_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_decomp vander(4)");
//  s += f;
//
//  f = test_QR_decomp_dim(vander12, 0.0005); /* FIXME: bad accuracy */
//  gdl_test(f, "  QR_decomp vander(12)");
//  s += f;
//
//  return s;
//}
//
//int
//test_QRPT_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  int signum;
//  size_t i, dim = m->size1;
//
//  gdl_permutation * perm = gdl_permutation_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * qr  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//  gdl_vector * norm = gdl_vector_alloc(dim);
//
//  gdl_matrix_memcpy(qr,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_QRPT_decomp(qr, d, perm, &signum, norm);
//  s += gdl_linalg_QRPT_solve(qr, d, perm, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(norm);
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(qr);
//  gdl_vector_free(rhs);
//  gdl_permutation_free(perm);
//
//  return s;
//}
//
//int test_QRPT_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QRPT_solve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_solve hilbert(2)");
//  s += f;
//
//  f = test_QRPT_solve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_solve hilbert(3)");
//  s += f;
//
//  f = test_QRPT_solve_dim(hilb4, hilb4_solution, 2 * 2048.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_solve hilbert(4)");
//  s += f;
//
//  f = test_QRPT_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  QRPT_solve hilbert(12)");
//  s += f;
//
//  f = test_QRPT_solve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_solve vander(2)");
//  s += f;
//
//  f = test_QRPT_solve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_solve vander(3)");
//  s += f;
//
//  f = test_QRPT_solve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_solve vander(4)");
//  s += f;
//
//  f = test_QRPT_solve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  QRPT_solve vander(12)");
//  s += f;
//
//  return s;
//}
//
//int
//test_QRPT_QRsolve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  int signum;
//  size_t i, dim = m->size1;
//
//  gdl_permutation * perm = gdl_permutation_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * qr  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * q  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * r  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//  gdl_vector * norm = gdl_vector_alloc(dim);
//
//  gdl_matrix_memcpy(qr,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_QRPT_decomp2(qr, q, r, d, perm, &signum, norm);
//  s += gdl_linalg_QRPT_QRsolve(q, r, perm, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(norm);
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(qr);
//  gdl_vector_free(rhs);
//  gdl_permutation_free(perm);
//
//  return s;
//}
//
//int test_QRPT_QRsolve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QRPT_QRsolve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_QRsolve hilbert(2)");
//  s += f;
//
//  f = test_QRPT_QRsolve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_QRsolve hilbert(3)");
//  s += f;
//
//  f = test_QRPT_QRsolve_dim(hilb4, hilb4_solution, 2 * 2048.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_QRsolve hilbert(4)");
//  s += f;
//
//  f = test_QRPT_QRsolve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  QRPT_QRsolve hilbert(12)");
//  s += f;
//
//  f = test_QRPT_QRsolve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_QRsolve vander(2)");
//  s += f;
//
//  f = test_QRPT_QRsolve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_QRsolve vander(3)");
//  s += f;
//
//  f = test_QRPT_QRsolve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_QRsolve vander(4)");
//  s += f;
//
//  f = test_QRPT_QRsolve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  QRPT_QRsolve vander(12)");
//  s += f;
//
//  return s;
//}
//
//int
//test_QRPT_decomp_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0, signum;
//  size_t i,j, M = m->size1, N = m->size2;
//
//  gdl_matrix * qr = gdl_matrix_alloc(M,N);
//  gdl_matrix * a  = gdl_matrix_alloc(M,N);
//  gdl_matrix * q  = gdl_matrix_alloc(M,M);
//  gdl_matrix * r  = gdl_matrix_alloc(M,N);
//  gdl_vector * d = gdl_vector_alloc(GDL_MIN(M,N));
//  gdl_vector * norm = gdl_vector_alloc(N);
//
//  gdl_permutation * perm = gdl_permutation_alloc(N);
//
//  gdl_matrix_memcpy(qr,m);
//
//  s += gdl_linalg_QRPT_decomp(qr, d, perm, &signum, norm);
//  s += gdl_linalg_QR_unpack(qr, d, q, r);
//
//  /* compute a = q r */
//  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, q, r, 0.0, a);
//
//
//  /* Compute QR P^T by permuting the elements of the rows of QR */
//
//  for (i = 0; i < M; i++) {
//    gdl_vector_view row = gdl_matrix_row (a, i);
//    gdl_permute_vector_inverse (perm, &row.vector);
//  }
//
//  for(i=0; i<M; i++) {
//    for(j=0; j<N; j++) {
//      double aij = gdl_matrix_get(a, i, j);
//      double mij = gdl_matrix_get(m, i, j);
//      int foo = check(aij, mij, eps);
//      if(foo) {
//        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_permutation_free (perm);
//  gdl_vector_free(norm);
//  gdl_vector_free(d);
//  gdl_matrix_free(qr);
//  gdl_matrix_free(a);
//  gdl_matrix_free(q);
//  gdl_matrix_free(r);
//
//  return s;
//}
//
//int test_QRPT_decomp(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QRPT_decomp_dim(m35, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp m(3,5)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(m53, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp m(5,3)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(s35, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp s(3,5)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(s53, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp s(5,3)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp hilbert(2)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp hilbert(3)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp hilbert(4)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp hilbert(12)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(vander2, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp vander(2)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(vander3, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp vander(3)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QRPT_decomp vander(4)");
//  s += f;
//
//  f = test_QRPT_decomp_dim(vander12, 0.0005); /* FIXME: bad accuracy */
//  gdl_test(f, "  QRPT_decomp vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_QR_update_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0;
//  size_t i,j,k, M = m->size1, N = m->size2;
//
//  gdl_vector * rhs = gdl_vector_alloc(N);
//  gdl_matrix * qr1  = gdl_matrix_alloc(M,N);
//  gdl_matrix * qr2  = gdl_matrix_alloc(M,N);
//  gdl_matrix * q1  = gdl_matrix_alloc(M,M);
//  gdl_matrix * r1  = gdl_matrix_alloc(M,N);
//  gdl_matrix * q2  = gdl_matrix_alloc(M,M);
//  gdl_matrix * r2  = gdl_matrix_alloc(M,N);
//  gdl_vector * d = gdl_vector_alloc(GDL_MIN(M,N));
//  gdl_vector * solution1 = gdl_vector_alloc(N);
//  gdl_vector * solution2 = gdl_vector_alloc(N);
//  gdl_vector * u = gdl_vector_alloc(M);
//  gdl_vector * v = gdl_vector_alloc(N);
//  gdl_vector * w = gdl_vector_alloc(M);
//
//  gdl_matrix_memcpy(qr1,m);
//  gdl_matrix_memcpy(qr2,m);
//  for(i=0; i<N; i++) gdl_vector_set(rhs, i, i+1.0);
//  for(i=0; i<M; i++) gdl_vector_set(u, i, sin(i+1.0));
//  for(i=0; i<N; i++) gdl_vector_set(v, i, cos(i+2.0) + sin(i*i+3.0));
//
//  for(i=0; i<M; i++) 
//    {
//      double ui = gdl_vector_get(u, i);
//      for(j=0; j<N; j++) 
//        {
//          double vj = gdl_vector_get(v, j);
//          double qij = gdl_matrix_get(qr1, i, j);
//          gdl_matrix_set(qr1, i, j, qij + ui * vj);
//        }
//    }
//
//  s += gdl_linalg_QR_decomp(qr2, d);
//  s += gdl_linalg_QR_unpack(qr2, d, q2, r2);
//
//  /* compute w = Q^T u */
//      
//  for (j = 0; j < M; j++)
//    {
//      double sum = 0;
//      for (i = 0; i < M; i++)
//          sum += gdl_matrix_get (q2, i, j) * gdl_vector_get (u, i);
//      gdl_vector_set (w, j, sum);
//    }
//
//  s += gdl_linalg_QR_update(q2, r2, w, v);
//
//  /* compute qr2 = q2 * r2 */
//
//  for (i = 0; i < M; i++)
//    {
//      for (j = 0; j< N; j++)
//        {
//          double sum = 0;
//          for (k = 0; k <= GDL_MIN(j,M-1); k++)
//            {
//              double qik = gdl_matrix_get(q2, i, k);
//              double rkj = gdl_matrix_get(r2, k, j);
//              sum += qik * rkj ;
//            }
//          gdl_matrix_set (qr2, i, j, sum);
//        }
//    }
//
//  for(i=0; i<M; i++) {
//    for(j=0; j<N; j++) {
//      double s1 = gdl_matrix_get(qr1, i, j);
//      double s2 = gdl_matrix_get(qr2, i, j);
//      
//      int foo = check(s1, s2, eps);
//      if(foo) {
//        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, s1, s2);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_vector_free(solution1);
//  gdl_vector_free(solution2);
//  gdl_vector_free(d);
//  gdl_vector_free(u);
//  gdl_vector_free(v);
//  gdl_vector_free(w);
//  gdl_matrix_free(qr1);
//  gdl_matrix_free(qr2);
//  gdl_matrix_free(q1);
//  gdl_matrix_free(r1);
//  gdl_matrix_free(q2);
//  gdl_matrix_free(r2);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_QR_update(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_QR_update_dim(m35, 2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update m(3,5)");
//  s += f;
//
//  f = test_QR_update_dim(m53, 2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update m(5,3)");
//  s += f;
//
//  f = test_QR_update_dim(hilb2,  2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update hilbert(2)");
//  s += f;
//
//  f = test_QR_update_dim(hilb3,  2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update hilbert(3)");
//  s += f;
//
//  f = test_QR_update_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update hilbert(4)");
//  s += f;
//
//  f = test_QR_update_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update hilbert(12)");
//  s += f;
//
//  f = test_QR_update_dim(vander2, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update vander(2)");
//  s += f;
//
//  f = test_QR_update_dim(vander3, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update vander(3)");
//  s += f;
//
//  f = test_QR_update_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  QR_update vander(4)");
//  s += f;
//
//  f = test_QR_update_dim(vander12, 0.0005); /* FIXME: bad accuracy */
//  gdl_test(f, "  QR_update vander(12)");
//  s += f;
//
//  return s;
//}
//
//int
//test_LQ_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, dim = m->size1;
//
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * lq  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  gdl_matrix_transpose_memcpy(lq,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_LQ_decomp(lq, d);
//  s += gdl_linalg_LQ_solve_T(lq, d, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(lq);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_LQ_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_LQ_solve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_solve hilbert(2)");
//  s += f;
//
//  f = test_LQ_solve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_solve hilbert(3)");
//  s += f;
//
//  f = test_LQ_solve_dim(hilb4, hilb4_solution, 4 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_solve hilbert(4)");
//  s += f;
//
//  f = test_LQ_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  LQ_solve hilbert(12)");
//  s += f;
//
//  f = test_LQ_solve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_solve vander(2)");
//  s += f;
//
//  f = test_LQ_solve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_solve vander(3)");
//  s += f;
//
//  f = test_LQ_solve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_solve vander(4)");
//  s += f;
//
//  f = test_LQ_solve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  LQ_solve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//
//
//int
//test_LQ_LQsolve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, dim = m->size1;
//
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * lq  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * q  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * l  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  gdl_matrix_transpose_memcpy(lq,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_LQ_decomp(lq, d);
//  s += gdl_linalg_LQ_unpack(lq, d, q, l);
//  s += gdl_linalg_LQ_LQsolve(q, l, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(lq);
//  gdl_matrix_free(q);
//  gdl_matrix_free(l);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_LQ_LQsolve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_LQ_LQsolve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_LQsolve hilbert(2)");
//  s += f;
//
//  f = test_LQ_LQsolve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_LQsolve hilbert(3)");
//  s += f;
//
//  f = test_LQ_LQsolve_dim(hilb4, hilb4_solution, 4 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_LQsolve hilbert(4)");
//  s += f;
//
//  f = test_LQ_LQsolve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  LQ_LQsolve hilbert(12)");
//  s += f;
//
//  f = test_LQ_LQsolve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_LQsolve vander(2)");
//  s += f;
//
//  f = test_LQ_LQsolve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_LQsolve vander(3)");
//  s += f;
//
//  f = test_LQ_LQsolve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_LQsolve vander(4)");
//  s += f;
//
//  f = test_LQ_LQsolve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  LQ_LQsolve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_LQ_lssolve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, M = m->size1, N = m->size2;
//
//  gdl_vector * rhs = gdl_vector_alloc(M);
//  gdl_matrix * lq  = gdl_matrix_alloc(N,M);
//  gdl_vector * d = gdl_vector_alloc(N);
//  gdl_vector * x = gdl_vector_alloc(N);
//  gdl_vector * r = gdl_vector_alloc(M);
//  gdl_vector * res = gdl_vector_alloc(M);
//
//  gdl_matrix_transpose_memcpy(lq,m);
//  for(i=0; i<M; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_LQ_decomp(lq, d);
//  s += gdl_linalg_LQ_lssolve_T(lq, d, rhs, x, res);
//
//  for(i=0; i<N; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("(%3d,%3d)[%d]: %22.18g   %22.18g\n", M, N, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//
//   /* compute residual r = b - m x */
//  if (M == N) {
//    gdl_vector_set_zero(r);
//  } else {
//    gdl_vector_memcpy(r, rhs);
//    gdl_blas_dgemv(CblasNoTrans, -1.0, m, x, 1.0, r);
//  };
//
//  for(i=0; i<N; i++) {
//    int foo = check(gdl_vector_get(res, i), gdl_vector_get(r,i), sqrt(eps));
//    if(foo) {
//      printf("(%3d,%3d)[%d]: %22.18g   %22.18g\n", M, N, i, gdl_vector_get(res, i), gdl_vector_get(r,i));
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(r);
//  gdl_vector_free(res);
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(lq);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_LQ_lssolve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_LQ_lssolve_dim(m53, m53_lssolution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_lssolve m(5,3)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_lssolve hilbert(2)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_lssolve hilbert(3)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(hilb4, hilb4_solution, 4 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_lssolve hilbert(4)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  LQ_lssolve hilbert(12)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_lssolve vander(2)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_lssolve vander(3)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_lssolve vander(4)");
//  s += f;
//
//  f = test_LQ_lssolve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  LQ_lssolve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//
//
//
//
//
//
//int
//test_LQ_decomp_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0;
//  size_t i,j, M = m->size1, N = m->size2;
//
//  gdl_matrix * lq = gdl_matrix_alloc(M,N);
//  gdl_matrix * a  = gdl_matrix_alloc(M,N);
//  gdl_matrix * q  = gdl_matrix_alloc(N,N);
//  gdl_matrix * l  = gdl_matrix_alloc(M,N);
//  gdl_vector * d = gdl_vector_alloc(GDL_MIN(M,N));
//
//  gdl_matrix_memcpy(lq,m);
//
//  s += gdl_linalg_LQ_decomp(lq, d);
//  s += gdl_linalg_LQ_unpack(lq, d, q, l);
//  
//   /* compute a = q r */
//  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, l, q, 0.0, a);
//
//  for(i=0; i<M; i++) {
//    for(j=0; j<N; j++) {
//      double aij = gdl_matrix_get(a, i, j);
//      double mij = gdl_matrix_get(m, i, j);
//      int foo = check(aij, mij, eps);
//      if(foo) {
//        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_vector_free(d);
//  gdl_matrix_free(lq);
//  gdl_matrix_free(a);
//  gdl_matrix_free(q);
//  gdl_matrix_free(l);
//
//  return s;
//}
//
//int test_LQ_decomp(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_LQ_decomp_dim(m35, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp m(3,5)");
//  s += f;
//
//  f = test_LQ_decomp_dim(m53, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp m(5,3)");
//  s += f;
//
//  f = test_LQ_decomp_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp hilbert(2)");
//  s += f;
//
//  f = test_LQ_decomp_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp hilbert(3)");
//  s += f;
//
//  f = test_LQ_decomp_dim(hilb4, 4 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp hilbert(4)");
//  s += f;
//
//  f = test_LQ_decomp_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp hilbert(12)");
//  s += f;
//
//  f = test_LQ_decomp_dim(vander2, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp vander(2)");
//  s += f;
//
//  f = test_LQ_decomp_dim(vander3, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp vander(3)");
//  s += f;
//
//  f = test_LQ_decomp_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_decomp vander(4)");
//  s += f;
//
//  f = test_LQ_decomp_dim(vander12, 0.0005);  /* FIXME: bad accuracy */
//  gdl_test(f, "  LQ_decomp vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//
//
//int
//test_PTLQ_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  int signum;
//  size_t i, dim = m->size1;
//
//  gdl_permutation * perm = gdl_permutation_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * lq  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//  gdl_vector * norm = gdl_vector_alloc(dim);
//
//  gdl_matrix_transpose_memcpy(lq,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_PTLQ_decomp(lq, d, perm, &signum, norm);
//  s += gdl_linalg_PTLQ_solve_T(lq, d, perm, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(norm);
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(lq);
//  gdl_vector_free(rhs);
//  gdl_permutation_free(perm);
//
//  return s;
//}
//
//int test_PTLQ_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_PTLQ_solve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_solve hilbert(2)");
//  s += f;
//
//  f = test_PTLQ_solve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_solve hilbert(3)");
//  s += f;
//
//  f = test_PTLQ_solve_dim(hilb4, hilb4_solution, 2 * 2048.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_solve hilbert(4)");
//  s += f;
//
//  f = test_PTLQ_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  PTLQ_solve hilbert(12)");
//  s += f;
//
//  f = test_PTLQ_solve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_solve vander(2)");
//  s += f;
//
//  f = test_PTLQ_solve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_solve vander(3)");
//  s += f;
//
//  f = test_PTLQ_solve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_solve vander(4)");
//  s += f;
//
//  f = test_PTLQ_solve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  PTLQ_solve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_PTLQ_LQsolve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  int signum;
//  size_t i, dim = m->size1;
//
//  gdl_permutation * perm = gdl_permutation_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * lq  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * q  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * l  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//  gdl_vector * norm = gdl_vector_alloc(dim);
//
//  gdl_matrix_transpose_memcpy(lq,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_PTLQ_decomp2(lq, q, l, d, perm, &signum, norm);
//  s += gdl_linalg_PTLQ_LQsolve_T(q, l, perm, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(norm);
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(lq);
//  gdl_vector_free(rhs);
//  gdl_permutation_free(perm);
//
//  return s;
//}
//
//int test_PTLQ_LQsolve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_PTLQ_LQsolve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_LQsolve hilbert(2)");
//  s += f;
//
//  f = test_PTLQ_LQsolve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_LQsolve hilbert(3)");
//  s += f;
//
//  f = test_PTLQ_LQsolve_dim(hilb4, hilb4_solution, 2 * 2048.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_LQsolve hilbert(4)");
//  s += f;
//
//  f = test_PTLQ_LQsolve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  PTLQ_LQsolve hilbert(12)");
//  s += f;
//
//  f = test_PTLQ_LQsolve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_LQsolve vander(2)");
//  s += f;
//
//  f = test_PTLQ_LQsolve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_LQsolve vander(3)");
//  s += f;
//
//  f = test_PTLQ_LQsolve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_LQsolve vander(4)");
//  s += f;
//
//  f = test_PTLQ_LQsolve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  PTLQ_LQsolve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_PTLQ_decomp_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0, signum;
//  size_t i,j, M = m->size1, N = m->size2;
//
//  gdl_matrix * lq = gdl_matrix_alloc(N,M);
//  gdl_matrix * a  = gdl_matrix_alloc(N,M);
//  gdl_matrix * q  = gdl_matrix_alloc(M,M);
//  gdl_matrix * l  = gdl_matrix_alloc(N,M);
//  gdl_vector * d = gdl_vector_alloc(GDL_MIN(M,N));
//  gdl_vector * norm = gdl_vector_alloc(N);
//
//  gdl_permutation * perm = gdl_permutation_alloc(N);
//
//  gdl_matrix_transpose_memcpy(lq,m);
//
//  s += gdl_linalg_PTLQ_decomp(lq, d, perm, &signum, norm);
//  s += gdl_linalg_LQ_unpack(lq, d, q, l);
//
//   /* compute a = l q */
//  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, l, q, 0.0, a);
//
//
//   /* Compute P LQ  by permuting the rows of LQ */
//
//  for (i = 0; i < M; i++) {
//    gdl_vector_view col = gdl_matrix_column (a, i);
//    gdl_permute_vector_inverse (perm, &col.vector);
//  }
//
//  for(i=0; i<M; i++) {
//    for(j=0; j<N; j++) {
//      double aij = gdl_matrix_get(a, j, i);
//      double mij = gdl_matrix_get(m, i, j);
//      int foo = check(aij, mij, eps);
//      if(foo) {
//        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_permutation_free (perm);
//  gdl_vector_free(norm);
//  gdl_vector_free(d);
//  gdl_matrix_free(lq);
//  gdl_matrix_free(a);
//  gdl_matrix_free(q);
//  gdl_matrix_free(l);
//
//  return s;
//}
//
//int test_PTLQ_decomp(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_PTLQ_decomp_dim(m35, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp m(3,5)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(m53, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp m(5,3)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(s35, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp s(3,5)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(s53, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp s(5,3)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp hilbert(2)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp hilbert(3)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp hilbert(4)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp hilbert(12)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(vander2, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp vander(2)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(vander3, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp vander(3)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  PTLQ_decomp vander(4)");
//  s += f;
//
//  f = test_PTLQ_decomp_dim(vander12, 0.0005);  /* FIXME: bad accuracy */
//  gdl_test(f, "  PTLQ_decomp vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_LQ_update_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0;
//  size_t i,j, M = m->size1, N = m->size2;
//
//  gdl_matrix * lq1  = gdl_matrix_alloc(N,M);
//  gdl_matrix * lq2  = gdl_matrix_alloc(N,M);
//  gdl_matrix * q1  = gdl_matrix_alloc(M,M);
//  gdl_matrix * l1  = gdl_matrix_alloc(N,M);
//  gdl_matrix * q2  = gdl_matrix_alloc(M,M);
//  gdl_matrix * l2  = gdl_matrix_alloc(N,M);
//  gdl_vector * d2 = gdl_vector_alloc(GDL_MIN(M,N));
//  gdl_vector * u = gdl_vector_alloc(M);
//  gdl_vector * v = gdl_vector_alloc(N);
//  gdl_vector * w = gdl_vector_alloc(M);
//
//  gdl_matrix_transpose_memcpy(lq1,m);
//  gdl_matrix_transpose_memcpy(lq2,m);
//  for(i=0; i<M; i++) gdl_vector_set(u, i, sin(i+1.0));
//  for(i=0; i<N; i++) gdl_vector_set(v, i, cos(i+2.0) + sin(i*i+3.0));
//
//  // lq1 is updated 
//
//  gdl_blas_dger(1.0, v, u, lq1);
//
//  // lq2 is first decomposed, updated later
//
//  s += gdl_linalg_LQ_decomp(lq2, d2);
//  s += gdl_linalg_LQ_unpack(lq2, d2, q2, l2);
//
//  // compute w = Q^T u
//
//  gdl_blas_dgemv(CblasNoTrans, 1.0, q2, u, 0.0, w);
//
//  // now lq2 is updated
//
//  s += gdl_linalg_LQ_update(q2, l2, v, w);
//
//  // multiply q2*l2
//
//  gdl_blas_dgemm(CblasNoTrans,CblasNoTrans,1.0,l2,q2,0.0,lq2);
//
//  // check lq1==lq2
//
//  for(i=0; i<N; i++) {
//    for(j=0; j<M; j++) {
//      double s1 = gdl_matrix_get(lq1, i, j);
//      double s2 = gdl_matrix_get(lq2, i, j);
//      
//      int foo = check(s1, s2, eps);
//      if(foo) {
//	  //printf("LQ:(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, s1, s2);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_vector_free(d2);
//  gdl_vector_free(u);
//  gdl_vector_free(v);
//  gdl_vector_free(w);
//  gdl_matrix_free(lq1);
//  gdl_matrix_free(lq2);
//  gdl_matrix_free(q1);
//  gdl_matrix_free(l1);
//  gdl_matrix_free(q2);
//  gdl_matrix_free(l2);
//
//  return s;
//}
//
//int test_LQ_update(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_LQ_update_dim(m35, 2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update m(3,5)");
//  s += f;
//
//  f = test_LQ_update_dim(m53, 2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update m(5,3)");
//  s += f;
//
//  f = test_LQ_update_dim(hilb2,  2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update hilbert(2)");
//  s += f;
//
//  f = test_LQ_update_dim(hilb3,  2 * 512.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update hilbert(3)");
//  s += f;
//
//  f = test_LQ_update_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update hilbert(4)");
//  s += f;
//
//  f = test_LQ_update_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update hilbert(12)");
//  s += f;
//
//  f = test_LQ_update_dim(vander2, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update vander(2)");
//  s += f;
//
//  f = test_LQ_update_dim(vander3, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update vander(3)");
//  s += f;
//
//  f = test_LQ_update_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  LQ_update vander(4)");
//  s += f;
//
//  f = test_LQ_update_dim(vander12, 0.0005);  /* FIXME: bad accuracy */
//  gdl_test(f, "  LQ_update vander(12)");
//  s += f;
//
//  return s;
//}

//int
//test_SV_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, dim = m->size1;
//
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * u  = gdl_matrix_alloc(dim,dim);
//  gdl_matrix * q  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_calloc(dim);
//  gdl_matrix_memcpy(u,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_SV_decomp(u, q, d, x);
//  s += gdl_linalg_SV_solve(u, q, d, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(u);
//  gdl_matrix_free(q);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_SV_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_SV_solve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  SV_solve hilbert(2)");
//  s += f;
//
//  f = test_SV_solve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  SV_solve hilbert(3)");
//  s += f;
//
//  f = test_SV_solve_dim(hilb4, hilb4_solution, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  SV_solve hilbert(4)");
//  s += f;
//
//  f = test_SV_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  SV_solve hilbert(12)");
//  s += f;
//
//  f = test_SV_solve_dim(vander2, vander2_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  SV_solve vander(2)");
//  s += f;
//
//  f = test_SV_solve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  SV_solve vander(3)");
//  s += f;
//
//  f = test_SV_solve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  SV_solve vander(4)");
//  s += f;
//
//  f = test_SV_solve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  SV_solve vander(12)");
//  s += f;
//
//  return s;
//}

int
test_SV_decomp_dim(const gdl_matrix * m, double eps)
{
  int s = 0;
  double di1;
  size_t i,j, M = m->size1, N = m->size2;

  gdl_matrix * v  = gdl_matrix_alloc(M,N);
  gdl_matrix * a  = gdl_matrix_alloc(M,N);
  gdl_matrix * q;
  gdl_matrix * dqt  = gdl_matrix_alloc(N,N);
  gdl_vector * d;
  
  gdl_linalg_svd_workspace * svd = gdl_linalg_svd_workspace_alloc (gdl_linalg_svd_golub_reinsch , M, N);

  gdl_matrix_memcpy(v,m);

  s += gdl_linalg_svd_workspace_perform (svd, v, GDL_MIN (M, N));
  
  d = gdl_linalg_svd_workspace_get_singular (svd);
  q = gdl_linalg_svd_workspace_get_right (svd);

  /* Check that singular values are non-negative and in non-decreasing
     order */
  
  di1 = 0.0;

  for (i = 0; i < N; i++)
    {
      double di = gdl_vector_get (d, i);

      if (gdl_isnan (di))
        {
          continue;  /* skip NaNs */
        }

      if (di < 0) {
        s++;
        printf("singular value %d = %22.18g < 0\n", i, di);
      }

      if(i > 0 && di > di1) {
        s++;
        printf("singular value %d = %22.18g vs previous %22.18g\n", i, di, di1);
      }

      di1 = di;
    }      
  
  /* Scale dqt = D Q^T */
  
  for (i = 0; i < N ; i++)
    {
      double di = gdl_vector_get (d, i);

      for (j = 0; j < N; j++)
        {
          double qji = gdl_matrix_get(q, j, i);
          gdl_matrix_set (dqt, i, j, qji * di);
        }
    }
            
  /* compute a = v dqt */
  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, v, dqt, 0.0, a);

  for(i=0; i<M; i++) {
    for(j=0; j<N; j++) {
      double aij = gdl_matrix_get(a, i, j);
      double mij = gdl_matrix_get(m, i, j);
      int foo = check(aij, mij, eps);
      if(foo) {
        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
      }
      s += foo;
    }
  }
  gdl_matrix_free(v);
  gdl_matrix_free(a);
  gdl_matrix_free(dqt);
  
  gdl_linalg_svd_workspace_free (svd);

  return s;
}

int test_SV_decomp(void)
{
  int f;
  int s = 0;

  f = test_SV_decomp_dim(m11, 2 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp m(1,1)");
  s += f;

  f = test_SV_decomp_dim(m51, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp m(5,1)");
  s += f;

  /* M<N not implemented yet */
#if 0
  f = test_SV_decomp_dim(m35, 2 * 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp m(3,5)");
  s += f;
#endif
  f = test_SV_decomp_dim(m53, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp m(5,3)");
  s += f;

  f = test_SV_decomp_dim(moler10, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp moler(10)");
  s += f;

  f = test_SV_decomp_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp hilbert(2)");
  s += f;

  f = test_SV_decomp_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp hilbert(3)");
  s += f;

  f = test_SV_decomp_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp hilbert(4)");
  s += f;

  f = test_SV_decomp_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp hilbert(12)");
  s += f;

  f = test_SV_decomp_dim(vander2, 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp vander(2)");
  s += f;

  f = test_SV_decomp_dim(vander3, 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp vander(3)");
  s += f;

  f = test_SV_decomp_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp vander(4)");
  s += f;

  f = test_SV_decomp_dim(vander12, 1e-4);
  gdl_test(f, "  SV_decomp vander(12)");
  s += f;

  f = test_SV_decomp_dim(row3, 10 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp row3");
  s += f;

  f = test_SV_decomp_dim(row5, 128 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp row5");
  s += f;

  f = test_SV_decomp_dim(row12, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp row12");
  s += f;

  f = test_SV_decomp_dim(inf5, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp inf5");
  s += f;

  f = test_SV_decomp_dim(nan5, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp nan5");
  s += f;


  {
    double i1, i2, i3, i4;
    double lower = -2, upper = 2;

    for (i1 = lower; i1 <= upper; i1++)
      {
        for (i2 = lower; i2 <= upper; i2++)
          {
            for (i3 = lower; i3 <= upper; i3++)
              {
                for (i4 = lower; i4 <= upper; i4++)
                  {
                    gdl_matrix_set (A22, 0,0, i1);
                    gdl_matrix_set (A22, 0,1, i2);
                    gdl_matrix_set (A22, 1,0, i3);
                    gdl_matrix_set (A22, 1,1, i4);
                    
                    f = test_SV_decomp_dim(A22, 16 * GDL_DBL_EPSILON);
                    gdl_test(f, "  SV_decomp (2x2) A=[%g, %g; %g, %g]", i1,i2,i3,i4);
                    s += f;
                  }
              }
          }
      }
  }

  {
    int i;
    double carry = 0, lower = 0, upper = 1;
    double *a = A33->data;

    for (i=0; i<9; i++) {
      a[i] = lower;
    }
    
    while (carry == 0.0) {
      f = test_SV_decomp_dim(A33, 64 * GDL_DBL_EPSILON);
      gdl_test(f, "  SV_decomp (3x3) A=[ %g, %g, %g; %g, %g, %g; %g, %g, %g]",
               a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
      
      /* increment */
      carry=1.0;
      for (i=9; i>0 && i--;) 
        {
          double v=a[i]+carry;
          carry = (v>upper) ? 1.0 : 0.0;
          a[i] = (v>upper) ? lower : v;
        }
    }
  }

#ifdef TEST_SVD_4X4
  {
    int i;
    double carry = 0, lower = 0, upper = 1;
    double *a = A44->data;

    for (i=0; i<16; i++) {
      a[i] = lower;
    }
    
    while (carry == 0.0) {
      f = test_SV_decomp_dim(A44, 64 * GDL_DBL_EPSILON);
      gdl_test(f, "  SV_decomp (4x4) A=[ %g, %g, %g, %g; %g, %g, %g, %g; %g, %g, %g, %g; %g, %g, %g, %g]",
               a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9],
               a[10], a[11], a[12], a[13], a[14], a[15]);
      
      /* increment */
      carry=1.0;
      for (i=16; i>0 && i--;) 
        {
          double v=a[i]+carry;
          carry = (v>upper) ? 1.0 : 0.0;
          a[i] = (v>upper) ? lower : v;
        }
    }
  }
#endif

  return s;
}


int
test_SV_decomp_mod_dim(const gdl_matrix * m, double eps)
{
  int s = 0;
  double di1;
  size_t i,j, M = m->size1, N = m->size2;

  gdl_matrix * v  = gdl_matrix_alloc(M,N);
  gdl_matrix * a  = gdl_matrix_alloc(M,N);
  gdl_matrix * q;
  gdl_matrix * dqt  = gdl_matrix_alloc(N,N);
  gdl_vector * d;
  
  gdl_linalg_svd_workspace * svd = gdl_linalg_svd_workspace_alloc (gdl_linalg_svd_m_golub_reinsch , M, N);

  gdl_matrix_memcpy(v,m);

  s += gdl_linalg_svd_workspace_perform (svd, v, GDL_MIN (M, N));

  d = gdl_linalg_svd_workspace_get_singular (svd);
  q = gdl_linalg_svd_workspace_get_right (svd);

  /* Check that singular values are non-negative and in non-decreasing
     order */
  
  di1 = 0.0;

  for (i = 0; i < N; i++)
    {
      double di = gdl_vector_get (d, i);

      if (gdl_isnan (di))
        {
          continue;  /* skip NaNs */
        }

      if (di < 0) {
        s++;
        printf("singular value %d = %22.18g < 0\n", i, di);
      }

      if(i > 0 && di > di1) {
        s++;
        printf("singular value %d = %22.18g vs previous %22.18g\n", i, di, di1);
      }

      di1 = di;
    }      
  
  /* Scale dqt = D Q^T */
  
  for (i = 0; i < N ; i++)
    {
      double di = gdl_vector_get (d, i);

      for (j = 0; j < N; j++)
        {
          double qji = gdl_matrix_get(q, j, i);
          gdl_matrix_set (dqt, i, j, qji * di);
        }
    }
            
  /* compute a = v dqt */
  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, v, dqt, 0.0, a);

  for(i=0; i<M; i++) {
    for(j=0; j<N; j++) {
      double aij = gdl_matrix_get(a, i, j);
      double mij = gdl_matrix_get(m, i, j);
      int foo = check(aij, mij, eps);
      if(foo) {
        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
      }
      s += foo;
    }
  }
  gdl_matrix_free(v);
  gdl_matrix_free(a);
  gdl_matrix_free(dqt);
  
  gdl_linalg_svd_workspace_free (svd);

  return s;
}

int test_SV_decomp_mod (void)
{
  int f;
  int s = 0;

  f = test_SV_decomp_mod_dim(m11, 2 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod m(1,1)");
  s += f;

  f = test_SV_decomp_mod_dim(m51, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod m(5,1)");
  s += f;

  /* M<N not implemented yet */
#if 0
  f = test_SV_decomp_mod_dim(m35, 2 * 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod m(3,5)");
  s += f;
#endif
  f = test_SV_decomp_mod_dim(m53, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod m(5,3)");
  s += f;

  f = test_SV_decomp_mod_dim(moler10, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod moler(10)");
  s += f;

  f = test_SV_decomp_mod_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod hilbert(2)");
  s += f;

  f = test_SV_decomp_mod_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod hilbert(3)");
  s += f;

  f = test_SV_decomp_mod_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod hilbert(4)");
  s += f;

  f = test_SV_decomp_mod_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod hilbert(12)");
  s += f;

  f = test_SV_decomp_mod_dim(vander2, 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod vander(2)");
  s += f;

  f = test_SV_decomp_mod_dim(vander3, 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod vander(3)");
  s += f;

  f = test_SV_decomp_mod_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod vander(4)");
  s += f;

  f = test_SV_decomp_mod_dim(vander12, 1e-4);
  gdl_test(f, "  SV_decomp_mod vander(12)");
  s += f;

  f = test_SV_decomp_mod_dim(row3, 10 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod row3");
  s += f;

  f = test_SV_decomp_mod_dim(row5, 128 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod row5");
  s += f;

  f = test_SV_decomp_mod_dim(row12, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod row12");
  s += f;

  f = test_SV_decomp_mod_dim(inf5, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod inf5");
  s += f;

  f = test_SV_decomp_mod_dim(nan5, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_mod nan5");
  s += f;


  {
    double i1, i2, i3, i4;
    double lower = -2, upper = 2;

    for (i1 = lower; i1 <= upper; i1++)
      {
        for (i2 = lower; i2 <= upper; i2++)
          {
            for (i3 = lower; i3 <= upper; i3++)
              {
                for (i4 = lower; i4 <= upper; i4++)
                  {
                    gdl_matrix_set (A22, 0,0, i1);
                    gdl_matrix_set (A22, 0,1, i2);
                    gdl_matrix_set (A22, 1,0, i3);
                    gdl_matrix_set (A22, 1,1, i4);
                    
                    f = test_SV_decomp_mod_dim(A22, 16 * GDL_DBL_EPSILON);
                    gdl_test(f, "  SV_decomp_mod (2x2) A=[%g, %g; %g, %g]", i1,i2,i3,i4);
                    s += f;
                  }
              }
          }
      }
  }

  {
    int i;
    double carry = 0, lower = 0, upper = 1;
    double *a = A33->data;

    for (i=0; i<9; i++) {
      a[i] = lower;
    }
    
    while (carry == 0.0) {
      f = test_SV_decomp_mod_dim(A33, 64 * GDL_DBL_EPSILON);
      gdl_test(f, "  SV_decomp_mod (3x3) A=[ %g, %g, %g; %g, %g, %g; %g, %g, %g]",
               a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
      
      /* increment */
      carry=1.0;
      for (i=9; i>0 && i--;) 
        {
          double v=a[i]+carry;
          carry = (v>upper) ? 1.0 : 0.0;
          a[i] = (v>upper) ? lower : v;
        }
    }
  }

#ifdef TEST_SVD_4X4
  {
    int i;
    double carry = 0, lower = 0, upper = 1;
    double *a = A44->data;

    for (i=0; i<16; i++) {
      a[i] = lower;
    }
    
    while (carry == 0.0) {
      f = test_SV_decomp_mod_dim(A44, 64 * GDL_DBL_EPSILON);
      gdl_test(f, "  SV_decomp_mod (4x4) A=[ %g, %g, %g, %g; %g, %g, %g, %g; %g, %g, %g, %g; %g, %g, %g, %g]",
               a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9],
               a[10], a[11], a[12], a[13], a[14], a[15]);
      
      /* increment */
      carry=1.0;
      for (i=16; i>0 && i--;) 
        {
          double v=a[i]+carry;
          carry = (v>upper) ? 1.0 : 0.0;
          a[i] = (v>upper) ? lower : v;
        }
    }
  }
#endif

  return s;
}

int
test_SV_decomp_sis2_dim (const gdl_matrix * m, double eps)
{
  int s = 0;
  double di1;
  size_t i,j, M = m->size1, N = m->size2;

  gdl_matrix * v  = gdl_matrix_alloc(M,N);
  gdl_matrix * a  = gdl_matrix_alloc(M,N);
  gdl_matrix * q;
  gdl_matrix * dqt  = gdl_matrix_alloc(N,N);
  gdl_vector * d;
  
  gdl_linalg_svd_workspace * svd = gdl_linalg_svd_workspace_alloc (gdl_linalg_svd_sis2 , M, N);

  gdl_matrix_memcpy(v, m);

  s += gdl_linalg_svd_workspace_perform (svd, v, N);

  d = gdl_linalg_svd_workspace_get_singular (svd);
  q = gdl_linalg_svd_workspace_get_right (svd);

  /* Check that singular values are non-negative and in non-decreasing
     order */
  
  di1 = 0.0;

  for (i = 0; i < N; i++)
    {
      double di = gdl_vector_get (d, i);

      if (gdl_isnan (di))
        {
          continue;  /* skip NaNs */
        }

      if (di < 0) {
        s++;
        printf("singular value %d = %22.18g < 0\n", i, di);
      }

      if(i > 0 && di > di1) {
        s++;
        printf("singular value %d = %22.18g vs previous %22.18g\n", i, di, di1);
      }

      di1 = di;
    }      
  
  /* Scale dqt = D Q^T */
  
  for (i = 0; i < N ; i++)
    {
      double di = gdl_vector_get (d, i);

      for (j = 0; j < N; j++)
        {
          double qji = gdl_matrix_get(q, j, i);
          gdl_matrix_set (dqt, i, j, qji * di);
        }
    }
            
  /* compute a = v dqt */
  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, v, dqt, 0.0, a);

  for(i=0; i<M; i++) {
    for(j=0; j<N; j++) {
      double aij = gdl_matrix_get(a, i, j);
      double mij = gdl_matrix_get(m, i, j);
      int foo = check(aij, mij, eps);
      if(foo) {
        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
      }
      s += foo;
    }
  }
  gdl_matrix_free(v);
  gdl_matrix_free(a);
  gdl_matrix_free(dqt);
  
  gdl_linalg_svd_workspace_free (svd);

  return s;
}

int test_SV_decomp_sis2 (void)
{
  int f;
  int s = 0;

  f = test_SV_decomp_sis2_dim(m11, 2 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 m(1,1)");
  s += f;

  f = test_SV_decomp_sis2_dim(m51, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 m(5,1)");
  s += f;

  /* M<N not implemented yet */
#if 0
  f = test_SV_decomp_sis2_dim(m35, 2 * 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 m(3,5)");
  s += f;
#endif
  f = test_SV_decomp_sis2_dim(m53, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 m(5,3)");
  s += f;

  f = test_SV_decomp_sis2_dim(moler10, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 moler(10)");
  s += f;

  f = test_SV_decomp_sis2_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 hilbert(2)");
  s += f;

  f = test_SV_decomp_sis2_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 hilbert(3)");
  s += f;

  f = test_SV_decomp_sis2_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 hilbert(4)");
  s += f;

  f = test_SV_decomp_sis2_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 hilbert(12)");
  s += f;

  f = test_SV_decomp_sis2_dim(vander2, 8.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 vander(2)");
  s += f;

  f = test_SV_decomp_sis2_dim(vander3, 64.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 vander(3)");
  s += f;

  f = test_SV_decomp_sis2_dim(vander4, 1024.0 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 vander(4)");
  s += f;

  f = test_SV_decomp_sis2_dim(vander12, 1e-4);
  gdl_test(f, "  SV_decomp_sis2 vander(12)");
  s += f;

  f = test_SV_decomp_sis2_dim(row3, 10 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 row3");
  s += f;

  f = test_SV_decomp_sis2_dim(row5, 128 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 row5");
  s += f;

  f = test_SV_decomp_sis2_dim(row12, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 row12");
  s += f;

  f = test_SV_decomp_sis2_dim(inf5, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 inf5");
  s += f;

  f = test_SV_decomp_sis2_dim(nan5, 1024 * GDL_DBL_EPSILON);
  gdl_test(f, "  SV_decomp_sis2 nan5");
  s += f;


  {
    double i1, i2, i3, i4;
    double lower = -2, upper = 2;

    for (i1 = lower; i1 <= upper; i1++)
      {
        for (i2 = lower; i2 <= upper; i2++)
          {
            for (i3 = lower; i3 <= upper; i3++)
              {
                for (i4 = lower; i4 <= upper; i4++)
                  {
                    gdl_matrix_set (A22, 0,0, i1);
                    gdl_matrix_set (A22, 0,1, i2);
                    gdl_matrix_set (A22, 1,0, i3);
                    gdl_matrix_set (A22, 1,1, i4);
                    
                    f = test_SV_decomp_sis2_dim(A22, 16 * GDL_DBL_EPSILON);
                    gdl_test(f, "  SV_decomp_sis2 (2x2) A=[%g, %g; %g, %g]", i1,i2,i3,i4);
                    s += f;
                  }
              }
          }
      }
  }

  {
    int i;
    double carry = 0, lower = 0, upper = 1;
    double *a = A33->data;

    for (i=0; i<9; i++) {
      a[i] = lower;
    }
    
    while (carry == 0.0) {
      f = test_SV_decomp_sis2_dim(A33, 64 * GDL_DBL_EPSILON);
      gdl_test(f, "  SV_decomp_sis2 (3x3) A=[ %g, %g, %g; %g, %g, %g; %g, %g, %g]",
               a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
      
      /* increment */
      carry=1.0;
      for (i=9; i>0 && i--;) 
        {
          double v=a[i]+carry;
          carry = (v>upper) ? 1.0 : 0.0;
          a[i] = (v>upper) ? lower : v;
        }
    }
  }

#ifdef TEST_SVD_4X4
  {
    int i;
    double carry = 0, lower = 0, upper = 1;
    double *a = A44->data;

    for (i=0; i<16; i++) {
      a[i] = lower;
    }
    
    while (carry == 0.0) {
      f = test_SV_decomp_sis2_dim(A44, 64 * GDL_DBL_EPSILON);
      gdl_test(f, "  SV_decomp_sis2 (4x4) A=[ %g, %g, %g, %g; %g, %g, %g, %g; %g, %g, %g, %g; %g, %g, %g, %g]",
               a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9],
               a[10], a[11], a[12], a[13], a[14], a[15]);
      
      /* increment */
      carry=1.0;
      for (i=16; i>0 && i--;) 
        {
          double v=a[i]+carry;
          carry = (v>upper) ? 1.0 : 0.0;
          a[i] = (v>upper) ? lower : v;
        }
    }
  }
#endif

  return s;
}
//int
//test_cholesky_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, dim = m->size1;
//
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_matrix * u  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * x = gdl_vector_calloc(dim);
//  gdl_matrix_memcpy(u,m);
//  for(i=0; i<dim; i++) gdl_vector_set(rhs, i, i+1.0);
//  s += gdl_linalg_cholesky_decomp(u);
//  s += gdl_linalg_cholesky_solve(u, rhs, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i), actual[i], eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//  gdl_vector_free(x);
//  gdl_matrix_free(u);
//  gdl_vector_free(rhs);
//
//  return s;
//}
//
//int test_cholesky_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_cholesky_solve_dim(hilb2, hilb2_solution, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  cholesky_solve hilbert(2)");
//  s += f;
//
//  f = test_cholesky_solve_dim(hilb3, hilb3_solution, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  cholesky_solve hilbert(3)");
//  s += f;
//
//  f = test_cholesky_solve_dim(hilb4, hilb4_solution, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  cholesky_solve hilbert(4)");
//  s += f;
//
//  f = test_cholesky_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  cholesky_solve hilbert(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_cholesky_decomp_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0;
//  size_t i,j, M = m->size1, N = m->size2;
//
//  gdl_matrix * v  = gdl_matrix_alloc(M,N);
//  gdl_matrix * a  = gdl_matrix_alloc(M,N);
//  gdl_matrix * l  = gdl_matrix_alloc(M,N);
//  gdl_matrix * lt  = gdl_matrix_alloc(N,N);
//
//  gdl_matrix_memcpy(v,m);
//
//  s += gdl_linalg_cholesky_decomp(v);
//  
//  /* Compute L LT */
//  
//  for (i = 0; i < N ; i++)
//    {
//      for (j = 0; j < N; j++)
//        {
//          double vij = gdl_matrix_get(v, i, j);
//          gdl_matrix_set (l, i, j, i>=j ? vij : 0);
//          gdl_matrix_set (lt, i, j, i<=j ? vij : 0);
//        }
//    }
//            
//  /* compute a = l lt */
//  gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, l, lt, 0.0, a);
//
//  for(i=0; i<M; i++) {
//    for(j=0; j<N; j++) {
//      double aij = gdl_matrix_get(a, i, j);
//      double mij = gdl_matrix_get(m, i, j);
//      int foo = check(aij, mij, eps);
//      if(foo) {
//        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_matrix_free(v);
//  gdl_matrix_free(a);
//  gdl_matrix_free(l);
//  gdl_matrix_free(lt);
//
//  return s;
//}
//
//int test_cholesky_decomp(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_cholesky_decomp_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  cholesky_decomp hilbert(2)");
//  s += f;
//
//  f = test_cholesky_decomp_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  cholesky_decomp hilbert(3)");
//  s += f;
//
//  f = test_cholesky_decomp_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  cholesky_decomp hilbert(4)");
//  s += f;
//
//  f = test_cholesky_decomp_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  cholesky_decomp hilbert(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_HH_solve_dim(const gdl_matrix * m, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i, dim = m->size1;
//
//  gdl_permutation * perm = gdl_permutation_alloc(dim);
//  gdl_matrix * hh  = gdl_matrix_alloc(dim,dim);
//  gdl_vector * d = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//  gdl_matrix_memcpy(hh,m);
//  for(i=0; i<dim; i++) gdl_vector_set(x, i, i+1.0);
//  s += gdl_linalg_HH_svx(hh, x);
//  for(i=0; i<dim; i++) {
//    int foo = check(gdl_vector_get(x, i),actual[i],eps);
//    if( foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//  gdl_vector_free(x);
//  gdl_vector_free(d);
//  gdl_matrix_free(hh);
//  gdl_permutation_free(perm);
//
//  return s;
//}
//
//int test_HH_solve(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_HH_solve_dim(hilb2, hilb2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  HH_solve hilbert(2)");
//  s += f;
//
//  f = test_HH_solve_dim(hilb3, hilb3_solution, 128.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  HH_solve hilbert(3)");
//  s += f;
//
//  f = test_HH_solve_dim(hilb4, hilb4_solution, 2.0 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  HH_solve hilbert(4)");
//  s += f;
//
//  f = test_HH_solve_dim(hilb12, hilb12_solution, 0.5);
//  gdl_test(f, "  HH_solve hilbert(12)");
//  s += f;
//
//  f = test_HH_solve_dim(vander2, vander2_solution, 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  HH_solve vander(2)");
//  s += f;
//
//  f = test_HH_solve_dim(vander3, vander3_solution, 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  HH_solve vander(3)");
//  s += f;
//
//  f = test_HH_solve_dim(vander4, vander4_solution, 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  HH_solve vander(4)");
//  s += f;
//
//  f = test_HH_solve_dim(vander12, vander12_solution, 0.05);
//  gdl_test(f, "  HH_solve vander(12)");
//  s += f;
//
//  return s;
//}
//
//
//int
//test_TDS_solve_dim(size_t dim, double d, double od, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i;
//
//  gdl_vector * offdiag = gdl_vector_alloc(dim-1);
//  gdl_vector * diag = gdl_vector_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  for(i=0; i<dim; i++) {
//    gdl_vector_set(diag, i, d);
//    gdl_vector_set(rhs,  i, i + 1.0);
//  }
//  for(i=0; i<dim-1; i++) {
//    gdl_vector_set(offdiag, i, od);
//  }
//
//  s += gdl_linalg_solve_symm_tridiag(diag, offdiag, rhs, x);
//
//  for(i=0; i<dim; i++) {
//    double si = gdl_vector_get(x, i);
//    double ai = actual[i];
//    int foo = check(si, ai, eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(rhs);
//  gdl_vector_free(diag);
//  gdl_vector_free(offdiag);
//
//  return s;
//}
//
//
//int test_TDS_solve(void)
//{
//  int f;
//  int s = 0;
//
//  {
//    double actual[] =  {0.0, 2.0};
//    f = test_TDS_solve_dim(2, 1.0, 0.5, actual, 8.0 * GDL_DBL_EPSILON);
//    gdl_test(f, "  solve_TDS dim=2 A");
//    s += f;
//  }
//
//  {
//    double actual[] =  {3.0/8.0, 15.0/8.0};
//    f = test_TDS_solve_dim(2, 1.0, 1.0/3.0, actual, 8.0 * GDL_DBL_EPSILON);
//    gdl_test(f, "  solve_TDS dim=2 B");
//    s += f;
//  }
//
//  {
//    double actual[] =  {5.0/8.0, 9.0/8.0, 2.0, 15.0/8.0, 35.0/8.0};
//    f = test_TDS_solve_dim(5, 1.0, 1.0/3.0, actual, 8.0 * GDL_DBL_EPSILON);
//    gdl_test(f, "  solve_TDS dim=5");
//    s += f;
//  }
//
//  return s;
//}
//
//int
//test_TDS_cyc_solve_one(const size_t dim, const double * d, const double * od,
//                      const double * r, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i;
//
//  gdl_vector * offdiag = gdl_vector_alloc(dim);
//  gdl_vector * diag = gdl_vector_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  for(i=0; i<dim; i++) {
//    gdl_vector_set(diag, i, d[i]);
//    gdl_vector_set(rhs,  i, r[i]);
//    gdl_vector_set(offdiag, i, od[i]);
//  }
//
//  s += gdl_linalg_solve_symm_cyc_tridiag(diag, offdiag, rhs, x);
//
//  for(i=0; i<dim; i++) {
//    double si = gdl_vector_get(x, i);
//    double ai = actual[i];
//    int foo = check(si, ai, eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(rhs);
//  gdl_vector_free(diag);
//  gdl_vector_free(offdiag);
//
//  return s;
//}
//
//int test_TDS_cyc_solve(void)
//{
//  int f;
//  int s = 0;
//
//#ifdef SUPPORT_UNDERSIZE_CYC
//  {
//    size_t dim = 1;
//    double diag[] = {  2 };
//    double offdiag[] = { 3 };
//    double rhs[] = { 7 };
//    double actual[] = { 3.5 };
//    
//    f = test_TDS_cyc_solve_one(dim, diag, offdiag, rhs, actual, 28.0 * GDL_DBL_EPSILON);
//    gdl_test(f, "  solve_TDS_cyc dim=%u A", dim);
//    s += f;
//  }
//
//  {
//    size_t dim = 2;
//    double diag[] = { 1, 2 };
//    double offdiag[] = { 3, 4 };
//    double rhs[] = { 7, -7 };
//    double actual[] = { -5, 4 };
//    
//    f = test_TDS_cyc_solve_one(dim, diag, offdiag, rhs, actual, 28.0 * GDL_DBL_EPSILON);
//    gdl_test(f, "  solve_TDS_cyc dim=%u A", dim);
//    s += f;
//  }
//#endif
//
//  {
//    size_t dim = 3;
//    double diag[] = { 1, 1, 1 };
//    double offdiag[] = { 3, 3, 3 };
//    double rhs[] = { 7, -7, 7 };
//    double actual[] = { -2, 5, -2 };
//    
//    f = test_TDS_cyc_solve_one(dim, diag, offdiag, rhs, actual, 28.0 * GDL_DBL_EPSILON);
//    gdl_test(f, "  solve_TDS_cyc dim=%u A", dim);
//    s += f;
//  }
//
//  {
//    size_t dim = 5;
//    double diag[] = { 4, 2, 1, 2, 4 };
//    double offdiag[] = { 1, 1, 1, 1, 1 };
//    double rhs[] = { 30, -24, 3, 21, -30 };
//    double actual[] = { 12, 3, -42, 42, -21 };
//
//    /*  f = test_TDS_cyc_solve_one(dim, diag, offdiag, rhs, actual, 7.0 * GDL_DBL_EPSILON);
//        FIXME: bad accuracy */
//    f = test_TDS_cyc_solve_one(dim, diag, offdiag, rhs, actual, 35.0 * GDL_DBL_EPSILON);
//    gdl_test(f, "  solve_TDS_cyc dim=%u B", dim);
//    s += f;
//  }
//
//  return s;
//}
//
//int
//test_TDN_solve_dim(size_t dim, double d, double a, double b, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i;
//
//  gdl_vector * abovediag = gdl_vector_alloc(dim-1);
//  gdl_vector * belowdiag = gdl_vector_alloc(dim-1);
//  gdl_vector * diag = gdl_vector_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  for(i=0; i<dim; i++) {
//    gdl_vector_set(diag, i, d);
//    gdl_vector_set(rhs,  i, i + 1.0);
//  }
//  for(i=0; i<dim-1; i++) {
//    gdl_vector_set(abovediag, i, a);
//    gdl_vector_set(belowdiag, i, b);
//  }
//
//  s += gdl_linalg_solve_tridiag(diag, abovediag, belowdiag, rhs, x);
//
//  for(i=0; i<dim; i++) {
//    double si = gdl_vector_get(x, i);
//    double ai = actual[i];
//    int foo = check(si, ai, eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(rhs);
//  gdl_vector_free(diag);
//  gdl_vector_free(abovediag);
//  gdl_vector_free(belowdiag);
//
//  return s;
//}
//
//
//int test_TDN_solve(void)
//{
//  int f;
//  int s = 0;
//  double actual[16];
//
//  actual[0] =  -7.0/3.0;
//  actual[1] =  5.0/3.0;
//  actual[2] =  4.0/3.0;
//  f = test_TDN_solve_dim(3, 1.0, 2.0, 1.0, actual, 2.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  solve_TDN dim=2 A");
//  s += f;
//
//  actual[0] =  0.75;
//  actual[1] =  0.75;
//  actual[2] =  2.625;
//  f = test_TDN_solve_dim(3, 1.0, 1.0/3.0, 1.0/2.0, actual, 2.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  solve_TDN dim=2 B");
//  s += f;
//
//  actual[0] =  99.0/140.0;
//  actual[1] =  41.0/35.0;
//  actual[2] =  19.0/10.0;
//  actual[3] =  72.0/35.0;
//  actual[4] =  139.0/35.0;
//  f = test_TDN_solve_dim(5, 1.0, 1.0/4.0, 1.0/2.0, actual, 35.0/8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  solve_TDN dim=5");
//  s += f;
//
//  return s;
//}
//
//int
//test_TDN_cyc_solve_dim(size_t dim, double d, double a, double b, const double * actual, double eps)
//{
//  int s = 0;
//  size_t i;
//
//  gdl_vector * abovediag = gdl_vector_alloc(dim);
//  gdl_vector * belowdiag = gdl_vector_alloc(dim);
//  gdl_vector * diag = gdl_vector_alloc(dim);
//  gdl_vector * rhs = gdl_vector_alloc(dim);
//  gdl_vector * x = gdl_vector_alloc(dim);
//
//  for(i=0; i<dim; i++) {
//    gdl_vector_set(diag, i, d);
//    gdl_vector_set(rhs,  i, i + 1.0);
//  }
//  for(i=0; i<dim; i++) {
//    gdl_vector_set(abovediag, i, a);
//    gdl_vector_set(belowdiag, i, b);
//  }
//
//  s += gdl_linalg_solve_cyc_tridiag(diag, abovediag, belowdiag, rhs, x);
//
//  for(i=0; i<dim; i++) {
//    double si = gdl_vector_get(x, i);
//    double ai = actual[i];
//    int foo = check(si, ai, eps);
//    if(foo) {
//      printf("%3d[%d]: %22.18g   %22.18g\n", dim, i, gdl_vector_get(x, i), actual[i]);
//    }
//    s += foo;
//  }
//
//  gdl_vector_free(x);
//  gdl_vector_free(rhs);
//  gdl_vector_free(diag);
//  gdl_vector_free(abovediag);
//  gdl_vector_free(belowdiag);
//
//  return s;
//}
//
//
//int test_TDN_cyc_solve(void)
//{
//  int f;
//  int s = 0;
//  double actual[16];
//
//  actual[0] =  3.0/2.0;
//  actual[1] = -1.0/2.0;
//  actual[2] =  1.0/2.0;
//  f = test_TDN_cyc_solve_dim(3, 1.0, 2.0, 1.0, actual, 32.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  solve_TDN_cyc dim=2 A");
//  s += f;
//
//  actual[0] = -5.0/22.0;
//  actual[1] = -3.0/22.0;
//  actual[2] =  29.0/22.0;
//  actual[3] = -9.0/22.0;
//  actual[4] =  43.0/22.0;
//  f = test_TDN_cyc_solve_dim(5, 3.0, 2.0, 1.0, actual, 66.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  solve_TDN_cyc dim=5");
//  s += f;
//
//  return s;
//}
//
//int
//test_bidiag_decomp_dim(const gdl_matrix * m, double eps)
//{
//  int s = 0;
//  size_t i,j,k,r, M = m->size1, N = m->size2;
//
//  gdl_matrix * A  = gdl_matrix_alloc(M,N);
//  gdl_matrix * a  = gdl_matrix_alloc(M,N);
//  gdl_matrix * b  = gdl_matrix_alloc(N,N);
//
//  gdl_matrix * u  = gdl_matrix_alloc(M,N);
//  gdl_matrix * v  = gdl_matrix_alloc(N,N);
//
//  gdl_vector * tau1  = gdl_vector_alloc(N);
//  gdl_vector * tau2  = gdl_vector_alloc(N-1);
//  gdl_vector * d  = gdl_vector_alloc(N);
//  gdl_vector * sd  = gdl_vector_alloc(N-1);
//
//  gdl_matrix_memcpy(A,m);
//
//  s += gdl_linalg_bidiag_decomp(A, tau1, tau2);
//  s += gdl_linalg_bidiag_unpack(A, tau1, u, tau2, v, d, sd);
//
//  gdl_matrix_set_zero(b);
//  for (i = 0; i < N; i++) gdl_matrix_set(b, i,i, gdl_vector_get(d,i));
//  for (i = 0; i < N-1; i++) gdl_matrix_set(b, i,i+1, gdl_vector_get(sd,i));
//  
//  /* Compute A = U B V^T */
//  
//  for (i = 0; i < M ; i++)
//    {
//      for (j = 0; j < N; j++)
//        {
//          double sum = 0;
//
//          for (k = 0; k < N; k++)
//            {
//              for (r = 0; r < N; r++)
//                {
//                  sum += gdl_matrix_get(u, i, k) * gdl_matrix_get (b, k, r)
//                    * gdl_matrix_get(v, j, r);
//                }
//            }
//          gdl_matrix_set (a, i, j, sum);
//        }
//    }
//
//  for(i=0; i<M; i++) {
//    for(j=0; j<N; j++) {
//      double aij = gdl_matrix_get(a, i, j);
//      double mij = gdl_matrix_get(m, i, j);
//      int foo = check(aij, mij, eps);
//      if(foo) {
//        printf("(%3d,%3d)[%d,%d]: %22.18g   %22.18g\n", M, N, i,j, aij, mij);
//      }
//      s += foo;
//    }
//  }
//
//  gdl_matrix_free(A);
//  gdl_matrix_free(a);
//  gdl_matrix_free(u);
//  gdl_matrix_free(v);
//  gdl_matrix_free(b);
//  gdl_vector_free(tau1);
//  gdl_vector_free(tau2);
//  gdl_vector_free(d);
//  gdl_vector_free(sd);
//
//  return s;
//}
//
//int test_bidiag_decomp(void)
//{
//  int f;
//  int s = 0;
//
//  f = test_bidiag_decomp_dim(m53, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  bidiag_decomp m(5,3)");
//  s += f;
//
//  f = test_bidiag_decomp_dim(m97, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  bidiag_decomp m(9,7)");
//  s += f;
//
//  f = test_bidiag_decomp_dim(hilb2, 2 * 8.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  bidiag_decomp hilbert(2)");
//  s += f;
//
//  f = test_bidiag_decomp_dim(hilb3, 2 * 64.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  bidiag_decomp hilbert(3)");
//  s += f;
//
//  f = test_bidiag_decomp_dim(hilb4, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  bidiag_decomp hilbert(4)");
//  s += f;
//
//  f = test_bidiag_decomp_dim(hilb12, 2 * 1024.0 * GDL_DBL_EPSILON);
//  gdl_test(f, "  bidiag_decomp hilbert(12)");
//  s += f;
//
//  return s;
//}


int main(void)
{
  m11 = create_general_matrix(1,1);
  m51 = create_general_matrix(5,1);

  m35 = create_general_matrix(3,5);
  m53 = create_general_matrix(5,3);
  m97 = create_general_matrix(9,7);

  s35 = create_singular_matrix(3,5);
  s53 = create_singular_matrix(5,3);

  hilb2 = create_hilbert_matrix(2);
  hilb3 = create_hilbert_matrix(3);
  hilb4 = create_hilbert_matrix(4);
  hilb12 = create_hilbert_matrix(12);

  vander2 = create_vandermonde_matrix(2);
  vander3 = create_vandermonde_matrix(3);
  vander4 = create_vandermonde_matrix(4);
  vander12 = create_vandermonde_matrix(12);

  moler10 = create_moler_matrix(10);

  row3 = create_row_matrix(3,3);
  row5 = create_row_matrix(5,5);
  row12 = create_row_matrix(12,12);

  A22 = create_2x2_matrix (0.0, 0.0, 0.0, 0.0);
  A33 = gdl_matrix_alloc(3,3);
  A44 = gdl_matrix_alloc(4,4);

  inf5 = create_diagonal_matrix (inf5_data, 5);
  gdl_matrix_set(inf5, 3, 3, GDL_POSINF);

  nan5 = create_diagonal_matrix (inf5_data, 5);
  gdl_matrix_set(nan5, 3, 3, GDL_NAN);

//  gdl_test(test_bidiag_decomp(),  "Bidiagonal Decomposition");
//  gdl_test(test_LU_solve(),       "LU Decomposition and Solve");
//  gdl_test(test_LUc_solve(),      "Complex LU Decomposition and Solve");
//  gdl_test(test_QR_decomp(),      "QR Decomposition");
//  gdl_test(test_QR_solve(),       "QR Solve");
//  gdl_test(test_LQ_solve(),       "LQ Solve");
//  gdl_test(test_PTLQ_solve(),     "PTLQ Solve");
//
//  gdl_test(test_LQ_decomp(),      "LQ Decomposition");
//  gdl_test(test_LQ_LQsolve(),     "LQ LQ Solve");
//  gdl_test(test_LQ_lssolve(),     "LQ LS Solve");
//  gdl_test(test_LQ_update(),      "LQ Rank-1 Update");
//  gdl_test(test_QRPT_decomp(),    "PTLQ Decomposition");
//  gdl_test(test_PTLQ_solve(),     "PTLQ Solve");
//
//  gdl_test(test_QR_QRsolve(),     "QR QR Solve");
//  gdl_test(test_QR_lssolve(),     "QR LS Solve");
//  gdl_test(test_QR_update(),      "QR Rank-1 Update");
//  gdl_test(test_QRPT_decomp(),    "QRPT Decomposition");
//  gdl_test(test_QRPT_solve(),     "QRPT Solve");
//  gdl_test(test_QRPT_QRsolve(),   "QRPT QR Solve");
    gdl_test(test_SV_decomp(),      "Singular Value Decomposition");
    gdl_test(test_SV_decomp_mod (),  "Singular Value Decomposition (Mod)");
    gdl_test(test_SV_decomp_sis2(), "Singular Value Decomposition (Sis2)");
//  gdl_test(test_SV_solve(),       "SVD Solve");
//  gdl_test(test_cholesky_decomp(),"Cholesky Decomposition");
//  gdl_test(test_cholesky_solve(), "Cholesky Solve");
//  gdl_test(test_HH_solve(),       "Householder solve");
//  gdl_test(test_TDS_solve(),      "Tridiagonal symmetric solve");
//  gdl_test(test_TDS_cyc_solve(),  "Tridiagonal symmetric cyclic solve");
//  gdl_test(test_TDN_solve(),      "Tridiagonal nonsymmetric solve");
//  gdl_test(test_TDN_cyc_solve(),  "Tridiagonal nonsymmetric cyclic solve");

//  gdl_matrix_free(m35);
//  gdl_matrix_free(m53);
//  gdl_matrix_free(m97);
//  gdl_matrix_free(s35);
//  gdl_matrix_free(s53);
//
//  gdl_matrix_free(hilb2);
//  gdl_matrix_free(hilb3);
//  gdl_matrix_free(hilb4);
//  gdl_matrix_free(hilb12);
//
//  gdl_matrix_free(vander2);
//  gdl_matrix_free(vander3);
//  gdl_matrix_free(vander4);
//  gdl_matrix_free(vander12);
//
//  gdl_matrix_free(moler10);
//
//  gdl_matrix_complex_free(c7);
//  gdl_matrix_free(row3);
//  gdl_matrix_free(row5);
//  gdl_matrix_free(row12);
//
//  gdl_matrix_free(A22);
//  gdl_matrix_free(A33);
//  gdl_matrix_free(A44);

  exit (gdl_test_summary());
}
