/*  
 *  linalg/varimax.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:08 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_pow_int.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_linalg.h>

int
gdl_linalg_varimax (gdl_matrix * X, gdl_matrix * T, gdl_matrix * Z, double epsilon, size_t max_iter)
{
	const size_t M = X->size1;
	const size_t N = X->size2;
	int status;
	size_t i, j, iter;
	double d, dpast, * z, u;
	gdl_vector * sc, * w;
	gdl_matrix * B, * V;
   gdl_linalg_svd_workspace * SVD;
      
   if (M > 10*N)
   {
   	SVD = gdl_linalg_svd_workspace_alloc (gdl_linalg_svd_m_golub_reinsch, N, N);
   }
   else
   {
   	SVD = gdl_linalg_svd_workspace_alloc (gdl_linalg_svd_golub_reinsch, N, N);
   }
   
   B  = gdl_matrix_alloc (N, N);
   sc = gdl_vector_alloc (M);
   w  = gdl_linalg_svd_workspace_get_singular_ptr (SVD);
   V  = gdl_linalg_svd_workspace_get_right_ptr (SVD);
   
   /* Normalized the matrix X */
	for(i = 0; i < M; i++)
	{
		gdl_vector_view row = gdl_matrix_row (X, i);
		gdl_vector_set (sc, i, gdl_blas_dnrm2 (&(row.vector)));
		gdl_blas_dscal (gdl_vector_get (sc, i), &(row.vector));
	}
	/* Initialize the rotation matrix as the identity matrix */
	gdl_matrix_set_identity (T);
   	
   for(iter = 1; iter <= max_iter; iter++)
   {
   	 /*  z <- x %*% TT */
   	gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, X, T, 0.0, Z);
   	
   	/* z <- z^3 - z %*% diag(drop(rep(1, p) %*% z^2))/p */
   	for(j = 0; j < N; j++)
   	{
   		gdl_vector_view col = gdl_matrix_column (Z, j);
   		u = gdl_blas_dnrm2 (&(col.vector));
   		gdl_vector_set (w, j, u*u/M);
   	}
   	
		for(i = 0; i < M; i++)
		{
			for(j = 0; j < N; j++) 
			{
				u   = gdl_matrix_get (Z, i, j);
				u  = u*(u*u - gdl_vector_get (w, j));
				gdl_matrix_set (Z, i, j, u);
			}
		}
		
		gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1.0, X, Z, 0.0, B);
			
	   gdl_linalg_svd_workspace_perform (SVD, B, N);
		
		/* TT <- B %*% t(V) */
		
		gdl_blas_dgemm (CblasNoTrans, CblasTrans, 1.0, B, V, 0.0, T);
		
		dpast = d;
		
		d = gdl_blas_dasum (w);
		
		//printf("VARIMAX Iter\t%d\t%e\n", iter, (d-dpast));
		
		if (fabs(d-dpast) < epsilon)
		{
			break;
		}		
   }
   if (iter > max_iter)
   {
		status = GDL_EMAXITER;
	}
	
	gdl_blas_dgemm (CblasNoTrans, CblasNoTrans, 1.0, X, T, 0.0, Z);
	
	/* z <- z * sc */
  	for(i = 0; i < M; i++)
  	{  
  		gdl_vector_view row = gdl_matrix_row (Z, i);
		gdl_blas_dscal (gdl_vector_get (sc, i), &(row.vector));
	}
	
	gdl_linalg_svd_workspace_free (SVD);
	gdl_vector_free (sc);
	gdl_matrix_free (B);
	
	return status;	
}

