/*  
 * 	list/list.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:45 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_list.h>

#include "private.c"

gdl_list *
gdl_list_alloc(const gdl_data_interface * type)
{
    gdl_list * l;
    
    l = GDL_CALLOC (gdl_list, 1);
    
    l->state = GDL_CALLOC (gdl_link, 1);
    l->state->next = l->state;
    l->state->prev = l->state;
    l->state->data = NULL;
    
    //l->links       = NULL;
        
    if (type != NULL)
    {
    	l->type = type;
    	// to avoid systematic test in routines.
    	if (l->type->compare == NULL)
    		l->compare = &gdl_default_link_compare;
    	else
    		l->compare = l->type->compare;
    }
    else
    {
    	l->type = &_gdl_list_type_default;
    }
    
    l->size      = 0;
    
    return l;
}
    
void *
gdl_list_search (gdl_list * l, void * data) 
{
    gdl_link * lk;
    if (l == NULL)
        return(NULL);
    lk = gdl_list_link_search (l, data);
    if (lk)
        return (lk->data);
    return NULL;
}

void *
gdl_list_rsearch (gdl_list * l, void * data) 
{
    gdl_link * lk;
    if (l == NULL)
        return(NULL);
    lk = gdl_list_link_rsearch (l, data);
    if (lk)
        return (lk->data);
    return NULL;
}

int
gdl_list_insert (gdl_list * l, void * data, size_t owner) 
{
    gdl_link * lkp, * lkn;

    if (l == NULL)
        return(1);
    lkp       = gdl_list_lower_search (l, data);
    /* Add the new link */
    lkn       = GDL_CALLOC (gdl_link, 1);
    lkn->data = data;
    lkp       = lkp->prev;
    lkn->next = lkp->next;
    (lkp->next)->prev = lkn;
    lkp->next         = lkn;
    lkn->prev         = lkp;
    lkn->owner        = owner;
    l->size++;
    return 0;
}

int
gdl_list_append (gdl_list * l, void * data, size_t owner) 
{
    gdl_link * lkp, * lkn;

    if (l == NULL)
        return(1);
    lkp = gdl_list_higher_search (l, data);
    /* Add the new link */
    lkn = GDL_CALLOC (gdl_link, 1);
    lkn->data = data;
    lkn->next = lkp->next;
    (lkp->next)->prev = lkn;
    lkp->next = lkn;
    lkn->prev = lkp;
    lkn->owner = owner;
    l->size++;
    return 1;
}

void
gdl_list_free (gdl_list * l)
{
    if (l == NULL)
        return;
    gdl_list_clear (l);
    GDL_FREE (l->state);
    GDL_FREE (l);
}

int
gdl_list_remove_first (gdl_list * l, void * data)
{
    gdl_link * lk;
    
    if (l == NULL)
        return(0);
    /*Find the first instance of this data */
    lk = gdl_list_link_search (l, data);
    if (lk != NULL) {
        gdl_link_free (l, lk);
        l->size--;
        return 1;
    }
    return 0;
}

int
gdl_list_remove_last (gdl_list * l, void * data)
{
    gdl_link * lk;
    
    if (l == NULL)
        return(0);
    /*Find the last instance of this data */
    lk = gdl_list_link_rsearch (l, data);
    if (lk != NULL) {
		gdl_link_free (l, lk);
		l->size--;
        return 1;
    }
    return 0;
}

int
gdl_list_remove_all (gdl_list * l, void * data)
{
    int count=0;
    if (l == NULL)
        return(0);
	while(gdl_list_remove_first (l, data))
	{
	    count++;
	}
	return count;
}

void
gdl_list_clear (gdl_list * l)
{
    gdl_link *  lk;
    
    if (l == NULL)
        return;
        
    lk = l->state->next;
    
    while(lk != l->state) {
        gdl_link * next = lk->next;
        gdl_link_free (l, lk);
        l->size--;
        lk = next;
    }
}

int
gdl_list_empty (const gdl_list * l)
{
    if (l == NULL)
        return(-1);
    return (l->size == 0);
}

gdl_link * 
gdl_list_front (gdl_list * l)
{
    if (l == NULL)
        return(NULL);
    return (l->state->next);
}

void *
gdl_list_shift_front (gdl_list * l)
{
	if (l == NULL)
        return(NULL);
    {
    	gdl_link *  lk  = l->state->next;
    	gdl_link * next = lk->next;
    	void * data = lk->data;
    	
    	lk->data = NULL;
    	gdl_link_free (l, lk);
    	
    	l->state->next = next;
    	next->prev     = l->state;
    	
    	l->size--;
        
        return data;
    }
}

void *
gdl_list_shift_back (gdl_list * l)
{
	if (l == NULL)
        return(NULL);
    {
    	gdl_link *  lk  = l->state->prev;
    	gdl_link * prev = lk->prev;
    	void * data = lk->data;
    	
    	lk->data = NULL;
    	gdl_link_free (l, lk);
    	
    	l->state->prev = prev;
    	prev->next     = l->state;
    	
    	l->size--;
        
        return data;
    }
}
    
gdl_link * 
gdl_list_end (gdl_list * l)
{
    if (l == NULL)
        return(NULL);
    return (l->state->prev);
}
    
size_t
gdl_list_size (gdl_list * l)
{
    if (l == NULL)
        return(-1);
    return l->size;
}

int
gdl_list_push_front (gdl_list * l, void * data, size_t owner) 
{
    gdl_link * lkp, * lkn;

    if (l == NULL)
        return(0);
    lkp = l->state;
    /* Add the new link */
    lkn = GDL_CALLOC (gdl_link, 1);
    lkn->data  = data;
    lkn->owner = owner;
    lkn->next  = lkp->next;
    (lkp->next)->prev = lkn;
    lkp->next  = lkn;
    lkn->prev  = lkp;    
    l->size++;
    return 1;
}

int
gdl_list_push_back (gdl_list * l, void * data, size_t owner) 
{
    gdl_link * lkp, * lkn;

    if (l == NULL)
        return(0);
    lkp = l->state->prev;
    /* Add the new link */
    lkn        = GDL_CALLOC (gdl_link, 1);
    lkn->data  = data;
    lkn->owner = owner;
    lkn->next  = lkp->next;
    (lkp->next)->prev = lkn;
    lkp->next  = lkn;
    lkn->prev  = lkp;    
    l->size++;
    return 1;
}

void *
gdl_link_get (gdl_link * lk)
{
    if (lk == NULL)
        return(NULL);
    return lk->data;
}

void
gdl_list_reverse (gdl_list * l)
{
    gdl_link * lk;
    gdl_link * lkprev;

    if (l == NULL)
        return;
    lkprev = l->state;
    for (lk = l->state->next; lk != l->state; lk = lk->next) {
        lkprev->next = lkprev->prev;
        lkprev->prev = lk;
        lkprev = lk;
    }
    /* Fix up the last node */
    lkprev->next = lkprev->prev;
    lkprev->prev = lk;
}

void
gdl_list_sort (gdl_list * l)
{
    gdl_list * ltmp;
    
    if (l == NULL)
        return;
    
    if(gdl_list_empty (l))
        return;
        
    gdl_list_quicksort (l, l->state->next, l->state->prev);
    
    return;
}

void
gdl_list_apply (gdl_list * l, gdl_data_apply apply, const void * user)
{
    gdl_link * lk;

    if ((l == NULL) || (apply == NULL))
        return;
    for(lk = l->state->next; lk != l->state; lk = lk->next) {
        if((apply(lk->data, user)) != GDL_SUCCESS)
                break;
    }
}

void *
gdl_list_get (gdl_list * l, size_t i)
{
	size_t ii;
	gdl_link * lk;

    if (l == NULL)
        return NULL;
    
    for(ii = 0,lk = l->state->next; lk != l->state; lk = lk->next, ii++)
    {
        if (ii == i)
        {
        	return lk->data;
        }
    }
    return NULL;
}

void
gdl_list_rapply (gdl_list * l, gdl_data_apply apply, const void * user)
{
    gdl_link * lk;

    if ((l == NULL) || (apply == NULL))
        return;
    for(lk = l->state->prev; lk != l->state; lk = lk->prev) {
        if((apply(lk->data, user)) != GDL_SUCCESS)
                break;
    }
}

void
gdl_list_merge (gdl_list * l1, gdl_list * l2)
{
    gdl_list_copy  (l1, l2);
    gdl_list_clear (l2);
}

gdl_list * 
gdl_list_clone (const gdl_list * old)
{
    gdl_list * cur;

    if (old == NULL)
        return(NULL);
    if (NULL == (cur = gdl_list_alloc(old->type)))
        return (NULL);
    if (0 != gdl_list_copy (cur, old))
        return NULL;
    return cur;
}

int
gdl_list_copy (gdl_list * cur, const gdl_list * old)
{
    gdl_link * lk;

    if ((old == NULL) || (cur == NULL))
        return(1);
    
    for(lk = old->state->next; lk != old->state; lk = lk->next)
    {
        if (0 != gdl_list_insert (cur, lk->data, lk->owner))
        {
            gdl_list_clear (cur);
            return (1);
        }
        if (lk->owner)
        {
        	lk->owner = 0;
        }
    }
    
    return (0);    
}

#include "factory.c"
#include "iterator.c"

const gdl_data_interface * gdl_list_default = &_gdl_list_type_default;
