/*  
 * 	list/test.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:45 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_test.h>
#include <gdl/gdl_list.h>

typedef struct data
{
	int i;
	double x;	
} test_data;

void
test_data_free (void * data)
{
	GDL_FREE (data);
}

int
test_data_compare (const void * data0, const void * data1)
{
	test_data * t0 = (test_data * ) data0;
	test_data * t1 = (test_data * ) data1;
	
	if (t0 == NULL || t1 == NULL)
		return -1;
	if (t0 == t1)
		return 0;
	if ((t0->i == t1->i) && (t0->x == t1->x))
		return 0;
	return 1;
}

int
test_data_apply (const void * data, const void * user)
{
	test_data * t = (test_data * ) data;
	
	t->x = t->x*t->x;
	
	return GDL_SUCCESS;
}

int
test_data_fwrite (FILE * stream, const void * data)
{
	size_t status;
	test_data * t = (test_data * ) data;
	
	status = fwrite (&t->i, sizeof (int), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&t->x, sizeof (double), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	
	return GDL_SUCCESS;
}

void *
test_data_fread (FILE * stream)
{
	size_t status;
	test_data * t = GDL_MALLOC (test_data, 1);
	
	status = fread (&t->i, sizeof (int), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&t->x, sizeof (double), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	
	return t;
}

static const gdl_data_interface test_data_type =
{
  &test_data_free,
  NULL,
  &test_data_compare,
  &test_data_fread,
  &test_data_fwrite
};

gdl_list *
test_list_alloc (void)
{
  gdl_list * list;
  
  list = gdl_list_alloc (&test_data_type);
  
  return list;
}

void
test_list_init (void)
{
  gdl_list * list = test_list_alloc();
  	
  gdl_test (list == 0,     "gdl_list_alloc test");
  
  gdl_list_free (list);
}

void
test_list_push_front (void)
{
	size_t i, status;
	gdl_list * list = test_list_alloc();
	test_data * data, * tmp;
	
	status = 1;
	for (i = 0; i < 100; i++)
	{
		data = GDL_MALLOC (test_data, 1);
		data->i = i;
		data->x = 2.0*i;
		
		gdl_list_push_front (list, data, 1);
		
		tmp = (test_data *) gdl_list_search (list, data);
		status &= test_data_compare (data, tmp);
	}
	
	gdl_test (gdl_list_size (list)  != 100, "gdl_list_push_front returns valid size");
	gdl_test (status == 1, "gdl_list_push_front returns valid data");
    
    gdl_list_free (list);
}

void
test_list_push_back (void)
{
	size_t i, status;
	gdl_list * list = test_list_alloc();
	test_data * data, * tmp;
	
	status = 1;
	for (i = 0; i < 100; i++)
	{
		data = GDL_MALLOC (test_data, 1);
		data->i = i;
		data->x = 2.0*i;
		gdl_list_push_back (list, data, 1);
		tmp = (test_data *) gdl_list_rsearch (list, data);
		status &= test_data_compare (data, tmp);
	}
	
	gdl_test (gdl_list_size (list)  != 100, "gdl_list_push_back returns valid size");
	gdl_test (status == 1, "gdl_list_push_back returns valid data");
    
    gdl_list_free (list);
}

void
test_list_insert (void)
{
	size_t i, status;
	gdl_list * list = test_list_alloc();
	test_data * data, * tmp;
	
	status = 1;
	for (i = 0; i < 100; i++)
	{
		data = GDL_MALLOC (test_data, 1);
		data->i = i;
		data->x = 2.0*i;
		gdl_list_insert (list, data, 1);
		tmp = (test_data *) gdl_list_search (list, data);
		status &= test_data_compare (data, tmp);
	}
	
	gdl_test (gdl_list_size (list)  != 100, "gdl_list_insert returns valid size");
	gdl_test (status == 1, "gdl_list_insert returns valid data");
    
    gdl_list_free (list);
}

void
test_list_append (void)
{
	size_t i, status;
	gdl_list * list = test_list_alloc();
	test_data * data, * tmp;
	
	status = 1;
	for (i = 0; i < 100; i++)
	{
		data = GDL_MALLOC (test_data, 1);
		data->i = i;
		data->x = 2.0*i;
		gdl_list_append (list, data, 1);
		tmp = (test_data *) gdl_list_rsearch (list, data);
		status &= test_data_compare (data, tmp);
	}
	
	gdl_test (gdl_list_size (list)  != 100, "gdl_list_append returns valid size");
	gdl_test (status == 1, "gdl_list_append returns valid data");
    
    gdl_list_free (list);
}

void
test_list_copy (void)
{
	size_t i;
	gdl_list * list1 = test_list_alloc();
	gdl_list * list2 = test_list_alloc();
	test_data * data;
	
	for (i = 0; i < 100; i++)
	{
		data = GDL_MALLOC (test_data, 1);
		data->i = i;
		data->x = 2.0*i;
		gdl_list_push_front (list1, data, 1);
	}
	
	gdl_list_copy (list2, list1);
	
	gdl_test (gdl_list_size (list1)  != gdl_list_size (list2), "gdl_list_copy seems to be ok");
	
	gdl_list_free (list1);
    gdl_list_free (list2);
}

void
test_list_apply (void)
{
	size_t i, status;
	gdl_list_itr * itr;
	gdl_list * list = test_list_alloc();
	test_data * data;
	
	status = 1;
	for (i = 0; i < 100; i++)
	{
		data = GDL_MALLOC (test_data, 1);
		data->i = i;
		data->x = 2.0*i;
		gdl_list_push_back (list, data, 1);
	}
	
	gdl_list_apply (list, &test_data_apply, NULL);
	
	status = 0;
	i      = 0;
	
	itr = gdl_list_iterator_front (list);
	
	do
	{
		data = (test_data *) gdl_list_iterator_value (itr);
		if (data->x != 4.0*i*i)
			status = 1;
		i++;
	} while (gdl_list_iterator_next (itr));
	
	gdl_test (status != 0, "gdl_list_apply seems to work fine");
	
	gdl_list_free (list);
}

void
test_list_fwrite (void)
{
	size_t i, status;
	FILE * stream;
	gdl_list * list = test_list_alloc();
	test_data * data;
	
	for (i = 0; i < 100; i++)
	{
		data = GDL_MALLOC (test_data, 1);
		data->i = i;
		data->x = 2.0*i;
		gdl_list_push_back (list, data, 1);
	}
	
	stream = gdl_fileopen ("test.dat", "w");
	
	status = gdl_list_fwrite (stream, list);
	
	gdl_fileclose ("test.dat", stream);
	
	gdl_test (status != GDL_SUCCESS, "gdl_list_fwrite returns ok");
}

void
test_list_fread (void)
{
	size_t i, status;
	FILE * stream;
	gdl_list_itr * itr;
	gdl_list * list = test_list_alloc();
	test_data * data;
	
	stream = gdl_fileopen ("test.dat", "r");
	
	status = gdl_list_fread (stream, list);
	
	gdl_fileclose ("test.dat", stream);
	
	gdl_test (status != GDL_SUCCESS, "gdl_list_fread returns ok");
	
	i = status = 0;
	
	itr = gdl_list_iterator_front (list);
	
	do
	{
		data = (test_data *) gdl_list_iterator_value (itr);
		
		if (data->i != i && data->x != 2.0*i)
			status = 1;		
		i++;
		
	} while (gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	gdl_test (status != 0, "gdl_list_fread returns correct data");
}

int
main (void)
{
	test_list_init ();
	test_list_push_front ();
	test_list_push_back ();
	test_list_insert ();
	test_list_append ();
	test_list_copy ();
	test_list_apply ();
	test_list_fwrite ();
	test_list_fread ();
	
	exit (gdl_test_summary());
}
