/* matrix/gdl_matrix_char.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_MATRIX_CHAR_H__
#define __GDL_MATRIX_CHAR_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_vector.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size1;
  size_t size2;
  size_t tda;
  char * data;
  gdl_block_char * block;
  int owner;
} gdl_matrix_char;

typedef struct
{
  gdl_matrix_char matrix;
} _gdl_matrix_char_view;

typedef _gdl_matrix_char_view gdl_matrix_char_view;

typedef struct
{
  gdl_matrix_char matrix;
} _gdl_matrix_char_const_view;

typedef const _gdl_matrix_char_const_view gdl_matrix_char_const_view;

/* Allocation */

gdl_matrix_char * 
gdl_matrix_char_alloc (const size_t n1, const size_t n2);

gdl_matrix_char * 
gdl_matrix_char_calloc (const size_t n1, const size_t n2);

gdl_matrix_char * 
gdl_matrix_char_alloc_from_block (gdl_block_char * b, 
                                   const size_t offset, 
                                   const size_t n1, 
                                   const size_t n2, 
                                   const size_t d2);

gdl_matrix_char * 
gdl_matrix_char_alloc_from_matrix (gdl_matrix_char * m,
                                    const size_t k1, 
                                    const size_t k2,
                                    const size_t n1, 
                                    const size_t n2);

gdl_vector_char * 
gdl_vector_char_alloc_row_from_matrix (gdl_matrix_char * m,
                                        const size_t i);

gdl_vector_char * 
gdl_vector_char_alloc_col_from_matrix (gdl_matrix_char * m,
                                        const size_t j);

void gdl_matrix_char_free (gdl_matrix_char * m);

/* Views */

_gdl_matrix_char_view 
gdl_matrix_char_submatrix (gdl_matrix_char * m, 
                            const size_t i, const size_t j, 
                            const size_t n1, const size_t n2);

_gdl_vector_char_view 
gdl_matrix_char_row (gdl_matrix_char * m, const size_t i);

_gdl_vector_char_view 
gdl_matrix_char_column (gdl_matrix_char * m, const size_t j);

_gdl_vector_char_view 
gdl_matrix_char_diagonal (gdl_matrix_char * m);

_gdl_vector_char_view 
gdl_matrix_char_subdiagonal (gdl_matrix_char * m, const size_t k);

_gdl_vector_char_view 
gdl_matrix_char_superdiagonal (gdl_matrix_char * m, const size_t k);

_gdl_matrix_char_view
gdl_matrix_char_view_array (char * base,
                             const size_t n1, 
                             const size_t n2);

_gdl_matrix_char_view
gdl_matrix_char_view_array_with_tda (char * base, 
                                      const size_t n1, 
                                      const size_t n2,
                                      const size_t tda);


_gdl_matrix_char_view
gdl_matrix_char_view_vector (gdl_vector_char * v,
                              const size_t n1, 
                              const size_t n2);

_gdl_matrix_char_view
gdl_matrix_char_view_vector_with_tda (gdl_vector_char * v,
                                       const size_t n1, 
                                       const size_t n2,
                                       const size_t tda);


_gdl_matrix_char_const_view 
gdl_matrix_char_const_submatrix (const gdl_matrix_char * m, 
                                  const size_t i, const size_t j, 
                                  const size_t n1, const size_t n2);

_gdl_vector_char_const_view 
gdl_matrix_char_const_row (const gdl_matrix_char * m, 
                            const size_t i);

_gdl_vector_char_const_view 
gdl_matrix_char_const_column (const gdl_matrix_char * m, 
                               const size_t j);

_gdl_vector_char_const_view
gdl_matrix_char_const_diagonal (const gdl_matrix_char * m);

_gdl_vector_char_const_view 
gdl_matrix_char_const_subdiagonal (const gdl_matrix_char * m, 
                                    const size_t k);

_gdl_vector_char_const_view 
gdl_matrix_char_const_superdiagonal (const gdl_matrix_char * m, 
                                      const size_t k);

_gdl_matrix_char_const_view
gdl_matrix_char_const_view_array (const char * base,
                                   const size_t n1, 
                                   const size_t n2);

_gdl_matrix_char_const_view
gdl_matrix_char_const_view_array_with_tda (const char * base, 
                                            const size_t n1, 
                                            const size_t n2,
                                            const size_t tda);

_gdl_matrix_char_const_view
gdl_matrix_char_const_view_vector (const gdl_vector_char * v,
                                    const size_t n1, 
                                    const size_t n2);

_gdl_matrix_char_const_view
gdl_matrix_char_const_view_vector_with_tda (const gdl_vector_char * v,
                                             const size_t n1, 
                                             const size_t n2,
                                             const size_t tda);

/* Operations */

char   gdl_matrix_char_get(const gdl_matrix_char * m, const size_t i, const size_t j);
void    gdl_matrix_char_set(gdl_matrix_char * m, const size_t i, const size_t j, const char x);

char * gdl_matrix_char_ptr(gdl_matrix_char * m, const size_t i, const size_t j);
const char * gdl_matrix_char_const_ptr(const gdl_matrix_char * m, const size_t i, const size_t j);

void gdl_matrix_char_set_zero (gdl_matrix_char * m);
void gdl_matrix_char_set_identity (gdl_matrix_char * m);
void gdl_matrix_char_set_all (gdl_matrix_char * m, char x);

gdl_matrix_char * gdl_matrix_char_fread (FILE * stream) ;
int gdl_matrix_char_fwrite (FILE * stream, const gdl_matrix_char * m) ;
int gdl_matrix_char_fscanf (FILE * stream, gdl_matrix_char * m);
int gdl_matrix_char_fprintf (FILE * stream, const gdl_matrix_char * m, const char * format);
 
int gdl_matrix_char_memcpy(gdl_matrix_char * dest, const gdl_matrix_char * src);
int gdl_matrix_char_swap(gdl_matrix_char * m1, gdl_matrix_char * m2);

int gdl_matrix_char_swap_rows(gdl_matrix_char * m, const size_t i, const size_t j);
int gdl_matrix_char_swap_columns(gdl_matrix_char * m, const size_t i, const size_t j);
int gdl_matrix_char_swap_rowcol(gdl_matrix_char * m, const size_t i, const size_t j);
int gdl_matrix_char_transpose (gdl_matrix_char * m);
int gdl_matrix_char_transpose_memcpy (gdl_matrix_char * dest, const gdl_matrix_char * src);

char gdl_matrix_char_max (const gdl_matrix_char * m);
char gdl_matrix_char_min (const gdl_matrix_char * m);
void gdl_matrix_char_minmax (const gdl_matrix_char * m, char * min_out, char * max_out);

void gdl_matrix_char_max_index (const gdl_matrix_char * m, size_t * imax, size_t *jmax);
void gdl_matrix_char_min_index (const gdl_matrix_char * m, size_t * imin, size_t *jmin);
void gdl_matrix_char_minmax_index (const gdl_matrix_char * m, size_t * imin, size_t * jmin, size_t * imax, size_t * jmax);

int gdl_matrix_char_isnull (const gdl_matrix_char * m);

int gdl_matrix_char_add (gdl_matrix_char * a, const gdl_matrix_char * b);
int gdl_matrix_char_sub (gdl_matrix_char * a, const gdl_matrix_char * b);
int gdl_matrix_char_mul_elements (gdl_matrix_char * a, const gdl_matrix_char * b);
int gdl_matrix_char_div_elements (gdl_matrix_char * a, const gdl_matrix_char * b);
int gdl_matrix_char_scale (gdl_matrix_char * a, const double x);
int gdl_matrix_char_add_constant (gdl_matrix_char * a, const double x);
int gdl_matrix_char_add_diagonal (gdl_matrix_char * a, const double x);

/***********************************************************************/
/* The functions below are obsolete                                    */
/***********************************************************************/
int gdl_matrix_char_get_row(gdl_vector_char * v, const gdl_matrix_char * m, const size_t i);
int gdl_matrix_char_get_col(gdl_vector_char * v, const gdl_matrix_char * m, const size_t j);
int gdl_matrix_char_set_row(gdl_matrix_char * m, const size_t i, const gdl_vector_char * v);
int gdl_matrix_char_set_col(gdl_matrix_char * m, const size_t j, const gdl_vector_char * v);

/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline 
char
gdl_matrix_char_get(const gdl_matrix_char * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VAL("first index out of range", GDL_EINVAL, 0) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VAL("second index out of range", GDL_EINVAL, 0) ;
    }
#endif
  return m->data[i * m->tda + j] ;
} 

extern inline 
void
gdl_matrix_char_set(gdl_matrix_char * m, const size_t i, const size_t j, const char x)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VOID("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VOID("second index out of range", GDL_EINVAL) ;
    }
#endif
  m->data[i * m->tda + j] = x ;
}

extern inline 
char *
gdl_matrix_char_ptr(gdl_matrix_char * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (char *) (m->data + (i * m->tda + j)) ;
} 

extern inline 
const char *
gdl_matrix_char_const_ptr(const gdl_matrix_char * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (const char *) (m->data + (i * m->tda + j)) ;
} 

#endif

__END_DECLS

#endif /* __GDL_MATRIX_CHAR_H__ */
