/* matrix/gdl_matrix_double.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_MATRIX_DOUBLE_H__
#define __GDL_MATRIX_DOUBLE_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_vector.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size1;
  size_t size2;
  size_t tda;
  double * data;
  gdl_block * block;
  int owner;
} gdl_matrix;

typedef struct
{
  gdl_matrix matrix;
} _gdl_matrix_view;

typedef _gdl_matrix_view gdl_matrix_view;

typedef struct
{
  gdl_matrix matrix;
} _gdl_matrix_const_view;

typedef const _gdl_matrix_const_view gdl_matrix_const_view;

/* Allocation */

gdl_matrix * 
gdl_matrix_alloc (const size_t n1, const size_t n2);

gdl_matrix * 
gdl_matrix_calloc (const size_t n1, const size_t n2);

gdl_matrix * 
gdl_matrix_alloc_from_block (gdl_block * b, 
                                   const size_t offset, 
                                   const size_t n1, 
                                   const size_t n2, 
                                   const size_t d2);

gdl_matrix * 
gdl_matrix_alloc_from_matrix (gdl_matrix * m,
                                    const size_t k1, 
                                    const size_t k2,
                                    const size_t n1, 
                                    const size_t n2);

gdl_vector * 
gdl_vector_alloc_row_from_matrix (gdl_matrix * m,
                                        const size_t i);

gdl_vector * 
gdl_vector_alloc_col_from_matrix (gdl_matrix * m,
                                        const size_t j);

void gdl_matrix_free (gdl_matrix * m);

/* Views */

_gdl_matrix_view 
gdl_matrix_submatrix (gdl_matrix * m, 
                            const size_t i, const size_t j, 
                            const size_t n1, const size_t n2);

_gdl_vector_view 
gdl_matrix_row (gdl_matrix * m, const size_t i);

_gdl_vector_view 
gdl_matrix_column (gdl_matrix * m, const size_t j);

_gdl_vector_view 
gdl_matrix_diagonal (gdl_matrix * m);

_gdl_vector_view 
gdl_matrix_subdiagonal (gdl_matrix * m, const size_t k);

_gdl_vector_view 
gdl_matrix_superdiagonal (gdl_matrix * m, const size_t k);

_gdl_matrix_view
gdl_matrix_view_array (double * base,
                             const size_t n1, 
                             const size_t n2);

_gdl_matrix_view
gdl_matrix_view_array_with_tda (double * base, 
                                      const size_t n1, 
                                      const size_t n2,
                                      const size_t tda);


_gdl_matrix_view
gdl_matrix_view_vector (gdl_vector * v,
                              const size_t n1, 
                              const size_t n2);

_gdl_matrix_view
gdl_matrix_view_vector_with_tda (gdl_vector * v,
                                       const size_t n1, 
                                       const size_t n2,
                                       const size_t tda);


_gdl_matrix_const_view 
gdl_matrix_const_submatrix (const gdl_matrix * m, 
                                  const size_t i, const size_t j, 
                                  const size_t n1, const size_t n2);

_gdl_vector_const_view 
gdl_matrix_const_row (const gdl_matrix * m, 
                            const size_t i);

_gdl_vector_const_view 
gdl_matrix_const_column (const gdl_matrix * m, 
                               const size_t j);

_gdl_vector_const_view
gdl_matrix_const_diagonal (const gdl_matrix * m);

_gdl_vector_const_view 
gdl_matrix_const_subdiagonal (const gdl_matrix * m, 
                                    const size_t k);

_gdl_vector_const_view 
gdl_matrix_const_superdiagonal (const gdl_matrix * m, 
                                      const size_t k);

_gdl_matrix_const_view
gdl_matrix_const_view_array (const double * base,
                                   const size_t n1, 
                                   const size_t n2);

_gdl_matrix_const_view
gdl_matrix_const_view_array_with_tda (const double * base, 
                                            const size_t n1, 
                                            const size_t n2,
                                            const size_t tda);

_gdl_matrix_const_view
gdl_matrix_const_view_vector (const gdl_vector * v,
                                    const size_t n1, 
                                    const size_t n2);

_gdl_matrix_const_view
gdl_matrix_const_view_vector_with_tda (const gdl_vector * v,
                                             const size_t n1, 
                                             const size_t n2,
                                             const size_t tda);

/* Operations */

double   gdl_matrix_get(const gdl_matrix * m, const size_t i, const size_t j);
void    gdl_matrix_set(gdl_matrix * m, const size_t i, const size_t j, const double x);

double * gdl_matrix_ptr(gdl_matrix * m, const size_t i, const size_t j);
const double * gdl_matrix_const_ptr(const gdl_matrix * m, const size_t i, const size_t j);

void gdl_matrix_set_zero (gdl_matrix * m);
void gdl_matrix_set_identity (gdl_matrix * m);
void gdl_matrix_set_all (gdl_matrix * m, double x);

gdl_matrix * gdl_matrix_fread (FILE * stream) ;
int gdl_matrix_fwrite (FILE * stream, const gdl_matrix * m) ;
int gdl_matrix_fscanf (FILE * stream, gdl_matrix * m);
int gdl_matrix_fprintf (FILE * stream, const gdl_matrix * m, const char * format);
 
int gdl_matrix_memcpy(gdl_matrix * dest, const gdl_matrix * src);
int gdl_matrix_swap(gdl_matrix * m1, gdl_matrix * m2);

int gdl_matrix_swap_rows(gdl_matrix * m, const size_t i, const size_t j);
int gdl_matrix_swap_columns(gdl_matrix * m, const size_t i, const size_t j);
int gdl_matrix_swap_rowcol(gdl_matrix * m, const size_t i, const size_t j);
int gdl_matrix_transpose (gdl_matrix * m);
int gdl_matrix_transpose_memcpy (gdl_matrix * dest, const gdl_matrix * src);

double gdl_matrix_max (const gdl_matrix * m);
double gdl_matrix_min (const gdl_matrix * m);
void gdl_matrix_minmax (const gdl_matrix * m, double * min_out, double * max_out);

void gdl_matrix_max_index (const gdl_matrix * m, size_t * imax, size_t *jmax);
void gdl_matrix_min_index (const gdl_matrix * m, size_t * imin, size_t *jmin);
void gdl_matrix_minmax_index (const gdl_matrix * m, size_t * imin, size_t * jmin, size_t * imax, size_t * jmax);

int gdl_matrix_isnull (const gdl_matrix * m);

int gdl_matrix_add (gdl_matrix * a, const gdl_matrix * b);
int gdl_matrix_sub (gdl_matrix * a, const gdl_matrix * b);
int gdl_matrix_mul_elements (gdl_matrix * a, const gdl_matrix * b);
int gdl_matrix_div_elements (gdl_matrix * a, const gdl_matrix * b);
int gdl_matrix_scale (gdl_matrix * a, const double x);
int gdl_matrix_add_constant (gdl_matrix * a, const double x);
int gdl_matrix_add_diagonal (gdl_matrix * a, const double x);
double  gdl_matrix_sum_all (const gdl_matrix * a);

/***********************************************************************/
/* The functions below are obsolete                                    */
/***********************************************************************/
int gdl_matrix_get_row(gdl_vector * v, const gdl_matrix * m, const size_t i);
int gdl_matrix_get_col(gdl_vector * v, const gdl_matrix * m, const size_t j);
int gdl_matrix_set_row(gdl_matrix * m, const size_t i, const gdl_vector * v);
int gdl_matrix_set_col(gdl_matrix * m, const size_t j, const gdl_vector * v);

/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline 
double
gdl_matrix_get(const gdl_matrix * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VAL("first index out of range", GDL_EINVAL, 0) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VAL("second index out of range", GDL_EINVAL, 0) ;
    }
#endif
  return m->data[i * m->tda + j] ;
} 

extern inline 
void
gdl_matrix_set(gdl_matrix * m, const size_t i, const size_t j, const double x)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_VOID("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_VOID("second index out of range", GDL_EINVAL) ;
    }
#endif
  m->data[i * m->tda + j] = x ;
}

extern inline 
double *
gdl_matrix_ptr(gdl_matrix * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (double *) (m->data + (i * m->tda + j)) ;
} 

extern inline 
const double *
gdl_matrix_const_ptr(const gdl_matrix * m, const size_t i, const size_t j)
{
#if GDL_RANGE_CHECK
  if (i >= m->size1)
    {
      GDL_ERROR_NULL("first index out of range", GDL_EINVAL) ;
    }
  else if (j >= m->size2)
    {
      GDL_ERROR_NULL("second index out of range", GDL_EINVAL) ;
    }
#endif
  return (const double *) (m->data + (i * m->tda + j)) ;
} 

#endif

__END_DECLS

#endif /* __GDL_MATRIX_DOUBLE_H__ */
