/* matrix/view_source.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2001 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

QUALIFIED_VIEW (_gdl_matrix,view)
FUNCTION (gdl_matrix, view_array) (QUALIFIER ATOMIC * array, 
                                   const size_t n1, const size_t n2)
{
  QUALIFIED_VIEW (_gdl_matrix,view) view = NULL_MATRIX_VIEW;

  if (n1 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n1 must be positive integer",
                     GDL_EINVAL, view);
    }
  else if (n2 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n2 must be positive integer",
                     GDL_EINVAL, view);
    }

  {
    TYPE(gdl_matrix) m = NULL_MATRIX;

    m.data = (ATOMIC *)array;
    m.size1 = n1;
    m.size2 = n2;
    m.tda = n2; 
    m.block = 0;
    m.owner = 0;

    ((VIEW(_gdl_matrix, view) *)&view)->matrix = m;    
    return view;
  }
}

QUALIFIED_VIEW (_gdl_matrix,view)
FUNCTION(gdl_matrix, view_array_with_tda) (QUALIFIER ATOMIC * base,
                                           const size_t n1, 
                                           const size_t n2,
                                           const size_t tda)
{
  QUALIFIED_VIEW (_gdl_matrix,view) view = NULL_MATRIX_VIEW;

  if (n1 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n1 must be positive integer",
                     GDL_EINVAL, view);
    }
  else if (n2 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n2 must be positive integer",
                     GDL_EINVAL, view);
    }
  else if (n2 > tda)
    {
      GDL_ERROR_VAL ("matrix dimension n2 must not exceed tda",
                     GDL_EINVAL, view);
    }


  {
    TYPE(gdl_matrix) m = NULL_MATRIX;

    m.data = (ATOMIC *)base;
    m.size1 = n1;
    m.size2 = n2;
    m.tda = tda;
    m.block = 0;
    m.owner = 0;

    ((VIEW(_gdl_matrix, view) *)&view)->matrix = m;    
    return view;
  }
}

QUALIFIED_VIEW (_gdl_matrix,view)
FUNCTION(gdl_matrix, view_vector) (QUALIFIED_TYPE(gdl_vector) * v,
                                   const size_t n1, 
                                   const size_t n2)
{
  QUALIFIED_VIEW (_gdl_matrix,view) view = NULL_MATRIX_VIEW;

  if (n1 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n1 must be positive integer",
                     GDL_EINVAL, view);
    }
  else if (n2 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n2 must be positive integer",
                     GDL_EINVAL, view);
    }
  else if (v->stride != 1) 
    {
      GDL_ERROR_VAL ("vector must have unit stride",
                     GDL_EINVAL, view);
    }
  else if (n1 * n2 > v->size)
    {
      GDL_ERROR_VAL ("matrix size exceeds size of original", 
                     GDL_EINVAL, view);
    }

  {
    TYPE(gdl_matrix) m = NULL_MATRIX;

    m.data = v->data;
    m.size1 = n1;
    m.size2 = n2;
    m.tda = n2;
    m.block = v->block;
    m.owner = 0;

    ((VIEW(_gdl_matrix, view) *)&view)->matrix = m;    
    return view;
  }
}


QUALIFIED_VIEW (_gdl_matrix,view)
FUNCTION(gdl_matrix, view_vector_with_tda) (QUALIFIED_TYPE(gdl_vector) * v,
                                            const size_t n1, 
                                            const size_t n2,
                                            const size_t tda)
{
  QUALIFIED_VIEW (_gdl_matrix,view) view = NULL_MATRIX_VIEW;

  if (n1 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n1 must be positive integer",
                     GDL_EINVAL, view);
    }
  else if (n2 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n2 must be positive integer",
                     GDL_EINVAL, view);
    }
  else if (v->stride != 1) 
    {
      GDL_ERROR_VAL ("vector must have unit stride",
                     GDL_EINVAL, view);
    }
  else if (n2 > tda)
    {
      GDL_ERROR_VAL ("matrix dimension n2 must not exceed tda",
                     GDL_EINVAL, view);
    }
  else if (n1 * tda > v->size)
    {
      GDL_ERROR_VAL ("matrix size exceeds size of original", 
                     GDL_EINVAL, view);
    }

  {
    TYPE(gdl_matrix) m = NULL_MATRIX;

    m.data = v->data;
    m.size1 = n1;
    m.size2 = n2;
    m.tda = tda;
    m.block = v->block;
    m.owner = 0;

    ((VIEW(_gdl_matrix, view) *)&view)->matrix = m;    
    return view;
  }
}

#ifdef JUNK
int
FUNCTION (gdl_matrix, view_from_matrix) (TYPE(gdl_matrix) * m, 
                                         TYPE(gdl_matrix) * mm, 
                                         const size_t k1,
                                         const size_t k2,
                                         const size_t n1, 
                                         const size_t n2)
{
  if (n1 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n1 must be positive integer",
                        GDL_EINVAL, 0);
    }
  else if (n2 == 0)
    {
      GDL_ERROR_VAL ("matrix dimension n2 must be positive integer",
                        GDL_EINVAL, 0);
    }
  else if (k1 + n1 > mm->size1)
    {
      GDL_ERROR_VAL ("submatrix dimension 1 exceeds size of original",
                        GDL_EINVAL, 0);
    }
  else if (k2 + n2 > mm->size2)
    {
      GDL_ERROR_VAL ("submatrix dimension 2 exceeds size of original",
                        GDL_EINVAL, 0);
    }

  m->data = mm->data + k1 * mm->tda + k2 ;
  m->size1 = n1;
  m->size2 = n2;
  m->tda = mm->tda;
  m->block = mm->block;
  m->owner = 0;

  return GDL_SUCCESS;
}

int
FUNCTION (gdl_vector, view_row_from_matrix) (TYPE(gdl_vector) * v,
                                             TYPE(gdl_matrix) * m,
                                             const size_t i)
{
  const size_t column_length = m->size1;

  if (i >= column_length)
    {
      GDL_ERROR ("row index is out of range", GDL_EINVAL);
    }

  if (v->block != 0)
    {
      GDL_ERROR ("vector already has memory allocated to it", GDL_ENOMEM);
    }

  v->data = m->data + MULTIPLICITY * i * m->tda ;
  v->size = m->size2;
  v->stride = 1;

  return GDL_SUCCESS;
}

int
FUNCTION (gdl_vector, view_col_from_matrix) (TYPE(gdl_vector) * v,
                                             TYPE(gdl_matrix) * m,
                                             const size_t j)
{
  const size_t row_length = m->size2;

  if (j >= row_length)
    {
      GDL_ERROR_VAL ("column index is out of range", GDL_EINVAL, 0);
    }

  if (v->block != 0)
    {
      GDL_ERROR ("vector already has memory allocated to it", GDL_ENOMEM);
    }

  v->data = m->data + MULTIPLICITY * j ;
  v->size = m->size1;
  v->stride = m->tda;

  return GDL_SUCCESS;
}
#endif /* JUNK */

