/* min/fsolver.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <config.h>
#include <stdlib.h>
#include <string.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_min.h>

#include "min.h"

static int 
compute_f_values (gdl_function * f, double x_minimum, double * f_minimum,
                  double x_lower, double * f_lower, 
                  double x_upper, double * f_upper);


static int 
compute_f_values (gdl_function * f, double x_minimum, double * f_minimum,
                  double x_lower, double * f_lower,
                  double x_upper, double * f_upper)
{
  SAFE_FUNC_CALL(f, x_lower, f_lower);
  SAFE_FUNC_CALL(f, x_upper, f_upper);
  SAFE_FUNC_CALL(f, x_minimum, f_minimum);

  return GDL_SUCCESS;
}

int
gdl_min_fminimizer_set (gdl_min_fminimizer * s, 
                        gdl_function * f, 
                        double x_minimum, double x_lower, double x_upper)
{
  int status ;

  double f_minimum, f_lower, f_upper;

  status = compute_f_values (f, x_minimum, &f_minimum, 
                             x_lower, &f_lower,  
                             x_upper, &f_upper);

  if (status != GDL_SUCCESS)
    {
      return status ;
    }
  
  status = gdl_min_fminimizer_set_with_values (s, f, x_minimum, f_minimum, 
                                               x_lower, f_lower,
                                               x_upper, f_upper);
  return status;
}

gdl_min_fminimizer *
gdl_min_fminimizer_alloc (const gdl_min_fminimizer_type * T) 
{
  gdl_min_fminimizer * s = 
    (gdl_min_fminimizer *) malloc (sizeof (gdl_min_fminimizer));

  if (s == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for minimizer struct",
                        GDL_ENOMEM, 0);
    };

  s->state = malloc (T->size);

  if (s->state == 0)
    {
      free (s);         /* exception in constructor, avoid memory leak */

      GDL_ERROR_VAL ("failed to allocate space for minimizer state",
                        GDL_ENOMEM, 0);
    };

  s->type = T ;
  s->function = NULL;

  return s;
}

int
gdl_min_fminimizer_set_with_values (gdl_min_fminimizer * s, gdl_function * f, 
                                    double x_minimum, double f_minimum, 
                                    double x_lower, double f_lower,
                                    double x_upper, double f_upper)
{
  s->function = f;
  s->x_minimum = x_minimum;
  s->x_lower = x_lower;
  s->x_upper = x_upper;

  if (x_lower > x_upper)
    {
      GDL_ERROR ("invalid interval (lower > upper)", GDL_EINVAL);
    }

  if (x_minimum >= x_upper || x_minimum <= x_lower) 
    {
      GDL_ERROR ("x_minimum must lie inside interval (lower < x < upper)",
                 GDL_EINVAL);
    }

  s->f_lower = f_lower;
  s->f_upper = f_upper;
  s->f_minimum = f_minimum;

  if (f_minimum >= f_lower || f_minimum >= f_upper)
    {
      GDL_ERROR ("endpoints do not enclose a minimum", GDL_EINVAL);
    }

  return (s->type->set) (s->state, s->function, 
                         x_minimum, f_minimum, 
                         x_lower, f_lower,
                         x_upper, f_upper);
}


int
gdl_min_fminimizer_iterate (gdl_min_fminimizer * s)
{
  return (s->type->iterate) (s->state, s->function, 
                             &(s->x_minimum), &(s->f_minimum),
                             &(s->x_lower), &(s->f_lower), 
                             &(s->x_upper), &(s->f_upper));
}

void
gdl_min_fminimizer_free (gdl_min_fminimizer * s)
{
  free (s->state);
  free (s);
}

const char *
gdl_min_fminimizer_name (const gdl_min_fminimizer * s)
{
  return s->type->name;
}

/* Deprecated, use x_minimum instead */
double
gdl_min_fminimizer_minimum (const gdl_min_fminimizer * s)
{
  return s->x_minimum;
}

double
gdl_min_fminimizer_x_minimum (const gdl_min_fminimizer * s)
{
  return s->x_minimum;
}

double
gdl_min_fminimizer_x_lower (const gdl_min_fminimizer * s)
{
  return s->x_lower;
}

double
gdl_min_fminimizer_x_upper (const gdl_min_fminimizer * s)
{
  return s->x_upper;
}

double
gdl_min_fminimizer_f_minimum (const gdl_min_fminimizer * s)
{
  return s->f_minimum;
}

double
gdl_min_fminimizer_f_lower (const gdl_min_fminimizer * s)
{
  return s->f_lower;
}

double
gdl_min_fminimizer_f_upper (const gdl_min_fminimizer * s)
{
  return s->f_upper;
}

