/*  
 *  mosaic/data.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_mosaic.h>

gdl_mosaic_data *
gdl_mosaic_data_alloc (const gdl_chromosome * chrom, gdl_gview_wrapper * data)
{	
	size_t i;
	gdl_mosaic_data * d;
	
	d = GDL_CALLOC (gdl_mosaic_data, 1);
	
	d->chrom = chrom;
	
	d->N    = gdl_gview_wrapper_accession_size (data);
	d->NC   = gdl_gview_wrapper_accession_size_c (data);
	d->L    = gdl_gview_wrapper_locus_size (data);
	d->P    = gdl_gview_wrapper_ploidy (data);
	d->seq  = gdl_gview_wrapper_allele_block_c (data, 0);
	d->gbuf = gdl_gview_wrapper_get_new (data);
	d->data = data;
	d->NA   = GDL_MALLOC (size_t, d->L);
	d->D    = GDL_MALLOC (double, d->L-1);
	d->W    = GDL_MALLOC (double, d->NC);
	 
	for (i = 0; i < d->L; i++)
	{
		if (i)
		{
		   gdl_gdistance * di = gdl_chromosome_get_distance (chrom, i-1, i);
			if (gdl_gdistance_is_defined (di))
			{
				d->D[i-1] = di->value;
			}
			else
			{
				d->D[i-1] = -1.0;
			}
			gdl_gdistance_free (di);
		}
		d->NA[i] = gdl_locus_allele (gdl_gview_wrapper_get_locus (data, i));
	}
	
	for (i = 0; i < d->NC; i++)
	{
		d->W[i] = gdl_gview_wrapper_accession_mult_c (data, i);
		d->W[i] /= (d->N*d->P);
	}
	
	return d;
}

void
gdl_mosaic_data_free (gdl_mosaic_data * d)
{
	if (d)
	{
		gdl_allele_block_free (d->seq);
		gdl_gvalues_get_free (d->gbuf);
		GDL_FREE (d->NA);
		GDL_FREE (d->D);
		GDL_FREE (d->W);
		GDL_FREE (d);
	}	
}

gdl_boolean
gdl_mosaic_data_is_missing (const gdl_mosaic_data * d, size_t c, size_t l)
{
	return (gdl_allele_block_get (d->seq, c, l, 0)==0) ? gdl_true : gdl_false;
}

size_t
gdl_mosaic_data_get_allele (const gdl_mosaic_data * d, size_t c, size_t l, size_t p)
{
	return gdl_allele_block_get (d->seq, c, l, p)-1;
}

double
gdl_mosaic_data_distance (const gdl_mosaic_data * d, size_t i1, size_t i2)
{
	if (i1 == i2) return 0;
	size_t i;
	double x=0;
	for (i = i1; i < i2; i++)
	{
		x+=d->D[i];
	}
	return x;
}

