/*  
 *  mosaic/result.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_collector.h>
#include <gdl/gdl_mosaic.h>
#include <gdl/gdl_mosaic_result.h>

gdl_mosaic_result *
gdl_mosaic_result_alloc (gdl_mosaic * m)
{
	size_t i, j, k;
	gdl_mosaic_result * r;
	
	r = GDL_CALLOC (gdl_mosaic_result, 1);
	
	r->N = m->data->N;
	r->L = m->data->L;
	r->P = m->data->P;
	r->K = m->clust->K;
	r->log = m->log;
	
	r->NA = GDL_MALLOC (size_t, r->L);
	r->D  = GDL_MALLOC (double, r->L-1);
	
	memcpy (r->NA, m->data->NA, sizeof(size_t)*r->L);
	memcpy (r->D, m->data->D, sizeof(double)*(r->L-1));
	
	r->rho = GDL_MALLOC (double, r->L-1);
	r->mu  = GDL_MALLOC (double, r->L);
	r->f   = GDL_MATRIX_ALLOC (double, r->L, r->K);
	
	memcpy (r->rho, m->param->rho, sizeof(double)*(r->L-1));
	memcpy (r->mu, m->param->mu, sizeof(double)*r->L);
	for (i = 0; i < r->L; i++)
	{
		memcpy (r->f[i], m->param->f[i], sizeof(double)*r->K);
	}
	
	r->haplo   = gdl_allele_block_clone (m->clust->haplo);
	r->missing = gdl_gview_collector_clone (gdl_gview_wrapper_missing_sites (m->data->data));
	r->labels  = gdl_glabels_wrapper_alloc (m->data->data, gdl_false, gdl_true);
	r->clust   = gdl_clustering_clone (gdl_gview_wrapper_clustering (m->data->data));
	r->chrom   = gdl_entity_clone (m->data->chrom);
	r->viterbi = gdl_allele_block_alloc (gdl_clustering_nclust (r->clust), r->L, r->P);
	
	for (i = 0; i < gdl_clustering_nclust (r->clust); i++)
	{
		for (j = 0; j < r->P; j++)
		{
			gdl_vector_uint * path = gdl_mosaic_hmm_viterbi (m, i, j, &gdl_mosaic_hmm_obs_proba_haplo);
			for (k = 0; k < path->size; k++)
			{
				gdl_allele_block_set (r->viterbi, i, k, j, gdl_vector_uint_get (path, k));
			}
			gdl_vector_uint_free (path);
		}
	}
	
	return r;
}

void
gdl_mosaic_result_free (gdl_mosaic_result * r)
{
	if (r)
	{
		GDL_FREE (r->NA);
		GDL_FREE (r->D);
		GDL_FREE (r->rho);
		GDL_FREE (r->mu);
		GDL_MATRIX_FREE (r->f, r->L);
		gdl_allele_block_free (r->haplo);
		gdl_gview_collector_free (r->missing);
		gdl_glabels_free (r->labels);
		gdl_clustering_free (r->clust);
		gdl_chromosome_free (r->chrom);
		gdl_allele_block_free (r->viterbi);
		GDL_FREE (r);
	}	
}

size_t *
gdl_mosaic_result_sort_accession (const gdl_mosaic_result * r)
{
	size_t i, * ii, j, k, maxk, nk, na, np, ng, nt, tot, * idx, * sidx, * tidx, * clust, ** prop;
	double x, * q;
	gdl_list     * group;
	gdl_list_itr * itr;
	
	nt = 0;
	tidx  = GDL_MALLOC (size_t, r->N);
	clust = GDL_MALLOC (size_t, r->N);
	prop  = GDL_MATRIX_ALLOC (size_t, r->N, r->K);
	
	for (i = 0; i < r->N; i++)
	{
		for (j = 0; j < r->L; j++)
		{
			for (k = 0; k < r->P; k++)
			{
				prop[i][gdl_allele_block_get (r->viterbi, gdl_clustering_cluster (r->clust, i), j, k)]++;
			}
		}
		for (tot = k = 0; k < r->K; k++)
		{
			if (prop[i][k] > tot)
			{
				tot  = prop[i][k];
				maxk = k;
			}
		}
		clust[i] = maxk;
	}
	
	for (j = 0; j < r->K; j++)
	{
		group = gdl_list_alloc (gdl_interface_uint);
		for (i = 0; i < r->N; i++)
		{
			if (clust[i]==j)
			{
				ii = GDL_MALLOC (size_t, 1);
				*ii = i;
				gdl_list_push_back (group, ii, 1);
			}
		}
		if ((ng=gdl_list_size (group))!=0)
		{
			q    = GDL_MALLOC (double, ng);
			idx  = GDL_MALLOC (size_t, ng);
			sidx = GDL_MALLOC (size_t, ng);
			
			i   = 0;
			itr = gdl_list_iterator_front (group);
			do
			{
				ii     = (size_t *) gdl_list_iterator_value (itr);
				idx[i] = *ii;
				q[i]   = prop[idx[i]][j];
				i++;
			}
			while (gdl_list_iterator_next (itr));
			
			gdl_list_iterator_free (itr);
			
			gdl_sort_index (sidx, q, 1, ng);
			
			for (i = 0; i < ng/2; i++)
			{
				k = sidx[i];
				sidx[i]      = idx[sidx[ng-1-i]];
				sidx[ng-1-i] = idx[k];
			}
			if (ng % 2 == 1)
			{
				sidx[ng/2] = idx[sidx[ng/2]];	
			}
			
			memcpy (&tidx[nt], sidx, sizeof (size_t)*ng);
			
			nt += ng;

			GDL_FREE (q);
			GDL_FREE (idx);
			GDL_FREE (sidx);
		}
		
		gdl_list_free (group);
	}
	
	GDL_MATRIX_FREE (prop, r->N);
	GDL_FREE (clust);
	
	return tidx;
}

gdl_mosaic_result *
gdl_mosaic_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_mosaic_result * r;
		
		r = GDL_CALLOC (gdl_mosaic_result, 1);
		
		status = fread (&r->N, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&r->L, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&r->P, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&r->K, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&r->log, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r->NA  = GDL_MALLOC (size_t, r->L);
		r->D   = GDL_MALLOC (double, r->L-1);
		r->rho = GDL_MALLOC (double, r->L-1);
		r->mu  = GDL_MALLOC (double, r->L);
		r->f   = GDL_MATRIX_ALLOC (double, r->L, r->K);
		
		status = fread (r->NA, sizeof(size_t), r->L, stream);
		GDL_FREAD_STATUS (status, r->L);
		status = fread (r->D, sizeof(double), r->L-1, stream);
		GDL_FREAD_STATUS (status, r->L-1);
		status = fread (r->rho, sizeof(double), r->L-1, stream);
		GDL_FREAD_STATUS (status, r->L-1);
		status = fread (r->mu, sizeof(double), r->L, stream);
		GDL_FREAD_STATUS (status, r->L);
		for (i = 0; i < r->L; i++)
		{
			status = fread (&(r->f[i][0]), sizeof(double), r->K, stream);
			GDL_FREAD_STATUS (status, r->K);	
		}
		r->haplo   = gdl_allele_block_fread (stream);
		GDL_FREAD_STATUS (r->haplo!=0, 1);	
		r->missing = gdl_gview_collector_fread (stream);
		GDL_FREAD_STATUS (r->missing!=0, 1);	
		r->labels  = gdl_glabels_fread (stream);
		GDL_FREAD_STATUS (r->labels!=0, 1);
		r->clust  = gdl_clustering_fread (stream);
		GDL_FREAD_STATUS (r->clust!=0, 1);
		r->chrom  = gdl_entity_fread (stream);
		GDL_FREAD_STATUS (r->chrom!=0, 1);
		r->viterbi = gdl_allele_block_fread (stream);
		GDL_FREAD_STATUS (r->viterbi!=0, 1);	
			
		return r;
	}
	return 0;	
}

int
gdl_mosaic_result_fwrite (FILE * stream, const gdl_mosaic_result * r)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->N, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->L, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->P, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->K, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->log, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (r->NA, sizeof(size_t), r->L, stream);
		GDL_FWRITE_STATUS (status, r->L);
		status = fwrite (r->D, sizeof(double), r->L-1, stream);
		GDL_FWRITE_STATUS (status, r->L-1);
		status = fwrite (r->rho, sizeof(double), r->L-1, stream);
		GDL_FWRITE_STATUS (status, r->L-1);
		status = fwrite (r->mu, sizeof(double), r->L, stream);
		GDL_FWRITE_STATUS (status, r->L);
		for (i = 0; i < r->L; i++)
		{
			status = fwrite (&(r->f[i][0]), sizeof(double), r->K, stream);
			GDL_FWRITE_STATUS (status, r->K);
		}
		status = gdl_allele_block_fwrite (stream, r->haplo);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);	
		status = gdl_gview_collector_fwrite (stream, r->missing);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);	
		status  = gdl_glabels_fwrite (stream, r->labels);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_clustering_fwrite (stream, r->clust);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);	
		status  = gdl_entity_fwrite (stream, r->chrom);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_allele_block_fwrite (stream, r->viterbi);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);	
			
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

int
gdl_mosaic_result_fprintf (FILE * stream, const gdl_mosaic_result * r)
{
	if (stream && r)
	{
		size_t i, j;
		double rho=0, mu=0;
		
		fprintf (stream, "---------------------------------------------\n");
		fprintf (stream, "--               gdl-mosaic                --\n");
		fprintf (stream, "---------------------------------------------\n");
		fprintf (stream, "Parameters:\n");
		fprintf (stream, "   N = %d (individuals)\n", r->N);
		fprintf (stream, "   L = %d (loci)\n", r->L);
		fprintf (stream, "   K = %d (clusters)\n", r->K);
		fprintf (stream, "---------------------------------------------\n");
		fprintf (stream, "Ln(Proba) = %.1f\n", r->log);
		fprintf (stream, "---------------------------------------------\n");
	   fprintf (stream, "Index\tLocus\tAllele\tDistance\tRho\tMu\tF-->\n");
	   for (i = 0; i < r->L; i++)
	   {
	   	fprintf (stream, "%d\t%s\t%d\t%.3f", i+1, gdl_glabels_locus (r->labels, i), r->NA[i], r->D[i]);
	   	if (i < r->L - 1) 
	   	{
	   		fprintf (stream, "\t%e\t%e", r->rho[i], r->mu[i]);
	   		rho += r->rho[i];
	   	}
	   	else
	   	{
	   		fprintf (stream, "\t-\t%e", r->mu[i]);
	   	}
	   	mu += r->mu[i];
	   	for (j = 0; j < r->K; j++)
	   	{
	   		fprintf (stream, "\t%1.2f", r->f[i][j]);	
	   	}
	   	fprintf (stream,"\n");
	   }
	   fprintf (stream, "---------------------------------------------\n");
	   fprintf (stream, " rho = %g\n", rho/(r->L-1));
	   fprintf (stream, " mu  = %g\n", mu/r->L);
	   fprintf (stream, "---------------------------------------------\n");
	   	   		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}


