/*  
 *  mosaic/work.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:01 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_mosaic.h>

gdl_mosaic *
gdl_mosaic_alloc (const size_t K, const gdl_chromosome * chrom, gdl_gview_wrapper * data, const gdl_rng * rng)
{
	gdl_mosaic * m;
	
	m = GDL_CALLOC (gdl_mosaic, 1);
	
	m->rng   = rng;
	m->data  = gdl_mosaic_data_alloc (chrom, data);
	m->clust = gdl_mosaic_cluster_alloc (K, m->data->L, m->data->NA);
	m->param = gdl_mosaic_params_alloc (K, m->data->L, m->data->D);
	m->util  = gdl_mosaic_util_alloc (K, m->data->L);
	
	return m;	
}

void
gdl_mosaic_free (gdl_mosaic * m)
{
	if (m)
	{
		gdl_mosaic_data_free (m->data);
		gdl_mosaic_cluster_free (m->clust);
		gdl_mosaic_params_free (m->param);
		gdl_mosaic_util_free (m->util);
		GDL_FREE (m);	
	}		
}

void
gdl_mosaic_ran_init (gdl_mosaic * m)
{
	gdl_mosaic_cluster_ran_init (m->clust, m->rng);
	gdl_mosaic_params_ran_init (m->param, m->rng);
}

int
gdl_mosaic_iterate_theta (gdl_mosaic * m, gdl_boolean rho_cst, gdl_boolean freq_cst)
{
	size_t i, j;
	double abs=0,sq=0;
	
	m->log = 0.;
	
	for (i = 0; i < m->data->NC; i++)
	{
		for (j = 0; j < m->data->P; j++)
		{
			gdl_mosaic_hmm_forward (m, i, j, &gdl_mosaic_hmm_obs_proba_theta);
			//printf ("LOG[%d] %g\n", i, m->util->log);
			m->log += m->util->log * m->data->W[i] * m->data->N * m->data->P;
			gdl_mosaic_hmm_backward (m, i, j, &gdl_mosaic_hmm_obs_proba_theta);
			gdl_mosaic_hmm_update_theta (m, i, j);
		}
	}
	
	gdl_mosaic_cluster_update (m->clust, &abs, &sq);
	gdl_mosaic_params_update (m->param, rho_cst, freq_cst, gdl_false, &abs, &sq);
	
	m->abs = abs;
	m->sq  = sq;
	
	return GDL_SUCCESS;
}

int
gdl_mosaic_iterate_haplo (gdl_mosaic * m, gdl_boolean rho_cst, gdl_boolean freq_cst, gdl_boolean mu_cst)
{
	size_t i, j;
	double abs=0,sq=0;
	
	m->log = 0.;
	
	for (i = 0; i < m->data->NC; i++)
	{
		for (j = 0; j < m->data->P; j++)
		{
			gdl_mosaic_hmm_forward (m, i, j, &gdl_mosaic_hmm_obs_proba_haplo);
			//printf ("LOG[%d] %g\n", i, m->util->log);
			m->log += m->util->log * m->data->W[i] * m->data->N * m->data->P;
			gdl_mosaic_hmm_backward (m, i, j, &gdl_mosaic_hmm_obs_proba_haplo);
			gdl_mosaic_hmm_update_haplo (m, i, j);
		}
	}
	
	gdl_mosaic_params_update (m->param, rho_cst, freq_cst, mu_cst, &abs, &sq);
	
	m->abs = abs;
	m->sq  = sq;
	
	return GDL_SUCCESS;
}

int
gdl_mosaic_perform (gdl_mosaic * m, gdl_boolean rho_cst, gdl_boolean freq_cst, gdl_boolean mu_cst, const double eps, const size_t max)
{
	size_t i, j, k, l;
	
	gdl_mosaic_ran_init (m);
	
	if (m->logger)
	{
		fprintf (m->logger, "Iter\tLog(Proba)\tAbs. Residual\tSq. Residual\n");	
	}
	
	for (i = 0; i < max; i++)
	{
		gdl_mosaic_iterate_theta (m, rho_cst, freq_cst);
		if (m->logger)
		{
			fprintf (m->logger, "%d\t%.1f\t%e\t%e\n", i+1, m->log, m->abs, m->sq);	
		}
		if (m->abs < eps)
		{
			break;	
		}
	}
	
//	for (i = 0; i < m->data->NC; i++)
//	{
//		for (j = 0; j < m->data->P; j++)
//		{
//			gdl_vector_uint * v = gdl_mosaic_hmm_viterbi (m, i, j, &gdl_mosaic_hmm_obs_proba_theta);
//			printf ("[%d]", i);
//			for (l = 0; l < v->size; l++)
//			{
//				printf (" %d", gdl_vector_uint_get (v, l));
//			}
//			printf ("\n");
//			gdl_vector_uint_free (v);
//		}	
//	}
	
	gdl_mosaic_cluster_haplo_init (m->clust, m->param->mu);
	
	for (i = 0; i < max; i++)
	{
		gdl_mosaic_iterate_haplo (m, rho_cst, freq_cst, mu_cst);
		if (m->logger)
		{
			fprintf (m->logger, "%d\t%.1f\t%e\t%e\n", i+1, m->log, m->abs, m->sq);	
		}
		if (m->abs < eps)
		{
			break;	
		}
	}
}

int
gdl_mosaic_imputation (gdl_mosaic * m, const double eps, const size_t max, size_t mode)
{
	size_t i;
	double abs, sq;
	
	if (m->logger)
	{
		fprintf (m->logger, "-------------------------\n");
		fprintf (m->logger, "-- Impute Missing Data --\n");
		fprintf (m->logger, "-------------------------\n");
		fprintf (m->logger, "Iter\tAbs. Residual\tSq. Residual\n");	
	}
	
	for (i = 0; i < max; i++)
	{
		abs=sq=0;
		if (mode)
		{
			gdl_mosaic_imputation_iterate (m, &abs, &sq, &gdl_mosaic_hmm_obs_proba_haplo);
		}
		else
		{
			gdl_mosaic_imputation_iterate (m, &abs, &sq, &gdl_mosaic_hmm_obs_proba_theta);
		}	
		if (m->logger)
		{
			fprintf (m->logger, "%d\t%e\t%e\n", i+1, abs, sq);	
		}
		if (abs < eps)
		{
			break;	
		}
	}
	
	return GDL_SUCCESS;
}

FILE *
gdl_mosaic_set_logger (gdl_mosaic * m, FILE * logger)
{
	FILE * old = m->logger;
	m->logger = logger;
	return old;
}
