/* multifit/covar.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2004 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_permutation.h>
#include <gdl/gdl_linalg.h>
#include <gdl/gdl_multifit_nlin.h>

/* Compute the covariance matrix

   cov = inv (J^T J)

   by QRP^T decomposition of J
*/

int
gdl_multifit_covar (const gdl_matrix * J, double epsrel, gdl_matrix * covar)
{
  double tolr;

  size_t i, j, k;
  size_t kmax = 0;

  gdl_matrix * r;
  gdl_vector * tau;
  gdl_vector * norm;
  gdl_permutation * perm;

  size_t m = J->size1, n = J->size2 ;
  
  if (m < n) 
    {
      GDL_ERROR ("Jacobian be rectangular M x N with M >= N", GDL_EBADLEN);
    }

  if (covar->size1 != covar->size2 || covar->size1 != n)
    {
      GDL_ERROR ("covariance matrix must be square and match second dimension of jacobian", GDL_EBADLEN);
    }

  r = gdl_matrix_alloc (m, n);
  tau = gdl_vector_alloc (n);
  perm = gdl_permutation_alloc (n) ;
  norm = gdl_vector_alloc (n) ;
  
  {
    int signum = 0;
    gdl_matrix_memcpy (r, J);
    gdl_linalg_QRPT_decomp (r, tau, perm, &signum, norm);
  }
  
  
  /* Form the inverse of R in the full upper triangle of R */

  tolr = epsrel * fabs(gdl_matrix_get(r, 0, 0));

  for (k = 0 ; k < n ; k++)
    {
      double rkk = gdl_matrix_get(r, k, k);

      if (fabs(rkk) <= tolr)
        {
          break;
        }

      gdl_matrix_set(r, k, k, 1.0/rkk);

      for (j = 0; j < k ; j++)
        {
          double t = gdl_matrix_get(r, j, k) / rkk;
          gdl_matrix_set (r, j, k, 0.0);

          for (i = 0; i <= j; i++)
            {
              double rik = gdl_matrix_get (r, i, k);
              double rij = gdl_matrix_get (r, i, j);
              
              gdl_matrix_set (r, i, k, rik - t * rij);
            }
        }
      kmax = k;
    }

  /* Form the full upper triangle of the inverse of R^T R in the full
     upper triangle of R */

  for (k = 0; k <= kmax ; k++)
    {
      for (j = 0; j < k; j++)
        {
          double rjk = gdl_matrix_get (r, j, k);

          for (i = 0; i <= j ; i++)
            {
              double rij = gdl_matrix_get (r, i, j);
              double rik = gdl_matrix_get (r, i, k);

              gdl_matrix_set (r, i, j, rij + rjk * rik);
            }
        }
      
      {
        double t = gdl_matrix_get (r, k, k);

        for (i = 0; i <= k; i++)
          {
            double rik = gdl_matrix_get (r, i, k);

            gdl_matrix_set (r, i, k, t * rik);
          };
      }
    }

  /* Form the full lower triangle of the covariance matrix in the
     strict lower triangle of R and in w */

  for (j = 0 ; j < n ; j++)
    {
      size_t pj = gdl_permutation_get (perm, j);
      
      for (i = 0; i <= j; i++)
        {
          size_t pi = gdl_permutation_get (perm, i);

          double rij;

          if (j > kmax)
            {
              gdl_matrix_set (r, i, j, 0.0);
              rij = 0.0 ;
            }
          else 
            {
              rij = gdl_matrix_get (r, i, j);
            }

          if (pi > pj)
            {
              gdl_matrix_set (r, pi, pj, rij); 
            } 
          else if (pi < pj)
            {
              gdl_matrix_set (r, pj, pi, rij);
            }

        }
      
      { 
        double rjj = gdl_matrix_get (r, j, j);
        gdl_matrix_set (covar, pj, pj, rjj);
      }
    }

     
  /* symmetrize the covariance matrix */

  for (j = 0 ; j < n ; j++)
    {
      for (i = 0; i < j ; i++)
        {
          double rji = gdl_matrix_get (r, j, i);

          gdl_matrix_set (covar, j, i, rji);
          gdl_matrix_set (covar, i, j, rji);
        }
    }

  gdl_matrix_free (r);
  gdl_permutation_free (perm);
  gdl_vector_free (tau);
  gdl_vector_free (norm);

  return GDL_SUCCESS;
}
