/* multifit_nlin/gdl_multifit_nlin.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef __GDL_MULTIFIT_NLIN_H__
#define __GDL_MULTIFIT_NLIN_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>

__BEGIN_DECLS

int gdl_multifit_gradient (const gdl_matrix * J, const gdl_vector * f,
                           gdl_vector * g);

int gdl_multifit_covar (const gdl_matrix * J, double epsrel, gdl_matrix * covar);


/* Definition of vector-valued functions with parameters based on gdl_vector */

struct gdl_multifit_function_struct
{
  int (* f) (const gdl_vector * x, void * params, gdl_vector * f);
  size_t n;   /* number of functions */
  size_t p;   /* number of independent variables */
  void * params;
};

typedef struct gdl_multifit_function_struct gdl_multifit_function ;

#define GDL_MULTIFIT_FN_EVAL(F,x,y) (*((F)->f))(x,(F)->params,(y))

typedef struct
  {
    const char *name;
    size_t size;
    int (*alloc) (void *state, size_t n, size_t p);
    int (*set) (void *state, gdl_multifit_function * function, gdl_vector * x, gdl_vector * f, gdl_vector * dx);
    int (*iterate) (void *state, gdl_multifit_function * function, gdl_vector * x, gdl_vector * f, gdl_vector * dx);
    void (*free) (void *state);
  }
gdl_multifit_fsolver_type;

typedef struct
  {
    const gdl_multifit_fsolver_type * type;
    gdl_multifit_function * function ;
    gdl_vector * x ;
    gdl_vector * f ;
    gdl_vector * dx ;
    void *state;
  }
gdl_multifit_fsolver;

gdl_multifit_fsolver *
gdl_multifit_fsolver_alloc (const gdl_multifit_fsolver_type * T, 
                            size_t n, size_t p);

void gdl_multifit_fsolver_free (gdl_multifit_fsolver * s);

int gdl_multifit_fsolver_set (gdl_multifit_fsolver * s, 
                                   gdl_multifit_function * f, 
                                   gdl_vector * x);

int gdl_multifit_fsolver_iterate (gdl_multifit_fsolver * s);

const char * gdl_multifit_fsolver_name (const gdl_multifit_fsolver * s);
gdl_vector * gdl_multifit_fsolver_position (const gdl_multifit_fsolver * s);

/* Definition of vector-valued functions and gradient with parameters
   based on gdl_vector */

struct gdl_multifit_function_fdf_struct
{
  int (* f) (const gdl_vector * x, void * params, gdl_vector * f);
  int (* df) (const gdl_vector * x, void * params, gdl_matrix * df);
  int (* fdf) (const gdl_vector * x, void * params, gdl_vector * f, gdl_matrix *df);
  size_t n;   /* number of functions */
  size_t p;   /* number of independent variables */
  void * params;
};

typedef struct gdl_multifit_function_fdf_struct gdl_multifit_function_fdf ;

#define GDL_MULTIFIT_FN_EVAL_F(F,x,y) ((*((F)->f))(x,(F)->params,(y)))
#define GDL_MULTIFIT_FN_EVAL_DF(F,x,dy) ((*((F)->df))(x,(F)->params,(dy)))
#define GDL_MULTIFIT_FN_EVAL_F_DF(F,x,y,dy) ((*((F)->fdf))(x,(F)->params,(y),(dy)))

typedef struct
  {
    const char *name;
    size_t size;
    int (*alloc) (void *state, size_t n, size_t p);
    int (*set) (void *state, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx);
    int (*iterate) (void *state, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx);
    void (*free) (void *state);
  }
gdl_multifit_fdfsolver_type;

typedef struct
  {
    const gdl_multifit_fdfsolver_type * type;
    gdl_multifit_function_fdf * fdf ;
    gdl_vector * x;
    gdl_vector * f;
    gdl_matrix * J;
    gdl_vector * dx;
    void *state;
  }
gdl_multifit_fdfsolver;


gdl_multifit_fdfsolver *
gdl_multifit_fdfsolver_alloc (const gdl_multifit_fdfsolver_type * T, 
                              size_t n, size_t p);

int
gdl_multifit_fdfsolver_set (gdl_multifit_fdfsolver * s, 
                                 gdl_multifit_function_fdf * fdf,
                                 gdl_vector * x);

int
gdl_multifit_fdfsolver_iterate (gdl_multifit_fdfsolver * s);

void
gdl_multifit_fdfsolver_free (gdl_multifit_fdfsolver * s);

const char * gdl_multifit_fdfsolver_name (const gdl_multifit_fdfsolver * s);
gdl_vector * gdl_multifit_fdfsolver_position (const gdl_multifit_fdfsolver * s);

int gdl_multifit_test_delta (const gdl_vector * dx, const gdl_vector * x, 
                             double epsabs, double epsrel);

int gdl_multifit_test_gradient (const gdl_vector * g, double epsabs);

/* extern const gdl_multifit_fsolver_type * gdl_multifit_fsolver_gradient; */

GDL_VAR const gdl_multifit_fdfsolver_type * gdl_multifit_fdfsolver_lmder;
GDL_VAR const gdl_multifit_fdfsolver_type * gdl_multifit_fdfsolver_lmsder;

typedef struct
{
	size_t max_iter;
	double epsilon;
	gdl_vector * unew;
	gdl_vector * uold;
	gdl_vector * gnew;
	gdl_vector * gold;
	gdl_vector * wold;
	gdl_vector * wnew;
} gdl_multifit_logistic_workspace;

gdl_multifit_logistic_workspace * gdl_multifit_logistic_alloc (const size_t M);
void gdl_multifit_logistic_free (gdl_multifit_logistic_workspace * w);

int
gdl_multifit_logistic (const gdl_matrix      * X,
                       const gdl_vector_uint * y,
                       gdl_vector * w,
                       double * loglikelihood,
                       gdl_multifit_logistic_workspace * work);

__END_DECLS

#endif /* __GDL_MULTIFIT_NLIN_H__ */
