/* multfit/lmder.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <gdl/gdl_common.h>

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <float.h>

#include <gdl/gdl_math.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_multifit_nlin.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_linalg.h>
#include <gdl/gdl_permutation.h>


typedef struct
  {
    size_t iter;
    double xnorm;
    double fnorm;
    double delta;
    double par;
    gdl_matrix *r;
    gdl_vector *tau;
    gdl_vector *diag;
    gdl_vector *qtf;
    gdl_vector *newton;
    gdl_vector *gradient;
    gdl_vector *x_trial;
    gdl_vector *f_trial;
    gdl_vector *df;
    gdl_vector *sdiag;
    gdl_vector *rptdx;
    gdl_vector *w;
    gdl_vector *work1;
    gdl_permutation * perm;
  }
lmder_state_t;

static int lmder_alloc (void *vstate, size_t n, size_t p);
static int lmder_set (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx);
static int lmsder_set (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx);
static int set (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx, int scale);
static int lmder_iterate (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx);
static void lmder_free (void *vstate);
static int iterate (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx, int scale);

#include "lmutil.c"
#include "lmpar.c"
#include "lmset.c"
#include "lmiterate.c"


static int
lmder_alloc (void *vstate, size_t n, size_t p)
{
  lmder_state_t *state = (lmder_state_t *) vstate;
  gdl_matrix *r;
  gdl_vector *tau, *diag, *qtf, *newton, *gradient, *x_trial, *f_trial,
   *df, *sdiag, *rptdx, *w, *work1;
  gdl_permutation *perm;

  r = gdl_matrix_calloc (n, p);

  if (r == 0)
    {
      GDL_ERROR ("failed to allocate space for r", GDL_ENOMEM);
    }

  state->r = r;

  tau = gdl_vector_calloc (GDL_MIN(n, p));

  if (tau == 0)
    {
      gdl_matrix_free (r);

      GDL_ERROR ("failed to allocate space for tau", GDL_ENOMEM);
    }

  state->tau = tau;

  diag = gdl_vector_calloc (p);

  if (diag == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);

      GDL_ERROR ("failed to allocate space for diag", GDL_ENOMEM);
    }

  state->diag = diag;

  qtf = gdl_vector_calloc (n);

  if (qtf == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);

      GDL_ERROR ("failed to allocate space for qtf", GDL_ENOMEM);
    }

  state->qtf = qtf;

  newton = gdl_vector_calloc (p);

  if (newton == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);

      GDL_ERROR ("failed to allocate space for newton", GDL_ENOMEM);
    }

  state->newton = newton;

  gradient = gdl_vector_calloc (p);

  if (gradient == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);

      GDL_ERROR ("failed to allocate space for gradient", GDL_ENOMEM);
    }

  state->gradient = gradient;

  x_trial = gdl_vector_calloc (p);

  if (x_trial == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);

      GDL_ERROR ("failed to allocate space for x_trial", GDL_ENOMEM);
    }

  state->x_trial = x_trial;

  f_trial = gdl_vector_calloc (n);

  if (f_trial == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);
      gdl_vector_free (x_trial);

      GDL_ERROR ("failed to allocate space for f_trial", GDL_ENOMEM);
    }

  state->f_trial = f_trial;

  df = gdl_vector_calloc (n);

  if (df == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);
      gdl_vector_free (x_trial);
      gdl_vector_free (f_trial);

      GDL_ERROR ("failed to allocate space for df", GDL_ENOMEM);
    }

  state->df = df;

  sdiag = gdl_vector_calloc (p);

  if (sdiag == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);
      gdl_vector_free (x_trial);
      gdl_vector_free (f_trial);
      gdl_vector_free (df);

      GDL_ERROR ("failed to allocate space for sdiag", GDL_ENOMEM);
    }

  state->sdiag = sdiag;


  rptdx = gdl_vector_calloc (n);

  if (rptdx == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);
      gdl_vector_free (x_trial);
      gdl_vector_free (f_trial);
      gdl_vector_free (df);
      gdl_vector_free (sdiag);

      GDL_ERROR ("failed to allocate space for rptdx", GDL_ENOMEM);
    }

  state->rptdx = rptdx;

  w = gdl_vector_calloc (n);

  if (w == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);
      gdl_vector_free (x_trial);
      gdl_vector_free (f_trial);
      gdl_vector_free (df);
      gdl_vector_free (sdiag);
      gdl_vector_free (rptdx);

      GDL_ERROR ("failed to allocate space for w", GDL_ENOMEM);
    }

  state->w = w;

  work1 = gdl_vector_calloc (p);

  if (work1 == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);
      gdl_vector_free (x_trial);
      gdl_vector_free (f_trial);
      gdl_vector_free (df);
      gdl_vector_free (sdiag);
      gdl_vector_free (rptdx);
      gdl_vector_free (w);

      GDL_ERROR ("failed to allocate space for work1", GDL_ENOMEM);
    }

  state->work1 = work1;

  perm = gdl_permutation_calloc (p);

  if (perm == 0)
    {
      gdl_matrix_free (r);
      gdl_vector_free (tau);
      gdl_vector_free (diag);
      gdl_vector_free (qtf);
      gdl_vector_free (newton);
      gdl_vector_free (gradient);
      gdl_vector_free (x_trial);
      gdl_vector_free (f_trial);
      gdl_vector_free (df);
      gdl_vector_free (sdiag);
      gdl_vector_free (rptdx);
      gdl_vector_free (w);
      gdl_vector_free (work1);

      GDL_ERROR ("failed to allocate space for perm", GDL_ENOMEM);
    }

  state->perm = perm;

  return GDL_SUCCESS;
}

static int
lmder_set (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx)
{
  int status = set (vstate, fdf, x, f, J, dx, 0);
  return status ;
}

static int
lmsder_set (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx)
{
  int status = set (vstate, fdf, x, f, J, dx, 1);
  return status ;
}

static int
lmder_iterate (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx)
{
  int status = iterate (vstate, fdf, x, f, J, dx, 0);
  return status;
}

static int
lmsder_iterate (void *vstate, gdl_multifit_function_fdf * fdf, gdl_vector * x, gdl_vector * f, gdl_matrix * J, gdl_vector * dx)
{
  int status = iterate (vstate, fdf, x, f, J, dx, 1);
  return status;
}

static void
lmder_free (void *vstate)
{
  lmder_state_t *state = (lmder_state_t *) vstate;

  gdl_permutation_free (state->perm);
  gdl_vector_free (state->work1);
  gdl_vector_free (state->w);
  gdl_vector_free (state->rptdx);
  gdl_vector_free (state->sdiag);
  gdl_vector_free (state->df);
  gdl_vector_free (state->f_trial);
  gdl_vector_free (state->x_trial);
  gdl_vector_free (state->gradient);
  gdl_vector_free (state->newton);
  gdl_vector_free (state->qtf);
  gdl_vector_free (state->diag);
  gdl_vector_free (state->tau);
  gdl_matrix_free (state->r);
}

static const gdl_multifit_fdfsolver_type lmder_type =
{
  "lmder",                      /* name */
  sizeof (lmder_state_t),
  &lmder_alloc,
  &lmder_set,
  &lmder_iterate,
  &lmder_free
};

static const gdl_multifit_fdfsolver_type lmsder_type =
{
  "lmsder",                     /* name */
  sizeof (lmder_state_t),
  &lmder_alloc,
  &lmsder_set,
  &lmsder_iterate,
  &lmder_free
};

const gdl_multifit_fdfsolver_type *gdl_multifit_fdfsolver_lmder = &lmder_type;
const gdl_multifit_fdfsolver_type *gdl_multifit_fdfsolver_lmsder = &lmsder_type;
