/* multifit/multilinear.c
 * 
 * Copyright (C) 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_multifit.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_linalg_svd.h>

/* Fit
 *
 *  y = X c
 *
 *  where X is an M x N matrix of M observations for N variables.
 *
 */

int
gdl_multifit_linear (const gdl_matrix * X,
                     const gdl_vector * y,
                     gdl_vector * c,
                     gdl_matrix * cov,
                     double *chisq, gdl_multifit_linear_workspace * work)
{
  size_t rank;
  int status  = gdl_multifit_linear_svd (X, y, GDL_DBL_EPSILON, &rank, c,
                                         cov, chisq, work);
  return status;
}

/* Handle the general case of the SVD with tolerance and rank */

int
gdl_multifit_linear_svd (const gdl_matrix * X,
                         const gdl_vector * y,
                         double tol,
                         size_t * rank,
                         gdl_vector * c,
                         gdl_matrix * cov,
                         double *chisq, gdl_multifit_linear_workspace * work)
{
  if (X->size1 != y->size)
    {
      GDL_ERROR
        ("number of observations in y does not match rows of matrix X",
         GDL_EBADLEN);
    }
  else if (X->size2 != c->size)
    {
      GDL_ERROR ("number of parameters c does not match columns of matrix X",
                 GDL_EBADLEN);
    }
  else if (cov->size1 != cov->size2)
    {
      GDL_ERROR ("covariance matrix is not square", GDL_ENOTSQR);
    }
  else if (c->size != cov->size1)
    {
      GDL_ERROR
        ("number of parameters does not match size of covariance matrix",
         GDL_EBADLEN);
    }
  else if (X->size1 != work->n || X->size2 != work->p)
    {
      GDL_ERROR
        ("size of workspace does not match size of observation matrix",
         GDL_EBADLEN);
    }
  else if (tol <= 0)
    {
      GDL_ERROR ("tolerance must be positive", GDL_EINVAL);
    }
  else
    {
      const size_t n = X->size1;
      const size_t p = X->size2;

      size_t i, j, p_eff;

      gdl_matrix *A = work->A;
      gdl_matrix *Q = work->Q;
      gdl_matrix *QSI = work->QSI;
      gdl_vector *S = work->S;
      gdl_vector *xt = work->xt;
      gdl_vector *D = work->D;

      /* Copy X to workspace,  A <= X */

      gdl_matrix_memcpy (A, X);

      /* Balance the columns of the matrix A */

      gdl_linalg_balance_columns (A, D);

      /* Decompose A into U S Q^T */

      gdl_linalg_SV_decomp_mod (A, QSI, Q, S, xt);
      
      /* Solve y = A c for c */

      gdl_blas_dgemv (CblasTrans, 1.0, A, y, 0.0, xt);

      /* Scale the matrix Q,  Q' = Q S^-1 */

      gdl_matrix_memcpy (QSI, Q);

      {
        double alpha0 = gdl_vector_get (S, 0);
        p_eff = 0;

        for (j = 0; j < p; j++)
          {
            gdl_vector_view column = gdl_matrix_column (QSI, j);
            double alpha = gdl_vector_get (S, j);

            if (alpha <= tol * alpha0) {
              alpha = 0.0;
            } else {
              alpha = 1.0 / alpha;
              p_eff++;
            }

            gdl_vector_scale (&column.vector, alpha);
          }

        *rank = p_eff;
      }

      gdl_vector_set_zero (c);

      gdl_blas_dgemv (CblasNoTrans, 1.0, QSI, xt, 0.0, c);

      /* Unscale the balancing factors */

      gdl_vector_div (c, D);

      /* Compute chisq, from residual r = y - X c */

      {
        double s2 = 0, r2 = 0;

        for (i = 0; i < n; i++)
          {
            double yi = gdl_vector_get (y, i);
            gdl_vector_const_view row = gdl_matrix_const_row (X, i);
            double y_est, ri;
            gdl_blas_ddot (&row.vector, c, &y_est);
            ri = yi - y_est;
            r2 += ri * ri;
          }

        s2 = r2 / (n - p_eff);   /* p_eff == rank */

        *chisq = r2;

        /* Form variance-covariance matrix cov = s2 * (Q S^-1) (Q S^-1)^T */

        for (i = 0; i < p; i++)
          {
            gdl_vector_view row_i = gdl_matrix_row (QSI, i);
            double d_i = gdl_vector_get (D, i);

            for (j = i; j < p; j++)
              {
                gdl_vector_view row_j = gdl_matrix_row (QSI, j);
                double d_j = gdl_vector_get (D, j);
                double s;

                gdl_blas_ddot (&row_i.vector, &row_j.vector, &s);

                gdl_matrix_set (cov, i, j, s * s2 / (d_i * d_j));
                gdl_matrix_set (cov, j, i, s * s2 / (d_i * d_j));
              }
          }
      }

      return GDL_SUCCESS;
    }
}

int
gdl_multifit_wlinear (const gdl_matrix * X,
                      const gdl_vector * w,
                      const gdl_vector * y,
                      gdl_vector * c,
                      gdl_matrix * cov,
                      double *chisq, gdl_multifit_linear_workspace * work)
{
  size_t rank;
  int status  = gdl_multifit_wlinear_svd (X, w, y, GDL_DBL_EPSILON, &rank, c,
                                          cov, chisq, work);
  return status;
}


int
gdl_multifit_wlinear_svd (const gdl_matrix * X,
                          const gdl_vector * w,
                          const gdl_vector * y,
                          double tol,
                          size_t * rank,
                          gdl_vector * c,
                          gdl_matrix * cov,
                          double *chisq, gdl_multifit_linear_workspace * work)
{
  if (X->size1 != y->size)
    {
      GDL_ERROR
        ("number of observations in y does not match rows of matrix X",
         GDL_EBADLEN);
    }
  else if (X->size2 != c->size)
    {
      GDL_ERROR ("number of parameters c does not match columns of matrix X",
                 GDL_EBADLEN);
    }
  else if (w->size != y->size)
    {
      GDL_ERROR ("number of weights does not match number of observations",
                 GDL_EBADLEN);
    }
  else if (cov->size1 != cov->size2)
    {
      GDL_ERROR ("covariance matrix is not square", GDL_ENOTSQR);
    }
  else if (c->size != cov->size1)
    {
      GDL_ERROR
        ("number of parameters does not match size of covariance matrix",
         GDL_EBADLEN);
    }
  else if (X->size1 != work->n || X->size2 != work->p)
    {
      GDL_ERROR
        ("size of workspace does not match size of observation matrix",
         GDL_EBADLEN);
    }
  else
    {
      const size_t n = X->size1;
      const size_t p = X->size2;

      size_t i, j, p_eff;

      gdl_matrix *A = work->A;
      gdl_matrix *Q = work->Q;
      gdl_matrix *QSI = work->QSI;
      gdl_vector *S = work->S;
      gdl_vector *t = work->t;
      gdl_vector *xt = work->xt;
      gdl_vector *D = work->D;

      /* Scale X,  A = sqrt(w) X */

      gdl_matrix_memcpy (A, X);

      for (i = 0; i < n; i++)
        {
          double wi = gdl_vector_get (w, i);

          if (wi < 0)
            wi = 0;

          {
            gdl_vector_view row = gdl_matrix_row (A, i);
            gdl_vector_scale (&row.vector, sqrt (wi));
          }
        }

      /* Balance the columns of the matrix A */

      gdl_linalg_balance_columns (A, D);

      /* Decompose A into U S Q^T */

      gdl_linalg_SV_decomp_mod (A, QSI, Q, S, xt);

      /* Solve sqrt(w) y = A c for c, by first computing t = sqrt(w) y */

      for (i = 0; i < n; i++)
        {
          double wi = gdl_vector_get (w, i);
          double yi = gdl_vector_get (y, i);
          if (wi < 0)
            wi = 0;
          gdl_vector_set (t, i, sqrt (wi) * yi);
        }

      gdl_blas_dgemv (CblasTrans, 1.0, A, t, 0.0, xt);

      /* Scale the matrix Q,  Q' = Q S^-1 */

      gdl_matrix_memcpy (QSI, Q);

      {
        double alpha0 = gdl_vector_get (S, 0);
        p_eff = 0;
        
        for (j = 0; j < p; j++)
          {
            gdl_vector_view column = gdl_matrix_column (QSI, j);
            double alpha = gdl_vector_get (S, j);

            if (alpha <= tol * alpha0) {
              alpha = 0.0;
            } else {
              alpha = 1.0 / alpha;
              p_eff++;
            }

            gdl_vector_scale (&column.vector, alpha);
          }

        *rank = p_eff;
      }

      gdl_vector_set_zero (c);

      /* Solution */

      gdl_blas_dgemv (CblasNoTrans, 1.0, QSI, xt, 0.0, c);

      /* Unscale the balancing factors */

      gdl_vector_div (c, D);

      /* Form covariance matrix cov = (Q S^-1) (Q S^-1)^T */

      for (i = 0; i < p; i++)
        {
          gdl_vector_view row_i = gdl_matrix_row (QSI, i);
          double d_i = gdl_vector_get (D, i);

          for (j = i; j < p; j++)
            {
              gdl_vector_view row_j = gdl_matrix_row (QSI, j);
              double d_j = gdl_vector_get (D, j);
              double s;

              gdl_blas_ddot (&row_i.vector, &row_j.vector, &s);

              gdl_matrix_set (cov, i, j, s / (d_i * d_j));
              gdl_matrix_set (cov, j, i, s / (d_i * d_j));
            }
        }

      /* Compute chisq, from residual r = y - X c */

      {
        double r2 = 0;

        for (i = 0; i < n; i++)
          {
            double yi = gdl_vector_get (y, i);
            double wi = gdl_vector_get (w, i);
            gdl_vector_const_view row = gdl_matrix_const_row (X, i);
            double y_est, ri;
            gdl_blas_ddot (&row.vector, c, &y_est);
            ri = yi - y_est;
            r2 += wi * ri * ri;
          }

        *chisq = r2;
      }

      return GDL_SUCCESS;
    }
}


int
gdl_multifit_linear_est (const gdl_vector * x,
                         const gdl_vector * c,
                         const gdl_matrix * cov, double *y, double *y_err)
{

  if (x->size != c->size)
    {
      GDL_ERROR ("number of parameters c does not match number of observations x",
         GDL_EBADLEN);
    }
  else if (cov->size1 != cov->size2)
    {
      GDL_ERROR ("covariance matrix is not square", GDL_ENOTSQR);
    }
  else if (c->size != cov->size1)
    {
      GDL_ERROR ("number of parameters c does not match size of covariance matrix cov",
         GDL_EBADLEN);
    }
  else
    {
      size_t i, j;
      double var = 0;
      
      gdl_blas_ddot(x, c, y);       /* y = x.c */

      /* var = x' cov x */

      for (i = 0; i < x->size; i++)
        {
          const double xi = gdl_vector_get (x, i);
          var += xi * xi * gdl_matrix_get (cov, i, i);

          for (j = 0; j < i; j++)
            {
              const double xj = gdl_vector_get (x, j);
              var += 2 * xi * xj * gdl_matrix_get (cov, i, j);
            }
        }

      *y_err = sqrt (var);

      return GDL_SUCCESS;
    }
}
