/* multimin/fdfminimizer.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Fabrice Rossi
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <config.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_multimin.h>

gdl_multimin_fdfminimizer *
gdl_multimin_fdfminimizer_alloc (const gdl_multimin_fdfminimizer_type * T,
                                 size_t n)
{
  int status;

  gdl_multimin_fdfminimizer *s =
    (gdl_multimin_fdfminimizer *) malloc (sizeof (gdl_multimin_fdfminimizer));

  if (s == 0)
    {
      GDL_ERROR_VAL ("failed to allocate space for minimizer struct",
                     GDL_ENOMEM, 0);
    }

  s->type = T;

  s->x = gdl_vector_calloc (n);

  if (s->x == 0) 
    {
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for x", GDL_ENOMEM, 0);
    }

  s->gradient = gdl_vector_calloc (n);

  if (s->gradient == 0) 
    {
      gdl_vector_free (s->x);
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for gradient", GDL_ENOMEM, 0);
    }

  s->dx = gdl_vector_calloc (n);

  if (s->dx == 0) 
    {
      gdl_vector_free (s->x);
      gdl_vector_free (s->gradient);
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for dx", GDL_ENOMEM, 0);
    }

  s->state = malloc (T->size);

  if (s->state == 0)
    {
      gdl_vector_free (s->x);
      gdl_vector_free (s->gradient);
      gdl_vector_free (s->dx);
      free (s);
      GDL_ERROR_VAL ("failed to allocate space for minimizer state",
                     GDL_ENOMEM, 0);
    }

  status = (T->alloc) (s->state, n);

  if (status != GDL_SUCCESS)
    {
      free (s->state);
      gdl_vector_free (s->x);
      gdl_vector_free (s->gradient);
      gdl_vector_free (s->dx);
      free (s);

      GDL_ERROR_VAL ("failed to initialize minimizer state", GDL_ENOMEM, 0);
    }

  return s;
}

int
gdl_multimin_fdfminimizer_set (gdl_multimin_fdfminimizer * s,
                               gdl_multimin_function_fdf * fdf,
                               const gdl_vector * x,
                               double step_size, double tol)
{
  if (s->x->size != fdf->n)
    {
      GDL_ERROR ("function incompatible with solver size", GDL_EBADLEN);
    }
  
  if (x->size != fdf->n) 
    {
      GDL_ERROR ("vector length not compatible with function", GDL_EBADLEN);
    }  
    
  s->fdf = fdf;

  gdl_vector_memcpy (s->x,x);
  gdl_vector_set_zero (s->dx);
  
  return (s->type->set) (s->state, s->fdf, s->x, &(s->f), s->gradient, step_size, tol);
}

void
gdl_multimin_fdfminimizer_free (gdl_multimin_fdfminimizer * s)
{
  (s->type->free) (s->state);
  free (s->state);
  gdl_vector_free (s->dx);
  gdl_vector_free (s->gradient);
  gdl_vector_free (s->x);
  free (s);
}

int
gdl_multimin_fdfminimizer_iterate (gdl_multimin_fdfminimizer * s)
{
  return (s->type->iterate) (s->state, s->fdf, s->x, &(s->f), s->gradient, s->dx);
}

int
gdl_multimin_fdfminimizer_restart (gdl_multimin_fdfminimizer * s)
{
  return (s->type->restart) (s->state);
}

const char * 
gdl_multimin_fdfminimizer_name (const gdl_multimin_fdfminimizer * s)
{
  return s->type->name;
}


gdl_vector * 
gdl_multimin_fdfminimizer_x (gdl_multimin_fdfminimizer * s)
{
  return s->x;
}

gdl_vector * 
gdl_multimin_fdfminimizer_dx (gdl_multimin_fdfminimizer * s)
{
  return s->dx;
}

gdl_vector * 
gdl_multimin_fdfminimizer_gradient (gdl_multimin_fdfminimizer * s)
{
  return s->gradient;
}

double 
gdl_multimin_fdfminimizer_minimum (gdl_multimin_fdfminimizer * s)
{
  return s->f;
}

