/* multireg/univar.c
 * 
 * Copyright (C) 2006 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <math.h>

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_multireg.h>

/**
 * Perform y = mu + a * x + e
 * Return a 
 * *mu = mu
 * *r2 = prop of variance explained
 * *p  = p-value of H0 a=0 vs H1 : a!=0
 * 
 * missing values must be encoded as NaN
 * 
 */
double
gdl_univreg_perform (const double y[],
                     const double x[],
                     const size_t n,
                     double * mu,
                     double * r2,
                     double * p)
{
	size_t i, nn;
	double a,xm,xx,ym,yy,xy,sig0,sig1,chi;
	
	xm = ym = xx = xy = 0;
	for (nn = i = 0; i < n; i++)
	{
		if (gdl_isnan(x[i]) || gdl_isnan(y[i]))
			continue;
		xm += x[i];
		xx += x[i]*x[i];
		ym += y[i];
		xy += y[i]*x[i];
		nn++;
	}
	a=(nn*xy-xm*ym)/(nn*xx-xm*xm);
	ym/=nn;
	xm/=nn;
	sig0 = sig1 = 0;
	for (i = 0; i < n; i++)
	{
		if (gdl_isnan(x[i]) || gdl_isnan(y[i]))
			continue;
		sig0 += (y[i]-ym)*(y[i]-ym);
		xy    = (y[i]-ym)-a*(x[i]-xm);
		sig1 += xy*xy;
	}
	chi = (sig0-sig1)*(nn - 2)/sig1;
	chi = (chi >= 0) ? chi : 0;
	*p  = gdl_sf_beta_inc (0.5*(nn - 2), 0.5,  (nn - 2)/((nn - 2) + chi));
	*r2 = 1-sig1/sig0;
	return chi;
}
