/* permutation/permutation.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_permutation.h>

size_t
gdl_permutation_size (const gdl_permutation * p)
{
  return p->size ;
}

size_t *
gdl_permutation_data (const gdl_permutation * p)
{
  return p->data ;
}

#ifndef HIDE_INLINE_STATIC
size_t
gdl_permutation_get (const gdl_permutation * p, const size_t i)
{
  if (gdl_check_range)
    {
      if (i >= p->size)         /* size_t is unsigned, can't be negative */
        {
          GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
        }
    }

  return p->data[i];
}
#endif

int
gdl_permutation_swap (gdl_permutation * p, const size_t i, const size_t j)
{
  const size_t size = p->size ;
  
  if (i >= size)
    {
      GDL_ERROR("first index is out of range", GDL_EINVAL);
    }

  if (j >= size)
    {
      GDL_ERROR("second index is out of range", GDL_EINVAL);
    }

  if (i != j)
    {
      size_t tmp = p->data[i];
      p->data[i] = p->data[j];
      p->data[j] = tmp;
    }
  
  return GDL_SUCCESS;
}


int
gdl_permutation_valid (gdl_permutation * p)
{
  const size_t size = p->size ;

  size_t i, j ;

  for (i = 0; i < size; i++) 
    {
      if (p->data[i] >= size)
        {
          GDL_ERROR("permutation index outside range", GDL_FAILURE) ;
        }

      for (j = 0; j < i; j++)
        {
          if (p->data[i] == p->data[j])
            {
              GDL_ERROR("duplicate permutation index", GDL_FAILURE) ;
            }
        }
    }
  
  return GDL_SUCCESS;
}

void
gdl_permutation_reverse (gdl_permutation * p)
{
  const size_t size = p->size ;

  size_t i ;
  
  for (i = 0; i < (size / 2); i++) 
    {
      size_t j = size - i - 1;

      size_t tmp = p->data[i] ;
      p->data[i] = p->data[j] ;
      p->data[j] = tmp ;
    }
}

int 
gdl_permutation_inverse (gdl_permutation * inv, const gdl_permutation * p)
{
  const size_t size = p->size ;

  size_t i ;

  if (inv->size != size)
    {
      GDL_ERROR("permutation lengths are not equal", GDL_EBADLEN);
    }
  
  for (i = 0; i < size; i++) 
    {
      inv->data[p->data[i]] = i ;
    }
  
  return GDL_SUCCESS ;
}

int
gdl_permutation_next (gdl_permutation * p)
{
  /* Replaces p with the next permutation (in the standard lexicographical
   * ordering).  Returns GDL_FAILURE if there is no next permutation.
   */
  const size_t size = p->size;
  size_t i, j, k;

  if (size < 2)
    {
      return GDL_FAILURE;
    }

  i = size - 2;

  while ((p->data[i] > p->data[i+1]) && (i != 0))
    {
      i--;
    }

  if ((i == 0) && (p->data[0] > p->data[1]))
    {
     return GDL_FAILURE;
    }

  k = i + 1;

  for (j = i + 2; j < size; j++ )
    {
      if ((p->data[j] > p->data[i]) && (p->data[j] < p->data[k]))
        {
          k = j;
        }
    }

  /* swap i and k */

  {
    size_t tmp = p->data[i];
    p->data[i] = p->data[k];
    p->data[k] = tmp;
  }

  for (j = i + 1; j <= ((size + i) / 2); j++)
    {
      size_t tmp = p->data[j];
      p->data[j] = p->data[size + i - j];
      p->data[size + i - j] = tmp;
    }

  return GDL_SUCCESS;
}

int
gdl_permutation_prev (gdl_permutation * p)
{
  const size_t size = p->size;
  size_t i, j, k;

  if (size < 2)
    {
      return GDL_FAILURE;
    }

  i = size - 2;

  while ((p->data[i] < p->data[i+1]) && (i != 0))
    {
      i--;
    }

  if ((i == 0) && (p->data[0] < p->data[1]))
    {
      return GDL_FAILURE;
    }

  k = i + 1;

  for (j = i + 2; j < size; j++ )
    {
      if ((p->data[j] < p->data[i]) && (p->data[j] > p->data[k]))
        {
          k = j;
        }
    }

  /* swap i and k */

  {
    size_t tmp = p->data[i];
    p->data[i] = p->data[k];
    p->data[k] = tmp;
  }

  for (j = i + 1; j <= ((size + i) / 2); j++)
    {
      size_t tmp = p->data[j];
      p->data[j] = p->data[size + i - j];
      p->data[size + i - j] = tmp;
    }

  return GDL_SUCCESS;
}

int
gdl_permutation_mul (gdl_permutation * p, const gdl_permutation * pa, const gdl_permutation * pb)
{
  size_t i;
  const size_t size = p->size;

  if (pa->size != size)
    {
      GDL_ERROR("size of result does not match size of pa", GDL_EINVAL);
    }

  if (pb->size != size)
    {
      GDL_ERROR("size of result does not match size of pb", GDL_EINVAL);
    }

  for (i = 0; i < size; i++)
    {
      p->data[i] = pb->data[pa->data[i]];
    }

  return GDL_SUCCESS;
}
int
gdl_permutation_memcpy (gdl_permutation * dest,
                        const gdl_permutation * src)
{
  const size_t src_size = src->size;
  const size_t dest_size = dest->size;

  if (src_size != dest_size)
    {
      GDL_ERROR ("permutation lengths are not equal", GDL_EBADLEN);
    }

  {
    size_t j;

    for (j = 0; j < src_size; j++)
      {
        dest->data[j] = src->data[j];
      }
  }

  return GDL_SUCCESS;
}
