/*  
 *  plot/param.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_graphics.h>
#include <gdl/gdl_plot.h>

gdl_plot_parameter *
gdl_plot_parameter_alloc (const gdl_plot_parameter_type * T, size_t size)
{
	gdl_plot_parameter * p;
	
	p = GDL_CALLOC (gdl_plot_parameter, 1);
	
	p->type  = T;
	p->size  = size;
	if (T->size)
	{
		p->data  = gdl_malloc (T->size*size);
	}
	
	return p;	
}

void
gdl_plot_parameter_free (gdl_plot_parameter * p)
{
	if (p)
	{
		size_t i;
		if (p->type->free)
		{
			for (i = 0; i < p->size; i++)
			{
				(p->type->free)(p->data[i]);
			}
		}
		GDL_FREE (p->data);
		GDL_FREE (p);
	}	
}

void
gdl_plot_parameter_set (gdl_plot_parameter * p, size_t i, void * x)
{
	p->data[i] = x;
}

void
gdl_plot_parameter_set_copy (gdl_plot_parameter * p, size_t i, const void * x)
{
	if (p->type->copy)
	{
		p->data[i] = (p->type->copy)(x);
	}
}

size_t
gdl_plot_parameter_size (const gdl_plot_parameter * p)
{
	return p->size;	
}

const void *
gdl_plot_parameter_get (const gdl_plot_parameter * p, size_t i)
{
	return p->data[i];
}

static int
gdl_plot_parameter_fprintf (FILE * stream, const gdl_plot_parameter * p)
{
	if (stream && p)
	{
		if (p->type->fprintf)
		{
			size_t i;
			for (i = 0; i < p->size; i++)
			{
				(p->type->fprintf)(stream, p->data[i]);
				fprintf (stream, "\n");
			}
		}
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

static int 
gdl_plot_parameter_fscanf (FILE * stream, gdl_plot_parameter * p)
{
	if (stream && p)
	{
		if (p->type->fscanf)
		{
			int c;
			size_t i;
			
			for (i = 0; i < p->size; i++)
			{
				p->data[i] = (p->type->fscanf)(stream);
				if (!p->data[i])
				{
					return GDL_EINVAL;
				}
				while (0==0)
				{
					c = fgetc(stream);
					if (c == '\n')
					{
						break;
					}
					if (c == EOF && i != p->size-1)
					{
						return GDL_SUCCESS;
					}
					else if (c == EOF && i < p->size-1)
					{
						return GDL_EINVAL;
					}
				}
			}
		}
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

static void
_gdl_plot_parameter_free (void * p)
{
	gdl_plot_parameter_free ((gdl_plot_parameter *)p);
}

static const gdl_data_interface gdl_plot_parameter_interface =
{
	&_gdl_plot_parameter_free,
	NULL,
	NULL,
	NULL,
	NULL
};

struct _gdl_plot_parameters
{
	gdl_hashtable * table;
};

gdl_plot_parameters *
gdl_plot_parameters_alloc (void)
{
	gdl_plot_parameters * par;
	
	par = GDL_MALLOC (gdl_plot_parameters, 1);
	
	par->table = gdl_hashtable_alloc (&gdl_plot_parameter_interface, 0);
	
	return par;
}

void
gdl_plot_parameters_free (gdl_plot_parameters * p)
{
	if (p)
	{
		gdl_hashtable_free (p->table);
		GDL_FREE (p);	
	}
}

gdl_plot_parameter *
gdl_plot_parameters_get (const gdl_plot_parameters * params, const gdl_string * name)
{
	return gdl_hashtable_lookup (params->table, name);
}

int
gdl_plot_parameters_set (gdl_plot_parameters * params, const gdl_string * name, gdl_plot_parameter * par)
{
	return gdl_hashtable_update (params->table, name, par, 1);
}

int
gdl_plot_parameters_fprintf (FILE * stream, const gdl_plot_parameters * par)
{
	if (stream && par)
	{
		if (gdl_hashtable_size (par->table))
		{
			gdl_hashtable_itr * itr;
			
			itr = gdl_hashtable_iterator (par->table);
			
			do
			{
				const gdl_string * key = gdl_hashtable_iterator_key (itr);
				gdl_plot_parameter * p = (gdl_plot_parameter *) gdl_hashtable_iterator_value (itr);
				if (p->type != gdl_plot_parameter_extra)
				{
					fprintf (stream, "<");
					fprintf (stream, key);
					fprintf (stream, " type=\"%s\" size=\"%d\">\n", p->type->name, p->size);
					gdl_plot_parameter_fprintf (stream, p);
					fprintf (stream, "<");
					fprintf (stream, key);
					fprintf (stream, "/>\n");
				}
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

static void
gdl_plot_parameters_fscanf_tag (const gdl_string * line, const size_t n, gdl_string ** key, gdl_string ** type, size_t * size)
{
	size_t i, j;
	
	gdl_string_free (*type);*type=NULL;
	gdl_string_free (*key);*key=NULL;
	
	for (i = 0; i < n;i++)
	{
		if (line[i] == '>')
		{
			break;	
		}
		if (line[i] == '<' && i+1 < n)
		{
			for (i++, j = 0; !isspace (line[i+j]); j++);
			*key = gdl_string_alloc (j);
			memcpy (*key, &line[i], sizeof (char)*j);
			i+=j-1;
		}
		if (line[i] == '=' && i+2 < n)
		{
			for (i+=2, j = 0; line[i+j] != '"'; j++);
			gdl_string * buf = gdl_string_alloc (j);
			memcpy (buf, &line[i], sizeof (char)*j);
			if (!(*type))
			{
				*type = buf;
			}
			else
			{
				
				*size = (size_t) atoi (buf);
				gdl_string_free (buf);
			}
			i+=j-1;
		}
	}
}

static const gdl_plot_parameter_type *
gdl_plot_parameter_type_sscanf (const gdl_string * name)
{
	if (name && !strcmp (name, gdl_plot_parameter_double->name))
	{
		return gdl_plot_parameter_double;
	}
	else if (name && !strcmp (name, gdl_plot_parameter_int->name))
	{
		return gdl_plot_parameter_int;
	}
	else if (name && !strcmp (name, gdl_plot_parameter_rgb->name))
	{
		return gdl_plot_parameter_rgb;
	}
	else if (name && !strcmp (name, gdl_plot_parameter_font->name))
	{
		return gdl_plot_parameter_font;
	}
	else if (name && !strcmp (name, gdl_plot_parameter_string->name))
	{
		return gdl_plot_parameter_string;
	}
	
	return NULL;
}

gdl_plot_parameters *
gdl_plot_parameters_fscanf (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i, size, n;
		gdl_string * line = NULL;
		gdl_string * key, * type;
		gdl_plot_parameters * params;
		gdl_plot_parameter  * par;
		const gdl_plot_parameter_type * T;
		
		params = gdl_plot_parameters_alloc ();
		
		status = gdl_getline (&line, &n, stream);
		
		while (status != -1)
		{
			if (n)
			{
				for (i = n-1; line[i]!='>' && i > 0; i--);
				if (i && line[i-1] != '/')
				{
					key = type = NULL;
					gdl_plot_parameters_fscanf_tag (line, n, &key, &type, &size);
					T   = gdl_plot_parameter_type_sscanf (type);
					par = gdl_plot_parameter_alloc (T, size);
					gdl_plot_parameter_fscanf (stream, par);
					gdl_plot_parameters_set (params, key, par);
					gdl_string_free (key);
					gdl_string_free (type);
					gdl_string_free (line);
				}
			}	
			line = NULL;
			status = gdl_getline (&line, &n, stream);
		}
		
		return params;
	}
	
	return NULL;
}

static void
gdl_plot_parameter_double_free (void * p)
{
	GDL_FREE (p);
}

static void *
gdl_plot_parameter_double_copy (const void * p)
{
	double * x = GDL_MALLOC (double, 1);
	*x = *((double *)p);
	return x;
}

static int
gdl_plot_parameter_double_fprintf (FILE * stream, const void * p)
{
	return fprintf (stream, "%f", *((double *)p));
}

static void *
gdl_plot_parameter_double_fscanf (FILE * stream)
{
	double * x = GDL_MALLOC (double, 1);
	float  y;
	fscanf (stream, "%f", &y);
	*x=(double)y;
	return x;
}

static const gdl_plot_parameter_type _gdl_plot_parameter_double =
{
	"double",
	sizeof (double),
	&gdl_plot_parameter_double_free,
	&gdl_plot_parameter_double_copy,
	&gdl_plot_parameter_double_fprintf,
	&gdl_plot_parameter_double_fscanf
};

static void
gdl_plot_parameter_int_free (void * p)
{
	GDL_FREE (p);
}

static void *
gdl_plot_parameter_int_copy (const void * p)
{
	int * x = GDL_MALLOC (int, 1);
	*x = *((int *)p);
	return x;
}

static int
gdl_plot_parameter_int_fprintf (FILE * stream, const void * p)
{
	return fprintf (stream, "%d", *((int *)p));
}

static void *
gdl_plot_parameter_int_fscanf (FILE * stream)
{
	int * x = GDL_MALLOC (int, 1);
	fscanf (stream, "%d", x);
	return x;
}

static const gdl_plot_parameter_type _gdl_plot_parameter_int =
{
	"int",
	sizeof (int),
	&gdl_plot_parameter_int_free,
	&gdl_plot_parameter_int_copy,
	&gdl_plot_parameter_int_fprintf,
	&gdl_plot_parameter_int_fscanf
};

static void
gdl_plot_parameter_rgb_free (void * p)
{
	gdl_rgb_color_free ((gdl_rgb_color *) p);
}

static void *
gdl_plot_parameter_rgb_copy (const void * p)
{
	gdl_rgb_color * x = gdl_rgb_color_alloc ();
	gdl_rgb_color_copy (x, (gdl_rgb_color *) p);
	return x;
}

static int
gdl_plot_parameter_rgb_fprintf (FILE * stream, const void * p)
{
	return gdl_rgb_color_fprintf (stream, (gdl_rgb_color *) p);
}

static void *
gdl_plot_parameter_rgb_fscanf (FILE * stream)
{
	return gdl_rgb_color_fscanf (stream);
}

static const gdl_plot_parameter_type _gdl_plot_parameter_rgb =
{
	"rgb",
	sizeof (gdl_rgb_color *),
	&gdl_plot_parameter_rgb_free,
	&gdl_plot_parameter_rgb_copy,
	&gdl_plot_parameter_rgb_fprintf,
	&gdl_plot_parameter_rgb_fscanf
};

static void
gdl_plot_parameter_font_free (void * p)
{
	gdl_font_free ((gdl_font *) p);
}

static void *
gdl_plot_parameter_font_copy (const void * p)
{
	gdl_font * x = gdl_font_alloc ();
	gdl_font_copy (x, (gdl_font *) p);
	return x;
}

static int
gdl_plot_parameter_font_fprintf (FILE * stream, const void * p)
{
	return gdl_font_fprintf (stream, (gdl_font *) p);
}

static void *
gdl_plot_parameter_font_fscanf (FILE * stream)
{
	return gdl_font_fscanf (stream);
}

static const gdl_plot_parameter_type _gdl_plot_parameter_font =
{
	"font",
	sizeof (gdl_font *),
	&gdl_plot_parameter_font_free,
	&gdl_plot_parameter_font_copy,
	&gdl_plot_parameter_font_fprintf,
	&gdl_plot_parameter_font_fscanf
};

static void
gdl_plot_parameter_string_free (void * p)
{
	gdl_string_free ((gdl_string *) p);
}

static void *
gdl_plot_parameter_string_copy (const void * p)
{
	if (p)
	{
		const gdl_string * s = (gdl_string *) p;
		gdl_string * x       = gdl_string_alloc (strlen(s));
		
		strcpy (x, s);
		
		return x;
	}
	
	return NULL;
}

static int
gdl_plot_parameter_string_fprintf (FILE * stream, const void * p)
{
	return fprintf (stream, "%s", (gdl_string *) p);
}

static void *
gdl_plot_parameter_string_fscanf (FILE * stream)
{
	int c;
	long offset;
	size_t n;
	gdl_string * s = NULL;
	
	offset = ftell (stream);
	c      = fgetc (stream);
	
	while (c != EOF)
	{
		if (c == '\n')
		{
			break;
		}
		else
		{
			gdl_string_cat (&s, c);
		}
		offset = ftell (stream);
		c      = fgetc (stream);
	}
	
	fseek (stream, offset, SEEK_SET);
	
	return s;
}

static const gdl_plot_parameter_type _gdl_plot_parameter_string =
{
	"string",
	sizeof (gdl_string *),
	&gdl_plot_parameter_string_free,
	&gdl_plot_parameter_string_copy,
	&gdl_plot_parameter_string_fprintf,
	&gdl_plot_parameter_string_fscanf
};

static const gdl_plot_parameter_type _gdl_plot_parameter_extra =
{
	"extra",
	sizeof (void *),
	NULL,
	NULL,
	NULL,
	NULL
};

const gdl_plot_parameter_type * gdl_plot_parameter_double = &_gdl_plot_parameter_double;
const gdl_plot_parameter_type * gdl_plot_parameter_int    = &_gdl_plot_parameter_int;
const gdl_plot_parameter_type * gdl_plot_parameter_rgb    = &_gdl_plot_parameter_rgb;
const gdl_plot_parameter_type * gdl_plot_parameter_font   = &_gdl_plot_parameter_font;
const gdl_plot_parameter_type * gdl_plot_parameter_string  = &_gdl_plot_parameter_string;
const gdl_plot_parameter_type * gdl_plot_parameter_extra   = &_gdl_plot_parameter_extra;
