/*  
 *  pstruct/result.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gstats.h>
#include <gdl/gdl_glabels.h>
#include <gdl/gdl_sort_double.h>
#include <gdl/gdl_pstruct.h>
#include <gdl/gdl_pstruct_result.h>
#include <gdl/gdl_pstruct_criterion.h>

static gdl_glabels *
_gdl_pstruct_result_get_labels (const gdl_pstruct_workspace * work)
{
	if (work->type    == gdl_pstruct_em_gadmixture
	    || work->type == gdl_pstruct_em_gmixture)
	{
		return gdl_glabels_wrapper_alloc (work->gwrap, gdl_false, gdl_true);
	}
	else if (work->type == gdl_pstruct_em_hadmixture)
	{
		return gdl_hlabels_alloc (work->hview, gdl_false, gdl_true);
	}
	return 0;	
}

gdl_pstruct_result *
gdl_pstruct_result_alloc (const gdl_pstruct_workspace * work)
{
	size_t i, j, k, * tmp;
	gdl_pstruct_result * r;
	
	r = GDL_CALLOC (gdl_pstruct_result, 1);
	
	r->type = work->type;
	
   r->k    = gdl_pstruct_workspace_population_size (work);
  	r->p    = gdl_pstruct_workspace_ploidy (work);
  	r->n    = gdl_pstruct_workspace_accession_size (work);
  	r->l    = gdl_pstruct_workspace_locus_size (work);
  	r->na   = GDL_MALLOC (size_t, r->l);
  	
  	for (r->tna = 0, i = 0; i < r->l; i++)
  	{
  		r->na[i] = gdl_pstruct_workspace_get_locus_f_size (work, i);
  		r->tna  += r->na[i];
  	}
  	
  	tmp    = GDL_MALLOC (size_t, 2);
	tmp[0] = r->k;
	tmp[1] = r->l;
	
	r->f  = gdl_hnblock_alloc (1, 1, tmp, &r->na);
	r->q  = gdl_block_alloc2 (2, r->k, r->n);
	r->pi = GDL_CALLOC (double, r->k);
	
	GDL_FREE (tmp);
  	
  	r->loglikelihood = gdl_pstruct_workspace_loglikelihood (work);
  	
  	r->residual      = work->residual;
  	
  	r->tw_residual   = work->tw_residual;
  	
  	for (i = 0; i < r->n; i++)
  	{
  		for (k = 0; k < r->k; k++)
  		{
  			gdl_block_set (r->q, k, i, gdl_pstruct_workspace_get_accession_q (work, k, i));
  			r->pi[k] += (gdl_block_get (r->q, k, i) - r->pi[k])/(i+1);
  		}
  	}
  	for (i = 0; i < r->l; i++)
  	{
  		for (j = 0; j < r->na[i]; j++)
  		{
	  		for (k = 0; k < r->k; k++)
	  		{
	  			gdl_hnblock_set (r->f, k, i, j, gdl_pstruct_workspace_get_locus_f (work, k, i, j));
	  		}
  		}
  	}
  	
  	r->labels = _gdl_pstruct_result_get_labels (work); 
 
	return r;	
}

void
gdl_pstruct_result_free (gdl_pstruct_result * r)
{
	if (r)
	{
		GDL_FREE (r->na);
		GDL_FREE (r->pi);
		gdl_hnblock_free (r->f);
		gdl_block_free (r->q);
		GDL_FREE (r);
	}
}

const gdl_glabels *
gdl_pstruct_result_labels (const gdl_pstruct_result * r)
{
	return r->labels;	
}

double
gdl_pstruct_result_loglikelihood (const gdl_pstruct_result * r)
{
	return r->loglikelihood;	
}

size_t
gdl_pstruct_result_population_size (const gdl_pstruct_result * r)
{
	return r->k;	
}

size_t
gdl_pstruct_result_accession_size (const gdl_pstruct_result * r)
{
	return r->n;
}

size_t
gdl_pstruct_result_locus_size (const gdl_pstruct_result * r)
{
	return r->l;	
}

size_t
gdl_pstruct_result_parameter_size (const gdl_pstruct_result * r)
{
	if (r->type == gdl_pstruct_em_gmixture)
	{
		return (r->tna-r->l+1)*r->k-1;
	}
	else if (r->type == gdl_pstruct_em_gadmixture)
	{
		return (r->tna-r->l+r->n)*r->k-r->n;
	}
	else if (r->type == gdl_pstruct_em_hadmixture)
	{
		return (r->tna-r->l+r->n)*r->k-r->n;
	}
	
	return 0;
}

double
gdl_pstruct_result_get_population_q (const gdl_pstruct_result * r, size_t pop)
{
	return r->pi[pop];
}

double
gdl_pstruct_result_get_population_divergence (const gdl_pstruct_result * r, size_t p1, size_t p2)
{
	size_t i, j, l;
	double d, * f1, * f2;
	
	f1 = GDL_MALLOC (double, r->tna);
	f2 = GDL_MALLOC (double, r->tna);
		
	for (l = i = 0; i < r->l; i++)
	{
		for (j = 0; j < r->na[i]; j++, l++)
		{
			f1[l] = gdl_hnblock_get (r->f, p1, i, j);
			f2[l] = gdl_hnblock_get (r->f, p2, i, j);
		}	
	}
	
	d = gdl_gstats_distance_kullback_leibler (f1, f2, r->tna);
	
	GDL_FREE (f1);
	GDL_FREE (f2);
	
	return d/r->l;
}

double
gdl_pstruct_result_get_accession_q (const gdl_pstruct_result * r, size_t pop, size_t indiv)
{
	return gdl_block_get (r->q, pop, indiv);	
}

static double
_gdl_pstruct_result_get_accession_q (const void * r, size_t pop, size_t indiv)
{
	return gdl_pstruct_result_get_accession_q ((gdl_pstruct_result *) r, pop, indiv);
}

double
gdl_pstruct_result_get_locus_f (const gdl_pstruct_result * r, size_t pop, size_t loc, size_t allele)
{
	return gdl_hnblock_get (r->f, pop, loc, allele);
}

static double
_gdl_pstruct_result_get_locus_f  (const void * r, size_t pop, size_t loc, size_t allele)
{
	return gdl_pstruct_result_get_locus_f ((gdl_pstruct_result *) r, pop, loc, allele);
}

size_t
gdl_pstruct_result_get_locus_f_size (const gdl_pstruct_result * r, size_t loc)
{
	return r->na[loc];	
}

double
gdl_pstruct_result_search_accession_q (const gdl_pstruct_result * r, size_t pop, const gdl_accession * accession)
{
	int i = gdl_glabels_search_accession (r->labels, gdl_entity_get_name (accession));
	if (i>=0)
	{
		return gdl_pstruct_result_get_accession_q (r, pop, i);
	}
	return GDL_NAN;
}

double
gdl_pstruct_result_search_locus_f (const gdl_pstruct_result * r, size_t pop, const gdl_locus * locus, const gdl_allele * allele)
{
	int i = gdl_glabels_search_locus (r->labels, gdl_entity_get_name (locus));
	if (i>=0)
	{
		int j = gdl_glabels_search_allele (r->labels, i, gdl_entity_get_name (allele));
		if (j>=0)
		{
			return gdl_pstruct_result_get_locus_f (r, pop, i, j);
		}
	}
	return GDL_NAN;		
}

size_t
gdl_pstruct_result_get_accession_q_max_index (const gdl_pstruct_result * r, size_t i)
{
	size_t k, maxk = 0;
	double max = 0.;
	
	for (k = 0; k < r->k; k++)
	{
		if (max < gdl_pstruct_result_get_accession_q (r, k, i))
		{
			max = gdl_pstruct_result_get_accession_q (r, k, i);
			maxk = k;
		}
	}
	
	return maxk;
}

size_t *
gdl_pstruct_result_qsort (const gdl_pstruct_result * r)
{
	size_t i, * ii, j, k, maxk, nk, na, ng, nt, * idx, * sidx, * tidx;
	double x, * q;
	gdl_list     * group;
	gdl_list_itr * itr;
	
	na = gdl_pstruct_result_accession_size (r);
	nk = gdl_pstruct_result_population_size (r);
	nt = 0;
	tidx = GDL_MALLOC (size_t, na);
	
	for (j = 0; j < nk; j++)
	{
		group = gdl_list_alloc (gdl_interface_uint);
		
		for (i = 0; i < na; i++)
		{
			if (gdl_pstruct_result_get_accession_q_max_index (r, i) == j)
			{
				ii = GDL_MALLOC (size_t, 1);
				*ii = i;
				gdl_list_push_back (group, ii, 1);
			}
		}
		
		if ((ng=gdl_list_size (group))!=0)
		{
			q    = GDL_MALLOC (double, ng);
			idx  = GDL_MALLOC (size_t, ng);
			sidx = GDL_MALLOC (size_t, ng);
			
			i   = 0;
			itr = gdl_list_iterator_front (group);
			do
			{
				ii     = (size_t *) gdl_list_iterator_value (itr);
				idx[i] = *ii;
				q[i]   = gdl_pstruct_result_get_accession_q (r, j, idx[i]);
				i++;
			}
			while (gdl_list_iterator_next (itr));
			
			gdl_list_iterator_free (itr);
			
			gdl_sort_index (sidx, q, 1, ng);
			
			for (i = 0; i < ng/2; i++)
			{
				k = sidx[i];
				sidx[i]      = idx[sidx[ng-1-i]];
				sidx[ng-1-i] = idx[k];
			}
			if (ng % 2 == 1)
			{
				sidx[ng/2] = idx[sidx[ng/2]];	
			}
			
			memcpy (&tidx[nt], sidx, sizeof (size_t)*ng);
			
			nt += ng;

			GDL_FREE (q);
			GDL_FREE (idx);
			GDL_FREE (sidx);
		}
		
		gdl_list_free (group);
	}
	
	return tidx;
}

gdl_pstruct_workspace *
gdl_pstruct_result_gupdate (const gdl_pstruct_result * r, gdl_gview_wrapper * gw, gdl_rng * rng)
{
	size_t i, j, n, na;
	int tmp, * aidx, * lidx;
	gdl_pstruct_workspace * w;
	
	n    = gdl_gview_wrapper_accession_size_c (gw);
	aidx = GDL_MALLOC (int, n);
	for (i = 0; i < n; i++)
	{
		gdl_accession * acc = gdl_gview_wrapper_get_accession_c (gw, i);
		aidx[i] = gdl_glabels_search_accession (r->labels, gdl_entity_get_name (acc));
		if (aidx[i] < 0)
		{
			printf ("New Accession %d %s ==> UPDATE\n", i, gdl_entity_get_name (acc));
		}
	}
	n = gdl_gview_wrapper_locus_size (gw);
	lidx = GDL_MALLOC (int, n);
	for (i = 0; i < n; i++)
	{
		gdl_locus * locus = gdl_gview_wrapper_get_locus (gw, i);
		lidx[i]           = gdl_glabels_search_locus (r->labels, gdl_entity_get_name (locus));
		if (lidx[i]<0)
		{
			printf ("New Locus %d %s ==> UPDATE\n", i, gdl_entity_get_name (locus));
		}
		if (lidx[i]>=0)
		{
			na = gdl_locus_allele (locus);
			if (na != r->na[i])
			{
				lidx[i]=-1;
				printf ("Locus %d %s: Number of alleles has changed (%d, %d)==> UPDATE\n", i, gdl_entity_get_name (locus), na, r->na[i]);	
				continue;
			}
			for (j = 0; j < na; j++)
			{
				gdl_allele * allele = gdl_locus_get_allele (locus, j);
				tmp = gdl_glabels_search_allele (r->labels, lidx[i], gdl_entity_get_name (allele));
				if (tmp != gdl_entity_get_idx (allele))
				{
					lidx[i]=-1;
					printf ("Locus %d %s Allele index has changed ==> UPDATE\n", i, gdl_entity_get_name (locus));	
					break;
				}
			}
		}
	}
	
	w = gdl_pstruct_workspace_galloc (r->type, gw, rng, r->k);
	
	gdl_pstruct_workspace_init_update (w, aidx, lidx, r, &_gdl_pstruct_result_get_accession_q, &_gdl_pstruct_result_get_locus_f);
	
	GDL_FREE (aidx);
	GDL_FREE (lidx);
		
	return w;
}

gdl_pstruct_workspace *
gdl_pstruct_result_hupdate (const gdl_pstruct_result * r, gdl_hview * h, gdl_rng * rng)
{
	return NULL;
}

static const gdl_pstruct_workspace_type *
_gdl_pstruct_result_type_fread (FILE * stream)
{
	gdl_string * name = gdl_string_fread (stream);
	
	if (!strcmp (name, gdl_pstruct_em_gadmixture->name))
	{
		return gdl_pstruct_em_gadmixture;
	}
	if (!strcmp (name, gdl_pstruct_em_gmixture->name))
	{
		return gdl_pstruct_em_gmixture;
	}
	else if (!strcmp (name, gdl_pstruct_em_hadmixture->name))
	{
		return gdl_pstruct_em_hadmixture;
	}
	
	return NULL;
}

static int
_gdl_pstruct_result_type_fwrite (FILE * stream, const gdl_pstruct_workspace_type * T)
{
	return gdl_string_fwrite (stream, T->name);
}

gdl_pstruct_result *
gdl_pstruct_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_pstruct_result * r = GDL_CALLOC (gdl_pstruct_result, 1); 
		
		r->type = _gdl_pstruct_result_type_fread (stream);
   	GDL_FREAD_STATUS (r->type!=0, 1);
   	status = fread (&(r->k), sizeof (size_t), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
   	status = fread (&(r->p), sizeof (size_t), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
   	status = fread (&(r->n), sizeof (size_t), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
   	status = fread (&(r->l), sizeof (size_t), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
   	r->na  = GDL_MALLOC (size_t, r->l);
   	status = fread (r->na, sizeof (size_t), r->l, stream);
   	GDL_FREAD_STATUS (status, r->l);
   	status = fread (&(r->tna), sizeof (size_t), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
   	status = fread (&(r->loglikelihood), sizeof (double), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
   	status = fread (&(r->residual), sizeof (double), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
   	status = fread (&(r->tw_residual), sizeof (double), 1, stream);
   	GDL_FREAD_STATUS (status, 1);
      r->f = gdl_hnblock_fread (stream);
	  	GDL_FREAD_STATUS (r->f!=0, 1);
  		r->q = gdl_block_fread (stream);
	  	GDL_FREAD_STATUS (r->q!=0, 1);
	  	r->pi = GDL_MALLOC (double, r->k);
	  	status = fread (r->pi, sizeof (double), r->k, stream);
	  	GDL_FREAD_STATUS (status, r->k);
	  	r->labels = gdl_glabels_fread (stream);
	  	GDL_FREAD_STATUS (r->labels!=0, 1);
	  	
	  	return r;
	}
	
	return NULL;
}

int
gdl_pstruct_result_fwrite (FILE * stream, const gdl_pstruct_result * r)
{
	if (stream && r)
	{
		int status;
		
		status = _gdl_pstruct_result_type_fwrite (stream, r->type);
   	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
   	status = fwrite (&(r->k), sizeof (size_t), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
   	status = fwrite (&(r->p), sizeof (size_t), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
   	status = fwrite (&(r->n), sizeof (size_t), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
   	status = fwrite (&(r->l), sizeof (size_t), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
   	status = fwrite (r->na, sizeof (size_t), r->l, stream);
   	GDL_FWRITE_STATUS (status, r->l);
   	status = fwrite (&(r->tna), sizeof (size_t), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
   	status = fwrite (&(r->loglikelihood), sizeof (double), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
   	status = fwrite (&(r->residual), sizeof (double), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
   	status = fwrite (&(r->tw_residual), sizeof (double), 1, stream);
   	GDL_FWRITE_STATUS (status, 1);
      status = gdl_hnblock_fwrite (stream, r->f);
	  	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
  		status = gdl_block_fwrite (stream, r->q);
	  	GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	  	status = fwrite (r->pi, sizeof (double), r->k, stream);
   	GDL_FWRITE_STATUS (status, r->k);
   	status = gdl_glabels_fwrite (stream, r->labels);
	  	GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

int
gdl_pstruct_result_fprintf (FILE * stream, const gdl_pstruct_result * r)
{
	if (stream && r)
	{
		size_t i, j, k, l;
		gdl_pstruct_criterion * criterion = gdl_pstruct_criterion_alloc (r);
		
		fprintf (stream, "Parameters:\n");
		fprintf (stream, "\tploidy %d\n", r->p);
		fprintf (stream, "\t%d individuals\n", r->n);
		fprintf (stream, "\t%d loci\n", r->l);
		fprintf (stream, "\t%d populations assumed\n", r->k);
		fprintf (stream, "\n--------------------------------------------\n\n");
		fprintf (stream, "Overall proportion of membership of the\n");
		fprintf (stream, "sample in each of the %d clusters\n\n", r->k);
		fprintf (stream, "Inferred Clusters\n");
		for (k = 0; k < r->k; k++)
		{
			fprintf (stream, "\t%d", k+1);
		}
		fprintf (stream, "\n");
		for (k = 0; k < r->k; k++)
		{
			fprintf (stream, "\t%1.4f", r->pi[k]);
		}
		fprintf (stream, "\n");
		fprintf (stream, "\n--------------------------------------------\n\n");
		fprintf (stream, "Allele-freq. divergence among pops (Kullback-Leibler distance),\n");
      fprintf (stream, "computed using point estimates of P.\n\n");
		for (k = 0; k < r->k; k++)
		{
			fprintf (stream, "\t%d", k+1);
		}
		fprintf (stream, "\n");
		for (i = 0; i < r->k; i++)
		{
			fprintf (stream, "%d", i+1);
			for (k = 0; k < r->k; k++)
			{
				if (k == i)
				{
					fprintf (stream, "\t-");
				}
				else
				{
					fprintf (stream, "\t%1.3f", gdl_pstruct_result_get_population_divergence (r, i, k));
				}
			}
			fprintf (stream, "\n");
		}
      fprintf (stream, "\n--------------------------------------------\n\n");
		fprintf (stream, "Ln Prob of Data = %.1f\n", r->loglikelihood);
		fprintf (stream, "            AIC = %.1f\n", gdl_pstruct_criterion_value (criterion, gdl_pstruct_criterion_AIC));
		fprintf (stream, "            BIC = %.1f\n", gdl_pstruct_criterion_value (criterion, gdl_pstruct_criterion_BIC));
		if (!gdl_isnan (r->residual))
		{
			fprintf (stream, "Residual LD     = %.5f (%.5f)\n", r->residual, r->tw_residual);
		}
		fprintf (stream, "\n");
		fprintf (stream, "Inferred ancestry of individuals:\n");
      fprintf (stream, "\tLabel\t:\tInferred clusters\n");
		for (i = 0; i < r->n; i++)
		{
			 fprintf (stream, "\n%d\t%s", i+1, gdl_glabels_accession (r->labels, i));	
			 for (k = 0; k < r->k; k++)
			 {
			 	 fprintf (stream, "\t%1.4f", gdl_pstruct_result_get_accession_q (r, k, i));
			 }
		}
		fprintf (stream, "\n\n--------------------------------------------\n\n");
 		fprintf (stream, "Estimated Allele Frequencies in each population\n");
 		for (i = 0; i < r->l; i++)
		{
			fprintf (stream, "\nLocus %d %s:\n", i+1, gdl_glabels_locus (r->labels, i));
			fprintf (stream, "%d alleles\n", r->na[i]);
			for (j = 0; j < r->na[i]; j++)
			{
				fprintf (stream, "\t%d\t%s", j+1, gdl_glabels_allele (r->labels, i, j));
				for (k = 0; k < r->k; k++)
				{
					fprintf (stream, "\t%1.4f", gdl_pstruct_result_get_locus_f (r, k, i, j));	
				}
				fprintf (stream, "\n");
			}
		}
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

gdl_pstruct_result *
gdl_pstruct_result_structure_read (FILE * stream, const gdl_gview * data)
{
   if (!stream)
   {
   	return 0;
   }
   
   int ch, idx, idx2;
   double x, s;
   size_t i, j, n, * n2;
   gdl_string * line=0;
   gdl_string * tmp;
   gdl_pstruct_result * result;

   result = GDL_CALLOC (gdl_pstruct_result, 1);
   
   result->p = gdl_gview_ploidy (data);
   result->n = gdl_gview_accession_size (data);
   result->l = gdl_gview_locus_size (data);
   result->na = GDL_MALLOC (size_t, result->l);
   result->tna = 0;
   for (i = 0; i < result->l; i++)
	{
		const gdl_locus * locus = gdl_gview_get_locus (data, i);
		result->na[i] = gdl_locus_allele (locus);
		result->tna += result->na[i];
	}
	// labels
	result->labels = gdl_glabels_alloc (data, 0, 0, gdl_true);

#define Params   "Run parameters:"
#define LnProba  "Estimated Ln Prob of Data   ="
#define Ancestry "Inferred ancestry of individuals:"
#define Locus    "Locus "
#define Values   "Values of parameters used in structure:"
 	
 	for(;;)
 	{  
 		if (gdl_getline (&line, &n, stream)==-1)
 			break;
 		if (n)
 		{
 			if (!strcmp(line, Params))
 			{
 				//printf ("%s\n", strtok (line, " "));
 				for(i=0;i<3;i++) {gdl_string_free(line);line=0;gdl_getline (&line, &n, stream);}
 				//printf ("K = %s\n", strtok (line, " "));
 				result->k = atoi(strtok (line, " "));
 				// alloc memory
 				n2    = GDL_MALLOC (size_t, 2);
				n2[0] = result->k;
				n2[1] = result->l;
				result->f  = gdl_hnblock_alloc (1, 1, n2, &result->na);
				result->q  = gdl_block_alloc2 (2, result->k, result->n);
				result->pi = GDL_CALLOC (double, result->k);
				GDL_FREE (n2);
 			}
 			else if (!strncmp (line, LnProba, strlen(LnProba)))
 			{
 				result->loglikelihood = (double)atof(&line[strlen(LnProba)]);
 			}
 			else if (!strcmp (line, Ancestry))
 			{
 				for (i=0;i<2;i++) {gdl_string_free(line);line=0;gdl_getline (&line, &n, stream);}
 				for (i = 0; i < result->n; i++)
 				{
	 				tmp = strtok(line, " ");
	 				tmp = strtok(0, " ");
	 				// Label
	 				idx = gdl_glabels_search_accession (result->labels, tmp); 
	 				for(j = 0; j < 3; j++) tmp = strtok(0, " ");
	 				// Ancestry
	 				for(s = j = 0; j < result->k; j++)
	 				{
	 					x = (double)atof(tmp);
	 					gdl_block_set (result->q, j, idx, x);
	 					result->pi[j] += x/result->n;
	 					s+=x;
	 					tmp = strtok(0, " ");
	 				}
	 				for(j = 0; j < result->k; j++)
	 				{
	 					x = gdl_block_get (result->q, j, idx);
	 					gdl_block_set (result->q, j, idx, x/s);
	 				}
	 				gdl_string_free(line);line=0;
	 				gdl_getline (&line, &n, stream);
 				}	 				
 			}
			else if (!strncmp (line, Locus, strlen(Locus)))
			{
				tmp=strtok(line, " ");tmp=strtok(0, " ");
				idx=atoi(tmp)-1;
				for (i = 0; i < 3; i++) {gdl_string_free(line);line=0;gdl_getline (&line, &n, stream);}
 				for (i = 0; i < result->na[idx]; i++)
 				{
 					tmp = strtok(line, " ");
 					idx2 = gdl_glabels_search_allele (result->labels, idx, tmp);
 					if (idx2==-1)
 					{
 						for (j = 0; j < result->na[idx]; j++)
 						{
 							if (atoi(tmp) == atoi(gdl_glabels_allele (result->labels, idx, j)))
 							{
 								break;
 							}
 						}
 						if (j < result->na[idx])
 						{
 							idx2 = j;	
 						}
 						else
 						{
 							GDL_ERROR_VAL ("Unmatched allele names between data and file", GDL_FAILURE, 0);
 						}
 					}
 					tmp = strtok(0, " ");
 					if (tmp[0]=='(') tmp = strtok(0, " ");
 					for (j = 0; j < result->k; j++)
 					{
 						x = (double)atof(tmp);
 						gdl_hnblock_set (result->f, j, idx, idx2, x);
 						tmp = strtok(0, " ");
 					}
 					gdl_string_free(line);line=0;
 					gdl_getline (&line, &n, stream);	
 				}
 				for (j = 0; j < result->k; j++)
 				{
	 				for (s = i = 0; i < result->na[idx]; i++)
	 				{
	 					s += gdl_hnblock_get (result->f, j, idx, i);	
	 				}
	 				for (i = 0; i < result->na[idx]; i++)
	 				{
	 					x = gdl_hnblock_get (result->f, j, idx, i);
	 					gdl_hnblock_set (result->f, j, idx, i, x/s);
	 				}
 				}
			}
		   else if (!strcmp (line, Values))
		   {
		   	gdl_string_free(line);line=0;
		   	gdl_getline (&line, &n, stream);
 				tmp = strtok(line, "	,");
 				while(tmp && strncmp(tmp, "NOADMIX", 7))
 				{
 					tmp = strtok(0, "	,");
 				}
 				strtok(tmp, "=");
 				if (atoi(strtok(0, "=")))
 				{
 					result->type = gdl_pstruct_em_gmixture;
 				}
 				else
 				{
 					result->type = gdl_pstruct_em_gadmixture;
 				}
		   }
 		}
 		gdl_string_free (line);line=0;
 	}
 	
 	return result;
}

gdl_pstruct_result *
gdl_pstruct_result_qmatrix_read (FILE * stream, const gdl_gview * data)
{
   if (!stream)
   {
   	return 0;
   }
   
   int ch, idx, idx2;
   double x, s;
   size_t i, j, n, * n2;
   gdl_string * line=0;
   gdl_string * tmp;
   gdl_pstruct_result * result;

   result = GDL_CALLOC (gdl_pstruct_result, 1);
   
   result->p   = gdl_gview_ploidy (data);
   result->n   = gdl_gview_accession_size (data);
   result->l   = gdl_gview_locus_size (data);
   result->na  = GDL_MALLOC (size_t, result->l);
   result->tna = 0;
   for (i = 0; i < result->l; i++)
	{
		const gdl_locus * locus = gdl_gview_get_locus (data, i);
		result->na[i] = gdl_locus_allele (locus);
		result->tna  += result->na[i];
	}
	// labels
	result->labels = gdl_glabels_alloc (data, 0, 0, gdl_true);

 	for(;;)
 	{  
 		if (gdl_getline (&line, &n, stream)==-1)
 			break;
 		if (n)
 		{
 			if (!strcmp(line, Params))
 			{
 				//printf ("%s\n", strtok (line, " "));
 				for(i=0;i<3;i++) {gdl_string_free(line);line=0;gdl_getline (&line, &n, stream);}
 				//printf ("K = %s\n", strtok (line, " "));
 				result->k = atoi(strtok (line, " "));
 				// alloc memory
 				n2    = GDL_MALLOC (size_t, 2);
				n2[0] = result->k;
				n2[1] = result->l;
				result->f  = gdl_hnblock_alloc (1, 1, n2, &result->na);
				result->q  = gdl_block_alloc2 (2, result->k, result->n);
				result->pi = GDL_CALLOC (double, result->k);
				GDL_FREE (n2);
 			}
 			else if (!strncmp (line, LnProba, strlen(LnProba)))
 			{
 				result->loglikelihood = (double)atof(&line[strlen(LnProba)]);
 			}
 			else if (!strcmp (line, Ancestry))
 			{
 				for (i=0;i<2;i++) {gdl_string_free(line);line=0;gdl_getline (&line, &n, stream);}
 				for (i = 0; i < result->n; i++)
 				{
	 				tmp = strtok(line, " ");
	 				tmp = strtok(0, " ");
	 				// Label
	 				idx = gdl_glabels_search_accession (result->labels, tmp); 
	 				for(j = 0; j < 3; j++) tmp = strtok(0, " ");
	 				// Ancestry
	 				for(s = j = 0; j < result->k; j++)
	 				{
	 					x = (double)atof(tmp);
	 					gdl_block_set (result->q, j, idx, x);
	 					result->pi[j] += x/result->n;
	 					s+=x;
	 					tmp = strtok(0, " ");
	 				}
	 				for(j = 0; j < result->k; j++)
	 				{
	 					x = gdl_block_get (result->q, j, idx);
	 					gdl_block_set (result->q, j, idx, x/s);
	 				}
	 				gdl_string_free(line);line=0;
	 				gdl_getline (&line, &n, stream);
 				}	 				
 			}
			else if (!strncmp (line, Locus, strlen(Locus)))
			{
				tmp=strtok(line, " ");tmp=strtok(0, " ");
				idx=atoi(tmp)-1;
				for (i = 0; i < 3; i++) {gdl_string_free(line);line=0;gdl_getline (&line, &n, stream);}
 				for (i = 0; i < result->na[idx]; i++)
 				{
 					tmp = strtok(line, " ");
 					idx2 = gdl_glabels_search_allele (result->labels, idx, tmp);
 					if (idx2==-1)
 					{
 						for (j = 0; j < result->na[idx]; j++)
 						{
 							if (atoi(tmp) == atoi(gdl_glabels_allele (result->labels, idx, j)))
 							{
 								break;
 							}
 						}
 						if (j < result->na[idx])
 						{
 							idx2 = j;	
 						}
 						else
 						{
 							GDL_ERROR_VAL ("Unmatched allele names between data and file", GDL_FAILURE, 0);
 						}
 					}
 					tmp = strtok(0, " ");
 					if (tmp[0]=='(') tmp = strtok(0, " ");
 					for (j = 0; j < result->k; j++)
 					{
 						x = (double)atof(tmp);
 						gdl_hnblock_set (result->f, j, idx, idx2, x);
 						tmp = strtok(0, " ");
 					}
 					gdl_string_free(line);line=0;
 					gdl_getline (&line, &n, stream);	
 				}
 				for (j = 0; j < result->k; j++)
 				{
	 				for (s = i = 0; i < result->na[idx]; i++)
	 				{
	 					s += gdl_hnblock_get (result->f, j, idx, i);	
	 				}
	 				for (i = 0; i < result->na[idx]; i++)
	 				{
	 					x = gdl_hnblock_get (result->f, j, idx, i);
	 					gdl_hnblock_set (result->f, j, idx, i, x/s);
	 				}
 				}
			}
		   else if (!strcmp (line, Values))
		   {
		   	gdl_string_free(line);line=0;
		   	gdl_getline (&line, &n, stream);
 				tmp = strtok(line, "	,");
 				while(tmp && strncmp(tmp, "NOADMIX", 7))
 				{
 					tmp = strtok(0, "	,");
 				}
 				strtok(tmp, "=");
 				if (atoi(strtok(0, "=")))
 				{
 					result->type = gdl_pstruct_em_gmixture;
 				}
 				else
 				{
 					result->type = gdl_pstruct_em_gadmixture;
 				}
		   }
 		}
 		gdl_string_free (line);line=0;
 	}
 	
 	return result;
}
