/*  
 *  result/result.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:41 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <stdio.h>  

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_meta.h>
#include <gdl/gdl_odb.h>
#include <gdl/gdl_result.h>

static const gdl_odriver *
gdl_result_odriver_fread (FILE * stream)
{
	gdl_odriver_registry * registry = gdl_odriver_registry_get ();
	gdl_string * name;
	
	name = gdl_string_fread (stream);
	GDL_FREAD_STATUS (name!=0, 1);
	
	return gdl_odriver_registry_get_driver (registry, name);
}

static int
gdl_result_odriver_fwrite (FILE * stream, const gdl_odriver * driver)
{
	return gdl_string_fwrite (stream, driver->name);
}

gdl_result *
gdl_result_alloc (const gdl_string * data_file,
                  const gdl_string * command,
                  void * object,
                  const gdl_odriver * driver)
{
	gdl_result * r;
	
	r = GDL_CALLOC (gdl_result, 1);
	
	r->data_file = gdl_string_clone (data_file);
	
	r->command   = gdl_string_clone (command);
	
	r->object = object;
	
	r->driver = driver;
	
	return r;	
}
           
void
gdl_result_free (gdl_result * r)
{
	if (r)
	{
		GDL_FREE (r->data_file);
		GDL_FREE (r);	
	}	
}

void *
gdl_result_object (gdl_result * r)
{
	return r->object;	
}

const gdl_odriver *
gdl_result_odriver (gdl_result * r)
{
	return r->driver;	
}

gdl_string *
gdl_result_data_file (gdl_result * r)
{
	return r->data_file;	
}

gdl_string *
gdl_result_command (gdl_result * r)
{
	return r->command;	
}

static gdl_meta * 
_gdl_result_meta (const void * obj)
{
	gdl_result * result = (gdl_result *) obj;
	
	return (result->driver->meta)(result->object);
}

static void * 
_gdl_result_fread (FILE * stream)
{
	if (stream)
	{
		gdl_result * result = GDL_CALLOC (gdl_result, 1);
		
		result->data_file = gdl_string_fread (stream);
		GDL_FREAD_STATUS (result->data_file!=0, 1);
		result->command = gdl_string_fread (stream);
		GDL_FREAD_STATUS (result->command!=0, 1);
		result->driver    = gdl_result_odriver_fread (stream);
		GDL_FREAD_STATUS (result->driver!=0, 1);
		result->object    = (result->driver->fread)(stream);
		GDL_FREAD_STATUS (result->object!=0, 1);
		
		return result;
	}
	return NULL;
}

static int 
_gdl_result_fwrite (FILE * stream, const void * obj)
{
	if (stream && obj)
	{
		int status;
		gdl_result * result = (gdl_result *) obj;
		
		status = gdl_string_fwrite (stream, result->data_file);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_string_fwrite (stream, result->command);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		status = gdl_result_odriver_fwrite (stream, result->driver);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = (result->driver->fwrite)(stream, result->object);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

static const gdl_odriver _gdl_result_driver =
{
	"gdl_result",
   "1.0",
   &_gdl_result_meta,
   &_gdl_result_fread,
   &_gdl_result_fwrite,
   NULL
};

const gdl_odriver * gdl_result_driver = &_gdl_result_driver;
