/*  
 *  rho/data.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_rho.h>

gdl_rho_data *
gdl_rho_data_alloc (const gdl_chromosome * chrom, gdl_gview_wrapper * data)
{	
	size_t i;
	gdl_rho_data * d;
	
	d = GDL_CALLOC (gdl_rho_data, 1);
	
	d->chrom = chrom;
	
	d->N    = gdl_gview_wrapper_accession_size (data);
	d->L    = gdl_gview_wrapper_locus_size (data);
	d->seq  = gdl_gview_wrapper_allele_block (data, 0);
	d->data = data;
	d->NA   = GDL_MALLOC (size_t, d->L);
	d->D    = GDL_MALLOC (double, d->L-1);
	d->idx  = GDL_MALLOC (size_t, d->N);
	 
	for (i = 0; i < d->L; i++)
	{
		if (i)
		{
		   gdl_gdistance * di = gdl_chromosome_get_distance (chrom, i-1, i);
			if (gdl_gdistance_is_defined (di))
			{
				d->D[i-1] = di->value;
			}
			else
			{
				d->D[i-1] = -1.0;
			}
			gdl_gdistance_free (di);
		}
		d->NA[i] = gdl_locus_allele (gdl_gview_wrapper_get_locus (data, i));
	}
	for (i = 0; i < d->N; i++)
	{
		d->idx[i] = i;
	}
	
	return d;
}

void
gdl_rho_data_free (gdl_rho_data * d)
{
	if (d)
	{
		gdl_allele_block_free (d->seq);
		GDL_FREE (d->NA);
		GDL_FREE (d->D);
		GDL_FREE (d->idx);
		GDL_FREE (d);
	}	
}

gdl_boolean
gdl_rho_data_is_missing (const gdl_rho_data * d, size_t c, size_t l)
{
	return (gdl_allele_block_get (d->seq, d->idx[c], l, 0)==0) ? gdl_true : gdl_false;
}

size_t
gdl_rho_data_get_allele (const gdl_rho_data * d, size_t c, size_t l)
{
	return gdl_allele_block_get (d->seq, d->idx[c], l, 0)-1;
}

void
gdl_rho_data_shuffle (gdl_rho_data * d, const gdl_rng * rng)
{
	size_t i, j, t;
	
	for (i = d->N - 1; i > 0; i--)
   {
      size_t j = gdl_rng_uniform_int(rng, i+1);
      t = d->idx[i];
      d->idx[i] = j;
      d->idx[j] = t;
   }
}	

