/*  
 *  rho/gdl_rho.h
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#ifndef __GDL_RHO_H__
#define __GDL_RHO_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_frequency_block.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gview_wrapper.h>

__BEGIN_DECLS

// data interface
typedef struct
{
	size_t N;
	size_t L;
	size_t * idx;
	size_t * NA;
	double * D;
	const gdl_chromosome   * chrom;   // the chromosome
	gdl_gview_wrapper      * data;    // genotypic view of the sub data set
	gdl_allele_block * seq;           // the observed haplotypes (nc * l)
} gdl_rho_data;

gdl_rho_data * gdl_rho_data_alloc (const gdl_chromosome * chrom, gdl_gview_wrapper * data);
void gdl_rho_data_free (gdl_rho_data * d);
gdl_boolean gdl_rho_data_is_missing (const gdl_rho_data * d, size_t c, size_t l);
size_t gdl_rho_data_get_allele (const gdl_rho_data * d, size_t c, size_t l);
double gdl_rho_data_weight (const gdl_rho_data * d, size_t K, size_t c);
void gdl_rho_data_shuffle (gdl_rho_data * d, const gdl_rng * rng);

typedef struct
{
	size_t L;
	double * lambda;
	double rho;
	double theta;
	double proba;
} gdl_rho_params;

gdl_rho_params * gdl_rho_params_alloc (const size_t L);
void gdl_rho_params_free (gdl_rho_params * p);
void gdl_rho_params_init (gdl_rho_params * p, size_t N);
gdl_rho_params * gdl_rho_params_clone (const gdl_rho_params * p);

typedef struct
{
	size_t L;
	size_t N;
	// forward-backward buffer
	double log;
	double * norm;
	double ** alpha;
} gdl_rho_util;

gdl_rho_util * gdl_rho_util_alloc (const size_t N, const size_t L);
void gdl_rho_util_free (gdl_rho_util * u);

typedef struct
{
	gdl_rho_data    * data;
	gdl_rho_params  * param;
	gdl_rho_util    * util;
	const gdl_rng   * rng;
	FILE * logger;
} gdl_rho;

gdl_rho * gdl_rho_alloc (const gdl_chromosome * chrom, gdl_gview_wrapper * data, const gdl_rng * rng);
void gdl_rho_free (gdl_rho * hmm);

int gdl_rho_perform (gdl_rho * m, gdl_boolean rho_cst, double eps, size_t max_iter, size_t rep);
double gdl_rho_loglikelihood (gdl_rho * m);
double gdl_rho_maximize_heterogeneous (gdl_rho * m);
double gdl_rho_maximize_rho (gdl_rho * m);
double gdl_rho_maximize_lambda (gdl_rho * m);
double gdl_rho_maximize_theta (gdl_rho * m);
void gdl_rho_summarize (gdl_rho * m, gdl_rho_params ** theta, size_t n);
gdl_vector_uint * gdl_rho_sample (const gdl_rho * r, const gdl_rng * rng);
// forward
void   gdl_rho_hmm_forward (gdl_rho * hmm, size_t k, size_t j);
double gdl_rho_hmm_forward_init (gdl_rho * hmm, size_t k, size_t j);
void   gdl_rho_hmm_forward_induc (gdl_rho * hmm, size_t k, size_t j, size_t l, double * pr);
// proba
double gdl_rho_hmm_obs_proba (const gdl_rho * hmm, size_t i, size_t k, size_t j, size_t l);
double gdl_rho_hmm_norec_proba (const gdl_rho * hmm, size_t k, size_t l, double d);
double gdl_rho_hmm_mut_proba (const gdl_rho * hmm, size_t k, size_t l);
double * gdl_rho_transition_proba (const gdl_rho * r, size_t i, size_t K, size_t l);

__END_DECLS

#endif /* __GDL_RHO_HMM__ */
