/*  
 *  rho/forward.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_collector.h>
#include <gdl/gdl_rho.h>
#include <gdl/gdl_rho_result.h>

gdl_rho_result *
gdl_rho_result_alloc (gdl_rho * m)
{
	size_t i, j, k;
	gdl_rho_result * r;
	
	r = GDL_CALLOC (gdl_rho_result, 1);
	
	r->N = m->data->N;
	r->L = m->data->L;
	r->theta = gdl_rho_params_clone (m->param); 
	
	return r;
}

void
gdl_rho_result_free (gdl_rho_result * r)
{
	if (r)
	{
		gdl_rho_params_free (r->theta);
		GDL_FREE (r);
	}	
}

gdl_rho_result *
gdl_rho_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_rho_result * r;
		
		r = GDL_CALLOC (gdl_rho_result, 1);
		
		status = fread (&r->N, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&r->L, sizeof(size_t), 1, stream);
		
		r->theta = gdl_rho_params_alloc (r->L);
		
		status = fread (&r->theta->rho, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&r->theta->theta, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (r->theta->lambda, sizeof(double), r->L-1, stream);
		GDL_FREAD_STATUS (status, r->L-1);
		
		return r;
	}
	return 0;	
}

int
gdl_rho_result_fwrite (FILE * stream, const gdl_rho_result * r)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->N, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->L, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->theta->rho, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->theta->theta, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (r->theta->lambda, sizeof(double), r->L-1, stream);
		GDL_FWRITE_STATUS (status, r->L-1);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

int
gdl_rho_result_fprintf (FILE * stream, const gdl_rho_result * r)
{
	if (stream && r)
	{
		size_t i, j;
		fprintf (stream, "---------------------------------------------\n");
		fprintf (stream, "--               gdl-rho                --\n");
		fprintf (stream, "---------------------------------------------\n");
		fprintf (stream, "Parameters:\n");
		fprintf (stream, "   N = %d (haplotypes)\n", r->N);
		fprintf (stream, "   L = %d (loci)\n", r->L);
		fprintf (stream, "---------------------------------------------\n");
		fprintf (stream, "Rho   = %e\n", r->theta->rho);
		fprintf (stream, "Theta = %e\n", r->theta->theta);
		fprintf (stream, "---------------------------------------------\n");
	   fprintf (stream, "Lambda:\n");
	   for (i = 0; i < r->L-1; i++)
	   {
	   	fprintf (stream, "%d\t%e\n", i+1, r->theta->lambda[i]);
	   }
	   fprintf (stream, "---------------------------------------------\n");
	   		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}


