/*  
 *  rho/sample.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <float.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_rho.h>

static size_t
gdl_rho_draw_origin (const gdl_rho * r, size_t i, size_t l)
{
	size_t j;
	gdl_ran_discrete_t * d;
	double * t;
	
	if (l)
	{
		t = gdl_rho_transition_proba (r, i, r->data->N, l-1);
		d = gdl_ran_discrete_preproc (r->data->N, t);
		j = gdl_ran_discrete (r->rng, d);
		gdl_ran_discrete_free (d);
		GDL_FREE (t);
	}
	else
	{
		j = gdl_rng_uniform_int (r->rng, r->data->N);
	}
	
	return j;
}

static size_t
gdl_rho_draw_allele (const gdl_rho * r, size_t i, size_t l)
{
	size_t a  = gdl_rho_data_get_allele (r->data, i, l);
	double mu = gdl_rho_hmm_mut_proba (r, r->data->N, l);
	
	if (gdl_rho_data_is_missing (r->data, i, l))
	{
		return 0;
	}
	else if (r->data->NA[l]==2)
	{
		return (gdl_ran_bernoulli (r->rng, mu)) ? 2-a : a+1;
	}
	else
	{
		if (gdl_ran_bernoulli (r->rng, mu))
		{
			size_t x;
			for(;;)
			{
				x = gdl_rng_uniform_int (r->rng, r->data->NA[l]);
				if (x != a) return x+1;
			}
		}
		else
		{
			return a+1;	
		}
	}	
}

gdl_vector_uint *
gdl_rho_sample (const gdl_rho * r, const gdl_rng * rng)
{
	gdl_vector_uint * h;
	size_t i, l;
	
	h  = gdl_vector_uint_alloc (r->data->L);
	for (l = 0; l < r->data->L; l++)
	{
		i = gdl_rho_draw_origin (r, i, l);
		gdl_vector_uint_set (h, l, gdl_rho_draw_allele (r, i, l));
	}
	
	return h;
}
