/* roots/convergence.c
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Reid Priedhorsky, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <config.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_roots.h>

int
gdl_root_test_interval (double x_lower, double x_upper, double epsabs, double epsrel)
{
  const double abs_lower = fabs(x_lower) ;
  const double abs_upper = fabs(x_upper) ;

  double min_abs, tolerance;

  if (epsrel < 0.0)
    GDL_ERROR ("relative tolerance is negative", GDL_EBADTOL);
  
  if (epsabs < 0.0)
    GDL_ERROR ("absolute tolerance is negative", GDL_EBADTOL);

  if (x_lower > x_upper)
    GDL_ERROR ("lower bound larger than upper bound", GDL_EINVAL);

  if ((x_lower > 0.0 && x_upper > 0.0) || (x_lower < 0.0 && x_upper < 0.0)) 
    {
      min_abs = GDL_MIN_DBL(abs_lower, abs_upper) ;
    }
  else
    {
      min_abs = 0;
    }

  tolerance = epsabs + epsrel * min_abs  ;
  
  if (fabs(x_upper - x_lower) < tolerance)
    return GDL_SUCCESS;
  
  return GDL_CONTINUE ;
}

int
gdl_root_test_delta (double x1, double x0, double epsabs, double epsrel)
{
  const double tolerance = epsabs + epsrel * fabs(x1)  ;

  if (epsrel < 0.0)
    GDL_ERROR ("relative tolerance is negative", GDL_EBADTOL);
  
  if (epsabs < 0.0)
    GDL_ERROR ("absolute tolerance is negative", GDL_EBADTOL);
  
  if (fabs(x1 - x0) < tolerance || x1 == x0)
    return GDL_SUCCESS;
  
  return GDL_CONTINUE ;
}

int
gdl_root_test_residual (double f, double epsabs)
{
  if (epsabs < 0.0)
    GDL_ERROR ("absolute tolerance is negative", GDL_EBADTOL);
 
  if (fabs(f) < epsabs)
    return GDL_SUCCESS;
  
  return GDL_CONTINUE ;
}

