/*  
 *  seq/io.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:41 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <ctype.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_locus_type.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_seq_io.h>
#include <gdl/gdl_seq.h>

gdl_seq_io *
gdl_seq_io_alloc (const gdl_seq_io_type * type)
{
	gdl_seq_io * r;
	
	r = GDL_CALLOC (gdl_seq_io, 1);
	
	r->type = type;
	
	return r;
}

void
gdl_seq_io_free (gdl_seq_io * reader)
{
	if (reader)
	{
		gdl_string_free (reader->filename);
		gdl_string_free (reader->error);
		GDL_FREE (reader);
	}
}

int
gdl_seq_io_open (gdl_seq_io * reader, 
                     const gdl_string * filename)
{
	if (reader && filename)
	{
		reader->stream = gdl_fileopen (filename, "r");
		if (reader->stream)
		{
			gdl_string_free (reader->filename);
			reader->filename = gdl_string_clone (filename);
			gdl_string_free (reader->error);
			reader->error = NULL;
			return GDL_SUCCESS;
		}
		else
			return GDL_EINVAL;
	}
	return GDL_EINVAL;
}

int
gdl_seq_io_parse (gdl_seq_io * reader)
{
	if (!reader)
	{
		return GDL_EINVAL;
	}
	else
	{
		return (reader->type->read)(reader);
	}
}

gdl_seqdb *
gdl_seq_io_close (gdl_seq_io * reader)
{
	if (reader && reader->stream)
	{
		gdl_seqdb * db = reader->db;
		gdl_fileclose (reader->filename, reader->stream);
		reader->db = 0;
		return db;
	}
	return 0;
}

int
gdl_seq_io_error (gdl_seq_io * reader, char * format, ...)
{
	if (reader == 0)
		return GDL_EINVAL;
	else
	{
		va_list ap; 
		
		va_start (ap, format);
		
		gdl_string_free (reader->error);
		
		reader->error = gdl_string_vsprintf (format, ap);
		
		va_end (ap);
	 	
	 	return GDL_SUCCESS;
	}		
}

int
gdl_seq_io_dump (FILE * stream, const gdl_seq_io * io, const gdl_seqdb * db)
{
	if (stream && io && db)
	{
		int status;
		status=(io->type->write)(stream, io, db);
		return status;
	}	
}

int
gdl_seq_io_string_dump (FILE * stream, const gdl_seq_io * io, const gdl_string * str)
{
	if (stream && io && str)
	{
		int status;
		status=(io->type->write_str)(stream, io, str);
		return status;
	}		
}

const gdl_seq_io_type *
gdl_seq_io_type_get (const gdl_string * name)
{
	if (!strcmp(name, gdl_seq_io_fasta->acronym))
		return gdl_seq_io_fasta;
	return 0;
}

