/*
 *  snp/cnv.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_nan.h>
#include <gdl/gdl_cnv_data.h>

gdl_cnv_data *
gdl_cnv_data_alloc (const size_t N, const size_t L)
{
	gdl_cnv_data * v;
	
	v = GDL_MALLOC (gdl_cnv_data, 1);
	
	v->N = N;
	v->L = L;
	v->data = GDL_CALLOC (double, v->N*v->L);
	v->clones = GDL_CALLOC (gdl_cnv_clone *, v->L);
	
	return v;	
}

void
gdl_cnv_data_free (gdl_cnv_data * v)
{
	if (v)
	{
		size_t i;
		for (i = 0; i < v->L; i++)
		{
			gdl_cnv_clone_free (v->clones[i]);	
		}
		GDL_FREE (v->clones);
		GDL_FREE (v->data);
		GDL_FREE (v);
	}	
}

gdl_boolean
gdl_cnv_data_is_missing (const gdl_cnv_data * v, size_t i, size_t j)
{
	return (gdl_isnan(gdl_cnv_data_get (v, i, j))) ? gdl_true : gdl_false;
}

double
gdl_cnv_data_get (const gdl_cnv_data * v, size_t i, size_t j)
{
	return v->data[i*v->L+j];
}

void
gdl_cnv_data_set (const gdl_cnv_data * v, size_t i, size_t j, const double u)
{
	v->data[i*v->L+j]=u;
}

gdl_cnv_data *
gdl_cnv_data_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i, N, L;
		gdl_cnv_data * v;
			
		status = fread (&N, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&L, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		v = gdl_cnv_data_alloc (N, L);
		
		status = fread (v->data, sizeof(double), N*L, stream);
		GDL_FREAD_STATUS (status, N*L);
		
		for (i = 0; i < v->L; i++)
		{
			v->clones[i] = gdl_cnv_clone_fread (stream);
			GDL_FREAD_STATUS (v->clones[i]!=0, 1);
		}
		
		return v;
	}
	return 0;
}

int
gdl_cnv_data_fwrite (FILE * stream, const gdl_cnv_data * v)
{
	if (stream && v)
	{
		size_t i;
		int status;
		
		status = fwrite (&v->N, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&v->L, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (v->data, sizeof(double), v->N*v->L, stream);
		GDL_FWRITE_STATUS (status, v->N*v->L);
		for (i = 0; i < v->L; i++)
		{
			status = gdl_cnv_clone_fwrite (stream, v->clones[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

gdl_cnv_data *
gdl_cnv_data_fscanf (FILE * stream)
{
	if (stream)
	{
		size_t i, j, k, l, n, m;
		gdl_string * name, * tok, * line=0;
		gdl_cnv_data * data = 0;
		long start, end, offset;
		
		offset = ftell (stream);
		l=m=0;
		while (gdl_getline (&line, &n, stream) != -1)
		{
			for(i=j=k=0;;k++)
			{
				tok = gdl_string_next_token (line, n, &i, &j);
				if (!tok) break;
				gdl_string_free (tok);
			}
			GDL_FREE (line);
			line=0;
			if (!m) l=k;
			if (k!=l)
			{
				GDL_ERROR_VAL (gdl_string_sprintf ("gdl_cnv_data_fscanf : Line %s has %d tokens when %d are expected\n", line, k, l),
				               GDL_FAILURE,
				               0);
			}
			m++;
		}
		
		data = gdl_cnv_data_alloc (l-3, m);
		
		fseek (stream, offset, SEEK_SET);
		m=0;
		while (gdl_getline (&line, &n, stream) != -1)
		{
			i=j=0;
			
			// read cnv clone info
			tok = gdl_string_next_token (line, n, &i, &j);
			name = gdl_string_clone (tok);
			gdl_string_free (tok);
			tok = gdl_string_next_token (line, n, &i, &j);
			start = atol(tok);
			gdl_string_free (tok);
			tok = gdl_string_next_token (line, n, &i, &j);
			end = atol(tok);
			data->clones[m] = gdl_cnv_clone_alloc (name, start, end);
			gdl_string_free (name);
			gdl_string_free (tok);
			
			// read cnv clone log data 
			for(k=0;k<data->N;k++)
			{
				tok = gdl_string_next_token (line, n, &i, &j);
				if (!strcmp(tok, "NA"))
				{
					gdl_cnv_data_set (data, k, m, GDL_NAN);
				}
				else
				{
					gdl_cnv_data_set (data, k, m, (double)atof(tok)); 	
				}
				gdl_string_free (tok);
			}
			GDL_FREE (line);
			line=0;
			m++;
		}
		
		return data;
	}
	return 0;
}
