/*
 *  snp/fwrite.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_snp_data.h>
#include <gdl/gdl_snp_map.h>
#include <gdl/gdl_snp_chromosome.h>

int
gdl_snp_data_fwrite (FILE * stream, const gdl_snp_data * v)
{
	if (stream && v)
	{
		int status;
		
		status = fwrite (&v->N, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&v->L, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&v->P, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_string_fwrite (stream, v->T->name);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		if (v->T == gdl_snp_data_genotype)
		{
			status = fwrite (v->data, sizeof(unsigned char), v->N*v->L, stream);
			GDL_FWRITE_STATUS (status, v->N*v->L);
		}
		else
		{
			status = fwrite (v->data, sizeof(unsigned char), v->N*v->L*v->P, stream);
			GDL_FWRITE_STATUS (status, v->N*v->L*v->P);	
		}		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

int
gdl_snp_fwrite (FILE * stream, const gdl_snp * snp)
{
	if (stream && snp)
	{
		int status;
		unsigned char has;
		
		status = gdl_string_fwrite (stream, snp->rs);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_string_fwrite (stream, snp->alleles[0]);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_string_fwrite (stream, snp->alleles[1]);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = fwrite (&snp->position, sizeof (long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&snp->rate, sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&snp->distance, sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		has = (snp->annot) ? '1' : '0';
		status = fwrite (&has, sizeof (unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has == '1')
		{
			status = gdl_snp_annot_fwrite (stream, snp->annot);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		status = fwrite (&snp->polypop, sizeof (unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&snp->ignore, sizeof (unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

int
gdl_snp_map_fwrite (FILE * stream, const gdl_snp_map * chrom)
{
	if (stream && chrom)
	{
		int status;
		size_t i;
		
		status = fwrite (&chrom->size, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < chrom->size; i++)
		{
			status = gdl_snp_fwrite (stream, chrom->snps[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

int
gdl_snp_chromosome_fwrite (FILE * stream, const gdl_snp_chromosome * v)
{
	if (stream)
	{
		size_t i;
		unsigned char has;
		int status;
		
		status = fwrite (&v->npop, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < v->npop; i++)
		{
			status = gdl_snp_data_fwrite (stream, v->pops[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			has = (v->cnvs && v->cnvs[i]) ? 'y' : 'n';
			status = fwrite (&has, sizeof(unsigned char), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			if (has=='y')
			{
				status = gdl_cnv_data_fwrite (stream, v->cnvs[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
		}
		status = gdl_snp_map_fwrite (stream, v->chrom);
		if (v->npop > 1)
		{
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			for (i = 0; i < v->chrom->size; i++)
			{
				status = fwrite (v->idx[i], sizeof(size_t), v->npop, stream);
				GDL_FWRITE_STATUS (status, v->npop);
			}
		}
	
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}
