/* statistics/medpolish.c
 * 
 * Copyright (C) 2008 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h> 

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_sort_double.h>

/**
 * gdl_stats_medpolish
 * 
 * x[1..nr][1..nc] the data matrix 
 * nr       the number of rows
 * nc       the numbre of colums
 * eps      double number greater than 0. A tolerance for convergence
 * max_iter the maximum number of iterations
 */
int
gdl_stats_medpolish (double ** x,
                     const size_t nr,
                     const size_t nc,
                     const double eps,
                     const size_t max_iter,
                     double *t,
                     double r[],
                     double c[],
                     double z[])
{
	int status = GDL_CONTINUE;
	size_t i, j, k;
	double newsum, delta, oldsum=0, *rdelta, *cdelta, *tmp;
	
	rdelta = GDL_MALLOC (double, nr);
	cdelta = GDL_MALLOC (double, nc);
	tmp    = GDL_MALLOC (double, GDL_MAX(nc, nr));
	// clean and init buffer
	for(i = 0; i < nc; i++) c[i]=0;
	for(i = 0; i < nr; i++)
	{
		r[i]=0;
		memcpy(z+(i*nc), x[i], sizeof(double)*nc);
	}
	*t=0;
	for(i = 0; i < max_iter; i++)
	{
		// row medians
		// rdelta <- apply(z, 1, median, na.rm = na.rm)
		// z <- z - matrix(rdelta, nrow = nr, ncol = nc)
		// r <- r + rdelta
		for(j = 0; j < nr; j++)
		{
			// compute the median
			gdl_sort (z+(j*nc), 1, nc);
			//printf ("%g %g %g\n", z[(j*nc)], z[(j*nc)+1], z[(j*nc)+2]);
			rdelta[j] = gdl_stats_median_from_sorted_data (z+(j*nc), 1, nc) ;
			// substract it
			for(k = 0; k < nc; k++)
			{
				if (!gdl_isnan (x[j][k]))
				{
					x[j][k]   -= rdelta[j];
					z[j*nc+k]  = x[j][k];
				}
			}
			r[j] += rdelta[j];	
		}
		// delta <- median(c, na.rm = na.rm)
		// c <- c - delta
        // t <- t + delta
        memcpy(tmp, c, sizeof(double)*nc);
		gdl_sort (tmp, 1, nc);
		delta = gdl_stats_median_from_sorted_data (tmp, 1, nc);
        for(j = 0; j < nc; j++)
        {
        	c[j] -= delta;
        }
        (*t) += delta;
		// cdelta <- apply(z, 2, median, na.rm = na.rm)
        // z <- z - matrix(cdelta, nrow = nr, ncol = nc, byrow = TRUE)
        // c <- c + cdelta
        // newsum <- sum(abs(z), na.rm = na.rm)
		for(newsum = 0, j = 0; j < nc; j++)
		{
			// compute the median
			gdl_sort (z+j, nc, nr);
			cdelta[j] = gdl_stats_median_from_sorted_data (z+j, nc, nr);
			// substract it
			for(k = 0; k < nr; k++)
			{
				if (!gdl_isnan (x[k][j]))
				{
					x[k][j]   -= cdelta[j];
					z[k*nc+j]  = x[k][j];
					newsum    += fabs (x[k][j]);
				} 
			}
			c[j] += cdelta[j];	
		}
		// delta <- median(r, na.rm = na.rm)
        // r <- r - delta
        // t <- t + delta
        memcpy(tmp, r, sizeof(double)*nr);
        gdl_sort (tmp, 1, nr);
		delta = gdl_stats_median_from_sorted_data (tmp, 1, nr);
		for(j = 0; j < nr; j++)
        {
        	r[j] -= delta;
        }
        (*t) += delta;
        if (newsum == 0 || fabs(newsum - oldsum) < eps * newsum)
        {
        	status = GDL_SUCCESS;
        	break;	
        }
        oldsum = newsum;
	}
	GDL_FREE (tmp);
	GDL_FREE (cdelta);
	GDL_FREE (rdelta);
	
	return status;
}
