/*  
 * 	string/string.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <string.h> 
#include <ctype.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>

gdl_string *
gdl_string_alloc (size_t n)
{
	return GDL_CALLOC (char, n+1);
}

gdl_string *
gdl_string_realloc (gdl_string * str, size_t n)
{
	size_t i, on = strlen (str);
	gdl_string * s = GDL_REALLOC (gdl_string, str, n+1);
	for (i = on; i < n; i++)
	{
	   s[i] = '\0';
	}
	return s;
}

void
gdl_string_free (gdl_string * str)
{
	GDL_FREE (str);	
}

gdl_string *
gdl_string_clone (const gdl_string * string)
{
	if (string == NULL)
		return NULL;
	else
	{
		size_t n     = strlen(string);
		gdl_string * clone = gdl_string_alloc (n);
		if (clone == 0)
		{
			GDL_ERROR_VAL ("Failed to allocate memory for gdl_string_clone",
                        GDL_ENOMEM, 0);	
		}
		memcpy (clone, string, n*sizeof(char));
		clone[n]     = '\0';
		return clone;
	}
}

void
gdl_string_clean (gdl_string * string)
{
	if (string == 0)
		return;	
	else
	{
		size_t i, n = strlen (string);
		for (i = 0; i < n; i++)
		{
			string[i] = '\0';	
		}
	}
}

size_t
gdl_string_sep (const gdl_string * sep, const gdl_string * string)
{
	gdl_string_nsep (sep, string, strlen(string));
}

size_t
gdl_string_nsep (const gdl_string * sep, const gdl_string * string, size_t n)
{
	size_t i, j, k, to, len, nsep;
	char * tmp;
	
	if (string == 0 || sep == 0 || n == 0)
	{
		return 0;	
	}
	
	nsep = 0;
	len  = strlen (sep);
	tmp  = GDL_CALLOC (char, len+1);
	
	for (i = 0; i < n; i++)
	{
		k = i % len;
		tmp[k] = string[i];
		if (k == len - 1 && !strcmp (tmp, sep))
		{
			nsep++;
		}
	}
	
	GDL_FREE (tmp);
	
	return nsep;
}

size_t
gdl_string_token (const gdl_string * sep, const gdl_string * string)
{
	if (sep && string)
		return gdl_string_ntoken (sep, string, strlen (string));
	return 0;
}

size_t
gdl_string_ntoken (const gdl_string * sep, const gdl_string * string, size_t n)
{
	if (sep && string && n)
		return gdl_string_nsep (sep, string, n) + 1;
	return 0;
}

size_t
gdl_string_token_start (const gdl_string * sep, const gdl_string * string, gdl_string ** token)
{
	if (sep && string)
		return gdl_string_ntoken_start (sep, string, strlen(string), token);
}

size_t
gdl_string_ntoken_start (const gdl_string * sep, const gdl_string * string, size_t n, gdl_string ** token)
{
	if (sep && string)
	{
		size_t next = 0;
		*token = gdl_string_alloc (1);
		gdl_string_ntoken_next (sep, string, n, token, &next);
		return next;
	}
	return (n);
}

void
gdl_string_token_next (const gdl_string * sep, const gdl_string * string, gdl_string ** token, size_t * next)
{
	if (sep && string)
		gdl_string_ntoken_next (sep, string, strlen(string), token, next);
}

void
gdl_string_ntoken_next (const gdl_string * sep, const gdl_string * string, size_t n, gdl_string ** token, size_t * next)
{
	if ((sep == 0 && string == 0) && n == 0)
	{
		gdl_string_free (*token);
		*token = NULL;
		return;
	}
	if (*next == n)
	{
		gdl_string_free (*token);
		*token = NULL;
		return;
	}
	else
	{
		size_t i, j, k, l, np, len = 0, fsep = 0;
		gdl_string * tmp;
		
		if ((*next) > 0)
			(*next) = (*next)+1;
				
		i   = *next;
		np  = strlen (sep);
		
		tmp = gdl_string_alloc (np);
		
		for (j = i; j < n; j++)
		{
			k = j - i;
			l = k % np;
			tmp [l] = string[j];
			if (l == np - 1 && !strcmp (tmp, sep))
			{
				break;
				fsep=1;
			}
			len++;
		}
		
		gdl_string_free (tmp);
		
		if (fsep)
		{
			len -= np;	
		}
		
		gdl_string_free (*token);
		
		*token = gdl_string_alloc (len);
		
		strncpy (*token, &string[*next], len);
		
		if (j + 1 < n)
			*next = j;
		else
			*next = n;
	}
}

gdl_string *
gdl_string_sprintf (const gdl_string * format, ...)
{
	va_list ap;
	int n;
	size_t size = strlen (format); 
	gdl_string * tmp; 
		
	tmp = gdl_string_alloc (size);
		
	while (1) {
        /* Try with the current allocated space */
        va_start (ap, format);
        n = vsnprintf (tmp, size, format, ap);
        va_end(ap);
        /* If ok, set name = tmp*/
        if (n > -1 && n < size)
        {
         	return tmp;
        } 
        /* Otherwise try it with more memory...  */
        if (n > -1)    
           size = n+1;
        else           
           size *= 2;
        tmp = gdl_string_realloc (tmp, size);
     }
     
     return (tmp);
}

gdl_string *
gdl_string_vsprintf (const gdl_string * format, va_list ap)
{
	int n;
	size_t size = strlen (format); 
	gdl_string * tmp; 
		
	tmp = gdl_string_alloc (size);
		
	while (1) {
        /* Try with the current allocated space */
        n = vsnprintf (tmp, size, format, ap);
        /* If ok, set name = tmp*/
        if (n > -1 && n < size)
        {
         	return tmp;
        } 
        /* Otherwise try it with more memory...  */
        if (n > -1)    
           size = n+1;
        else           
           size *= 2;
        tmp = gdl_string_realloc (tmp, size);
     }
     
     return (tmp);
}

size_t 
gdl_string_scat (gdl_string * string, const gdl_string * format, ...)
{
	va_list ap;
	size_t e;
	gdl_string * sub;
	
	va_start (ap, format);
	sub = gdl_string_vsprintf (format, ap);
	va_end (ap);
	e = strlen (sub);
	strncpy (string, sub, e);
	gdl_string_free (sub);
	
	return e;
}

gdl_boolean
gdl_string_is_space (const gdl_string * string)
{
	size_t i, n = strlen (string);
	for (i = 0; i < n; i++)
	{
		if (!isspace (string[i]))
		{
			return gdl_false;
		}
	}
	return gdl_true;
}

size_t
gdl_string_cat (gdl_string ** string, const int c)
{
	size_t i = (*string!=NULL) ? strlen(*string) : 0;
	
	if (i)
	{
		gdl_string * new = gdl_string_alloc (i+1);
		memcpy (new, *string, sizeof (char)*i);
		gdl_string_free (*string);
		*string = new;
	}
	else
	{
		*string = gdl_string_alloc (1);
	}
	
	(*string)[i] = (char)c;
	
	return i+1;
}

gdl_string *
gdl_string_next_token (const gdl_string * str, const size_t n, size_t * start, size_t * end)
{
	size_t i,j;
	
	if (*end >= n-1)
	{
		return 0;
	}
	else
	{
		gdl_string * tok;
		
		i=*end;
		for (j = i; isspace(str[j]) && j < n; j++);
		*start=j;
	   for (i = j; !isspace(str[i]) && i < n; i++);
	   *end=i;
	  
	   tok = gdl_string_alloc (i-j);
		strncpy (tok, &str[j], i-j);
		
		return tok;
	}
}

gdl_string **
gdl_string_split (const gdl_string * str, const gdl_string * del, size_t * n)
{
	if (!str || !del) {*n=0;return 0;}	
	
	size_t i, j, k, len, dlen, nn;
	gdl_string ** toks;
	
	len  = strlen (str);
	dlen = strlen (del);
	
	for (nn = 1, j = i = 0; i < len; i++)
	{
		if (del[j]==str[i])
		{
			j++;	
		}
		if (j == dlen)
		{
			j = 0;
			nn++;
		}
	}
	
	toks = GDL_MALLOC (gdl_string *, nn);
	
	for (nn = 0, k = j = i = 0; i < len; i++)
	{
		if (del[j]==str[i])
		{
			j++;	
		}
		if (j == dlen)
		{
			toks[nn] = gdl_string_alloc (i-j-k+1);
			strncpy (toks[nn], &str[k], i-j-k+1);
			k = i+1;
			j = 0;
			nn++;
		}
	}
	// last one
	toks[nn] = gdl_string_alloc (len-k);
	strncpy (toks[nn], &str[k], len-k);
	
	*n = nn+1;
	
	return toks;
}

/*
 * alpha    = lowalpha | upalpha
 */
#define IS_ALPHA(x) (IS_LOWALPHA(x) || IS_UPALPHA(x))


/*
 * lowalpha = "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" | "j" |
 *            "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" | "s" | "t" |
 *            "u" | "v" | "w" | "x" | "y" | "z"
 */

#define IS_LOWALPHA(x) (((x) >= 'a') && ((x) <= 'z'))

/*
 * upalpha = "A" | "B" | "C" | "D" | "E" | "F" | "G" | "H" | "I" | "J" |
 *           "K" | "L" | "M" | "N" | "O" | "P" | "Q" | "R" | "S" | "T" |
 *           "U" | "V" | "W" | "X" | "Y" | "Z"
 */
#define IS_UPALPHA(x) (((x) >= 'A') && ((x) <= 'Z'))

#ifdef IS_DIGIT
#undef IS_DIGIT
#endif
/*
 * digit = "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" | "8" | "9"
 */
#define IS_DIGIT(x) (((x) >= '0') && ((x) <= '9'))

/*
 * alphanum = alpha | digit
 */

#define IS_ALPHANUM(x) (IS_ALPHA(x) || IS_DIGIT(x))

/*
 * hex = digit | "A" | "B" | "C" | "D" | "E" | "F" |
 *               "a" | "b" | "c" | "d" | "e" | "f"
 */

#define IS_HEX(x) ((IS_DIGIT(x)) || (((x) >= 'a') && ((x) <= 'f')) || \
	    (((x) >= 'A') && ((x) <= 'F')))

/*
 * mark = "-" | "_" | "." | "!" | "~" | "*" | "'" | "(" | ")"
 */

#define IS_MARK(x) (((x) == '-') || ((x) == '_') || ((x) == '.') ||	\
    ((x) == '!') || ((x) == '~') || ((x) == '*') || ((x) == '\'') ||	\
    ((x) == '(') || ((x) == ')'))


/*
 * reserved = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" | "$" | "," |
 * 	      "[" | "]"
 */

#define IS_RESERVED(x) (((x) == ';') || ((x) == '/') || ((x) == '?') ||	\
        ((x) == ':') || ((x) == '@') || ((x) == '&') || ((x) == '=') ||	\
	((x) == '+') || ((x) == '$') || ((x) == ',') || ((x) == '[') || \
	((x) == ']'))

/*
 * unreserved = alphanum | mark
 */

#define IS_UNRESERVED(x) (IS_ALPHANUM(x) || IS_MARK(x))

/*
 * escaped = "%" hex hex
 */

#define IS_ESCAPED(p) ((*(p) == '%') && (IS_HEX((p)[1])) &&		\
	    (IS_HEX((p)[2])))

gdl_string *
gdl_string_escape (const gdl_string * str, const gdl_string * exception)
{
    gdl_string * ret, ch;
    const gdl_string * in;
    unsigned int len, out;
    
    if (!str) return 0;
    
    len = strlen(str);
    
    if (!(len > 0)) return 0;

    len += 20;
    ret  = gdl_string_alloc(len);
    in   = (const gdl_string *) str;
    out  = 0;
    
    while(*in != 0) 
    {
	    if (len - out <= 3)
	    {
	       len += 20;
	       ret = gdl_string_realloc(ret, len);
	    }
		ch = *in;

		if ((ch != '@') && (!IS_UNRESERVED(ch)) && (!strchr(exception, ch)))
		{
		    unsigned char val;
		    ret[out++] = '%';
		    val = ch >> 4;
		    if (val <= 9)
			    ret[out++] = '0' + val;
		    else
			    ret[out++] = 'A' + val - 0xA;
		    val = ch & 0xF;
		    if (val <= 9)
			    ret[out++] = '0' + val;
		    else
			    ret[out++] = 'A' + val - 0xA;
		    in++;
		}
		else
		{
		    ret[out++] = *in++;
		}
    }
    
    ret[out] = '\0';
    
    return(ret);
}

static int is_hex(char c)
{
    if (((c >= '0') && (c <= '9')) ||
        ((c >= 'a') && (c <= 'f')) ||
        ((c >= 'A') && (c <= 'F')))
	return(1);
    return(0);
}

gdl_string *
gdl_string_unescape (const gdl_string * str, int len, gdl_string * target)
{
    gdl_string * ret, * out;
    const gdl_string * in;

    if (!str)
			return 0;
    if (len <= 0) len = strlen(str);
    if (len < 0)  return 0;

    if (!target)
    	ret = gdl_string_alloc (len + 1);
	 else
		ret = target;
		
    in  = str;
    out = ret;
    
    while(len > 0)
    {
	    if ((len > 2) && (*in == '%') && (is_hex(in[1])) && (is_hex(in[2])))
	    {
		    in++;
		    if ((*in >= '0') && (*in <= '9')) 
		        *out = (*in - '0');
		    else if ((*in >= 'a') && (*in <= 'f'))
		        *out = (*in - 'a') + 10;
		    else if ((*in >= 'A') && (*in <= 'F'))
		        *out = (*in - 'A') + 10;
		    in++;
		    if ((*in >= '0') && (*in <= '9')) 
		        *out = *out * 16 + (*in - '0');
		    else if ((*in >= 'a') && (*in <= 'f'))
		        *out = *out * 16 + (*in - 'a') + 10;
		    else if ((*in >= 'A') && (*in <= 'F'))
		        *out = *out * 16 + (*in - 'A') + 10;
		    in++;
		    len -= 3;
		    out++;
		 }
		 else
		 {
	    	*out++ = *in++;
	    	len--;
		 }
    }
    
    *out = '\0';
    
    return ret;
}

static void
_gdl_string_free (void * s)
{
	gdl_string_free (s);	
}

static void *
_gdl_string_clone (const void * s)
{
	return gdl_string_clone ((gdl_string *)s);	
}

static int
_gdl_string_compare (const void * s1, const void * s2)
{
	return strcmp ((gdl_string *)s1, (gdl_string *)s2);	
}

static void *
_gdl_string_fread (FILE * stream)
{
	return gdl_string_fread (stream);	
}

static int
_gdl_string_fwrite (FILE * stream, const void * s)
{
	return gdl_string_fwrite (stream, s);
}

static const gdl_data_interface _gdl_string_interface =
{
	&_gdl_string_free,
	&_gdl_string_clone,
	&_gdl_string_compare,
	&_gdl_string_fread,
	&_gdl_string_fwrite
};

const gdl_data_interface * gdl_string_interface = &_gdl_string_interface;
