/* vector/gdl_vector_double.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_VECTOR_DOUBLE_H__
#define __GDL_VECTOR_DOUBLE_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_gblock.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size;
  size_t stride;
  double *data;
  gdl_block *block;
  int owner;
} 
gdl_vector;

typedef struct
{
  gdl_vector vector;
} _gdl_vector_view;

typedef _gdl_vector_view gdl_vector_view;

typedef struct
{
  gdl_vector vector;
} _gdl_vector_const_view;

typedef const _gdl_vector_const_view gdl_vector_const_view;


/* Allocation */

gdl_vector *gdl_vector_alloc (const size_t n);
gdl_vector *gdl_vector_calloc (const size_t n);

gdl_vector *gdl_vector_alloc_from_block (gdl_block * b,
                                                     const size_t offset, 
                                                     const size_t n, 
                                                     const size_t stride);

gdl_vector *gdl_vector_alloc_from_vector (gdl_vector * v,
                                                      const size_t offset, 
                                                      const size_t n, 
                                                      const size_t stride);

void gdl_vector_free (gdl_vector * v);

/* Views */

_gdl_vector_view 
gdl_vector_view_array (double *v, size_t n);

_gdl_vector_view 
gdl_vector_view_array_with_stride (double *base,
                                         size_t stride,
                                         size_t n);

_gdl_vector_const_view 
gdl_vector_const_view_array (const double *v, size_t n);

_gdl_vector_const_view 
gdl_vector_const_view_array_with_stride (const double *base,
                                               size_t stride,
                                               size_t n);

_gdl_vector_view 
gdl_vector_subvector (gdl_vector *v, 
                            size_t i, 
                            size_t n);

_gdl_vector_view 
gdl_vector_subvector_with_stride (gdl_vector *v, 
                                        size_t i,
                                        size_t stride,
                                        size_t n);

_gdl_vector_const_view 
gdl_vector_const_subvector (const gdl_vector *v, 
                                  size_t i, 
                                  size_t n);

_gdl_vector_const_view 
gdl_vector_const_subvector_with_stride (const gdl_vector *v, 
                                              size_t i, 
                                              size_t stride,
                                              size_t n);

/* Operations */

double gdl_vector_get (const gdl_vector * v, const size_t i);
void gdl_vector_set (gdl_vector * v, const size_t i, double x);

double *gdl_vector_ptr (gdl_vector * v, const size_t i);
const double *gdl_vector_const_ptr (const gdl_vector * v, const size_t i);

void gdl_vector_set_zero (gdl_vector * v);
void gdl_vector_set_all (gdl_vector * v, double x);
int gdl_vector_set_basis (gdl_vector * v, size_t i);

gdl_vector * gdl_vector_fread (FILE * stream);
int gdl_vector_fwrite (FILE * stream, const gdl_vector * v);
gdl_vector * gdl_vector_fscanf (FILE * stream);
int gdl_vector_fprintf (FILE * stream, const gdl_vector * v,
                              const char *format);

int gdl_vector_memcpy (gdl_vector * dest, const gdl_vector * src);

int gdl_vector_reverse (gdl_vector * v);

int gdl_vector_swap (gdl_vector * v, gdl_vector * w);
int gdl_vector_swap_elements (gdl_vector * v, const size_t i, const size_t j);

double gdl_vector_max (const gdl_vector * v);
double gdl_vector_min (const gdl_vector * v);
void gdl_vector_minmax (const gdl_vector * v, double * min_out, double * max_out);

size_t gdl_vector_max_index (const gdl_vector * v);
size_t gdl_vector_min_index (const gdl_vector * v);
void gdl_vector_minmax_index (const gdl_vector * v, size_t * imin, size_t * imax);

int gdl_vector_add (gdl_vector * a, const gdl_vector * b);
int gdl_vector_sub (gdl_vector * a, const gdl_vector * b);
int gdl_vector_mul (gdl_vector * a, const gdl_vector * b);
int gdl_vector_div (gdl_vector * a, const gdl_vector * b);
int gdl_vector_scale (gdl_vector * a, const double x);
int gdl_vector_add_constant (gdl_vector * a, const double x);

int gdl_vector_isnull (const gdl_vector * v);

#ifdef HAVE_INLINE

extern inline
double
gdl_vector_get (const gdl_vector * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
    }
#endif
  return v->data[i * v->stride];
}

extern inline
void
gdl_vector_set (gdl_vector * v, const size_t i, double x)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
    }
#endif
  v->data[i * v->stride] = x;
}

extern inline
double *
gdl_vector_ptr (gdl_vector * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (double *) (v->data + i * v->stride);
}

extern inline
const double *
gdl_vector_const_ptr (const gdl_vector * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (const double *) (v->data + i * v->stride);
}


#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_VECTOR_DOUBLE_H__ */


