/* vector/gdl_vector_ulong.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_VECTOR_ULONG_H__
#define __GDL_VECTOR_ULONG_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_types.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_check_range.h>
#include <gdl/gdl_gblock.h>

__BEGIN_DECLS

typedef struct 
{
  size_t size;
  size_t stride;
  unsigned long *data;
  gdl_block_ulong *block;
  int owner;
} 
gdl_vector_ulong;

typedef struct
{
  gdl_vector_ulong vector;
} _gdl_vector_ulong_view;

typedef _gdl_vector_ulong_view gdl_vector_ulong_view;

typedef struct
{
  gdl_vector_ulong vector;
} _gdl_vector_ulong_const_view;

typedef const _gdl_vector_ulong_const_view gdl_vector_ulong_const_view;


/* Allocation */

gdl_vector_ulong *gdl_vector_ulong_alloc (const size_t n);
gdl_vector_ulong *gdl_vector_ulong_calloc (const size_t n);

gdl_vector_ulong *gdl_vector_ulong_alloc_from_block (gdl_block_ulong * b,
                                                     const size_t offset, 
                                                     const size_t n, 
                                                     const size_t stride);

gdl_vector_ulong *gdl_vector_ulong_alloc_from_vector (gdl_vector_ulong * v,
                                                      const size_t offset, 
                                                      const size_t n, 
                                                      const size_t stride);

void gdl_vector_ulong_free (gdl_vector_ulong * v);

/* Views */

_gdl_vector_ulong_view 
gdl_vector_ulong_view_array (unsigned long *v, size_t n);

_gdl_vector_ulong_view 
gdl_vector_ulong_view_array_with_stride (unsigned long *base,
                                         size_t stride,
                                         size_t n);

_gdl_vector_ulong_const_view 
gdl_vector_ulong_const_view_array (const unsigned long *v, size_t n);

_gdl_vector_ulong_const_view 
gdl_vector_ulong_const_view_array_with_stride (const unsigned long *base,
                                               size_t stride,
                                               size_t n);

_gdl_vector_ulong_view 
gdl_vector_ulong_subvector (gdl_vector_ulong *v, 
                            size_t i, 
                            size_t n);

_gdl_vector_ulong_view 
gdl_vector_ulong_subvector_with_stride (gdl_vector_ulong *v, 
                                        size_t i,
                                        size_t stride,
                                        size_t n);

_gdl_vector_ulong_const_view 
gdl_vector_ulong_const_subvector (const gdl_vector_ulong *v, 
                                  size_t i, 
                                  size_t n);

_gdl_vector_ulong_const_view 
gdl_vector_ulong_const_subvector_with_stride (const gdl_vector_ulong *v, 
                                              size_t i, 
                                              size_t stride,
                                              size_t n);

/* Operations */

unsigned long gdl_vector_ulong_get (const gdl_vector_ulong * v, const size_t i);
void gdl_vector_ulong_set (gdl_vector_ulong * v, const size_t i, unsigned long x);

unsigned long *gdl_vector_ulong_ptr (gdl_vector_ulong * v, const size_t i);
const unsigned long *gdl_vector_ulong_const_ptr (const gdl_vector_ulong * v, const size_t i);

void gdl_vector_ulong_set_zero (gdl_vector_ulong * v);
void gdl_vector_ulong_set_all (gdl_vector_ulong * v, unsigned long x);
int gdl_vector_ulong_set_basis (gdl_vector_ulong * v, size_t i);

gdl_vector_ulong * gdl_vector_ulong_fread (FILE * stream);
int gdl_vector_ulong_fwrite (FILE * stream, const gdl_vector_ulong * v);
gdl_vector_ulong * gdl_vector_ulong_fscanf (FILE * stream);
int gdl_vector_ulong_fprintf (FILE * stream, const gdl_vector_ulong * v,
                              const char *format);

int gdl_vector_ulong_memcpy (gdl_vector_ulong * dest, const gdl_vector_ulong * src);

int gdl_vector_ulong_reverse (gdl_vector_ulong * v);

int gdl_vector_ulong_swap (gdl_vector_ulong * v, gdl_vector_ulong * w);
int gdl_vector_ulong_swap_elements (gdl_vector_ulong * v, const size_t i, const size_t j);

unsigned long gdl_vector_ulong_max (const gdl_vector_ulong * v);
unsigned long gdl_vector_ulong_min (const gdl_vector_ulong * v);
void gdl_vector_ulong_minmax (const gdl_vector_ulong * v, unsigned long * min_out, unsigned long * max_out);

size_t gdl_vector_ulong_max_index (const gdl_vector_ulong * v);
size_t gdl_vector_ulong_min_index (const gdl_vector_ulong * v);
void gdl_vector_ulong_minmax_index (const gdl_vector_ulong * v, size_t * imin, size_t * imax);

int gdl_vector_ulong_add (gdl_vector_ulong * a, const gdl_vector_ulong * b);
int gdl_vector_ulong_sub (gdl_vector_ulong * a, const gdl_vector_ulong * b);
int gdl_vector_ulong_mul (gdl_vector_ulong * a, const gdl_vector_ulong * b);
int gdl_vector_ulong_div (gdl_vector_ulong * a, const gdl_vector_ulong * b);
int gdl_vector_ulong_scale (gdl_vector_ulong * a, const double x);
int gdl_vector_ulong_add_constant (gdl_vector_ulong * a, const double x);

int gdl_vector_ulong_isnull (const gdl_vector_ulong * v);

#ifdef HAVE_INLINE

extern inline
unsigned long
gdl_vector_ulong_get (const gdl_vector_ulong * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
    }
#endif
  return v->data[i * v->stride];
}

extern inline
void
gdl_vector_ulong_set (gdl_vector_ulong * v, const size_t i, unsigned long x)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
    }
#endif
  v->data[i * v->stride] = x;
}

extern inline
unsigned long *
gdl_vector_ulong_ptr (gdl_vector_ulong * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (unsigned long *) (v->data + i * v->stride);
}

extern inline
const unsigned long *
gdl_vector_ulong_const_ptr (const gdl_vector_ulong * v, const size_t i)
{
#if GDL_RANGE_CHECK
  if (i >= v->size)
    {
      GDL_ERROR_NULL ("index out of range", GDL_EINVAL);
    }
#endif
  return (const unsigned long *) (v->data + i * v->stride);
}


#endif /* HAVE_INLINE */

__END_DECLS

#endif /* __GDL_VECTOR_ULONG_H__ */


