/*  
 * 	view/view.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_hmap.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_view.h>

gdl_view *
gdl_view_alloc ()
{
	gdl_view * v = GDL_CALLOC (gdl_view, 1);
	
	v->_ginit = gdl_true;
	v->_tinit = gdl_true;
	v->_oinit = gdl_true;
	
	return v;
}

void
gdl_view_free (gdl_view * v)
{
	if (v)
	{
		gdl_fview_free (v->tdata);
		gdl_fview_free (v->odata);
		gdl_gview_free (v->gdata);
		gdl_hview_free (v->hdata);
		gdl_gmap_free (v->gmap);
		gdl_hmap_free (v->hmap);
		gdl_mask_free (v->hdata_mask);
		gdl_mask_free (v->hmap_mask);
		gdl_locus_type_registry_free (v->locus_type);
		GDL_FREE (v);
	}
}

int
gdl_view_add_gview (gdl_view * v, const gdl_string * name, gdl_gview * g)
{
	if (v->_ginit)
	{
		size_t p = gdl_gview_ploidy (g);
		
		v->gdata = gdl_gview_alloc (gdl_gview_virtual);
		gdl_gview_init (v->gdata, 0, 0, p);
		
		v->_ginit = gdl_false;
	}
	return gdl_gview_add (v->gdata, name, g, 1);
}

int
gdl_view_add_tview (gdl_view * v, const gdl_string * name, gdl_fview * t)
{
	if (v->_tinit)
	{
		v->tdata = gdl_fview_alloc (gdl_fview_virtual);
		v->_tinit = gdl_false;
	}
	return gdl_fview_add (v->tdata, name, t, 1);
}

int
gdl_view_add_oview (gdl_view * v, const gdl_string * name, gdl_fview * o)
{
	if (v->_oinit)
	{
		v->odata = gdl_fview_alloc (gdl_fview_virtual);
		v->_oinit = gdl_false;
	}
	return gdl_fview_add (v->odata, name, o, 1);	
}

void
gdl_view_set_gview (gdl_view * v, gdl_gview * g)
{
	v->gdata = g;
}

void
gdl_view_set_hview (gdl_view * v, gdl_hview * h)
{
	v->hdata = h;
	v->hdata_mask = gdl_mask_clone (gdl_hview_get_gmask (h));
}

void
gdl_view_set_gmap (gdl_view * v, gdl_gmap * g)
{
	v->gmap = g;	
}

void
gdl_view_set_hmap (gdl_view * v, gdl_hmap * h)
{
	v->hmap = h;
	v->hmap_mask = gdl_mask_clone (gdl_hmap_get_gmask (h));
}

gdl_gview *
gdl_view_get_gview (const gdl_view * v)
{
	return v->gdata;	
}

gdl_fview *
gdl_view_get_tview (const gdl_view * v)
{
	return v->tdata;	
}

gdl_fview *
gdl_view_get_oview (const gdl_view * v)
{
	return v->odata;	
}

gdl_hview *
gdl_view_get_hview (const gdl_view * v)
{
	return v->hdata;
}

gdl_gmap *
gdl_view_get_gmap (const gdl_view * v)
{
	return v->gmap;
}

gdl_hmap *
gdl_view_get_hmap (const gdl_view * v)
{
	return v->hmap;	
}

gdl_gmap *
gdl_view_extract_gmap (const gdl_view * v, const gdl_mask * m)
{
	if (v->gdata && v->gmap)
	{
		return gdl_gview_extract_gmap (v->gdata, m, v->gmap);
	}
	return NULL;
}

gdl_gmap *
gdl_view_extract_hgmap (const gdl_view * v, const gdl_mask * m)
{
	if (v->gdata && v->hmap)
	{
		const gdl_gmap * gmap = gdl_hmap_get_gmap (v->hmap);
		return gdl_gview_extract_gmap (v->gdata, m, gmap);
	}
	return NULL;
}

gdl_gmap *
gdl_view_extract_gmap_clone (const gdl_view * v, const gdl_mask * m)
{
	if (v->gdata && v->gmap)
	{
		return gdl_gview_extract_gmap_clone (v->gdata, m, v->gmap);
	}
	return NULL;
}

gdl_gmap *
gdl_view_extract_hgmap_clone (const gdl_view * v, const gdl_mask * m)
{
	if (v->gdata && v->hmap)
	{
		const gdl_gmap * gmap = gdl_hmap_get_gmap (v->hmap);
		return gdl_gview_extract_gmap_clone (v->gdata, m, gmap);
	}
	return NULL;
}

int
gdl_view_add_locus_type (gdl_view * v, const gdl_locus_type_registry * r)
{
	if (v && r)
	{
		if (!v->locus_type)
		{
			v->locus_type = gdl_locus_type_registry_alloc ();
		}
		
		gdl_locus_type_registry_merge (v->locus_type, r);

		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

gdl_locus_type_registry *
gdl_view_get_locus_type (const gdl_view * v)
{
	return v->locus_type;
}
