/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package exenne.components.treetable.dnd;

import exenne.components.treetable.Treetable;
import exenne.components.transfer.PackObjects;
import java.awt.Component;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.Iterator;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.TransferHandler;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

/**
 * Clasa ce face drag an drop la un treetable
 * @author iulian
 */
public class TreetableTransferHandler extends TransferHandler {
    //Treetable-ul pentru care e construit
    protected TransferDataProvider dataProvider;
    protected PackObjects localPack;

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Constructia Transferabilului
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * Construiesc un transfer hadler legat de treetable
     */
    public TreetableTransferHandler(TransferDataProvider dataProvider) {
        this.dataProvider = dataProvider;
        this.localPack = null;
    }

    /**
     * Obtin actiunile care se pot face
     * @param comp
     * @return
     */
    @Override
    public int getSourceActions(JComponent comp) {
        Treetable treetable = dataProvider.getTreetable();
        if (treetable.equals(comp))
            return dataProvider.getSourceActions();
        return NONE;
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Creeare pachet
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * DRAG / COPY:
     * Creez obiectul care va contine datele de transfer
     */
    @Override
    public Transferable createTransferable(JComponent comp) {
        Treetable treetable = dataProvider.getTreetable();
        PackObjects pack; //pachetul de obiecte

        //Deleg responsabilitatea creearii pachetului catre dataProvider
        if (treetable.equals(comp)) {
            //Obtin pachet din data provider
            pack = dataProvider.createPack();
            if (pack == null) return null;

            //Marchez ownerul
            pack.setOwner(treetable);

            //Creez transferabilul
            localPack = pack;
            if (pack.size() == 0) return null;
            else return new TransferPackObjects(pack);
        } else return null;
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Etapa de EXPORT
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * DRAG / COPY:
     * Metoda de cleanup dupa impachetare
     */
    @Override
    public void exportDone(JComponent comp, Transferable trans, int action) {
         Treetable treetable = dataProvider.getTreetable();

        //Verific daca e vorba de aceiasi componenta
        if (!treetable.equals(comp))
            return;

        //Continui cu exportul
        try {
            //Daca actiunea este diferita de move, obiectele initialenu sunt afectate
            if (action != MOVE) {
                return;
            }

            //Daca actiunea este MOVE atunci trebuie sa sterg obiectele initiale
            //asta doar daca nu au fost sterse deja
            if (!dataProvider.isInhibitExport()) {
                DataFlavor dataFlavor = new DataFlavor(PackObjects.class, dataProvider.getFlavourName());
                PackObjects packObjects = (PackObjects)
                        trans.getTransferData(dataFlavor);

                //Daca este chiar acelasi obiect folosesc pachetul local
                if (packObjects.isOwner(treetable))
                    packObjects = localPack;

                //Deleg actiunea de stergere in data provider
                dataProvider.exportDone(packObjects);
            }
        } catch (UnsupportedFlavorException ex) {
            ex.printStackTrace();
            System.err.println("Unsupported Flavor");
        } catch (IOException ex) {
            ex.printStackTrace();
            System.err.println("Eroare la operatii IO (Serializabil)");
        } catch (Exception ex) {
            ex.printStackTrace();
            System.err.println("Unusual exception in DnD");
        }
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Etapa de CAN IMPORT
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

    /**
     * DROP / PASTE:
     *  Metoda apelata tot timpul de catre DnD
     *  pentru a accepta transferul
     */
    @Override
    public boolean canImport(TransferSupport support) {
        Treetable treetable = dataProvider.getTreetable();
        Component comp = support.getComponent();
        //Verific daca e vorba de aceiasi componenta
        if (!treetable.equals(comp))
            return false;

        DefaultTreeModel treeModel = (DefaultTreeModel)
                treetable.getOutlineModel().getTreeModel();
        boolean isDropOn = false;

        //we only support one data flavor
        DataFlavor dataFlavor = new DataFlavor(PackObjects.class,dataProvider.getFlavourName());
        if (!support.isDataFlavorSupported(dataFlavor)) {
            return false;
        }

        // Verific ce materiale am in packet
        try {
            //Obtin packetul transportat
            PackObjects packObjects = (PackObjects)
                support.getTransferable().
                getTransferData(dataFlavor);

            //Verific ca pachetul sa nu fie null
            if (packObjects == null)
                return false;

            // Daca pachetul este local, iar locatia in care s-a tras este diferita
            // decat ownerul sau, atunci nu se poate continua
            if (!dataProvider.isGlobalPack())
                if (!packObjects.isOwner(treetable)) return false;

            //Daca este chiar acelasi obiect folosesc pachetul local
            if (packObjects.isOwner(treetable))
                packObjects = localPack;

            //Obtin locatia de drop
            int row;
            if (support.isDrop()) {
                JTable.DropLocation dl = (JTable.DropLocation)support.getDropLocation();
                row = dl.getRow();
                if (!dl.isInsertRow()) isDropOn = true;
            } else row = treetable.getSelectedRow();
            if (row < 0 ) row = treetable.getRowCount();
            
            //Identific zona de drop
            DefaultMutableTreeNode node = ((DefaultMutableTreeNode) treetable.getValueAt(row, 0));

            //Decid cum trebuie sa se comporte la Paste
            if (!support.isDrop() && (node != null)) {
                if (dataProvider.isTreatPasteAsDropOn(packObjects, node))
                    isDropOn = true;
            }

             //Decid cum trebuie sa se comporte la dropOn
            if (isDropOn && (node != null))
            {
                if (dataProvider.isTreatDropOnAsNear(packObjects, node))
                    isDropOn = false;
            }           

            //Identific parintele
            DefaultMutableTreeNode parentNode;
            if (node == null) {
                parentNode = (DefaultMutableTreeNode) treeModel.getRoot();
            } else {
                if (isDropOn) {
                    parentNode = (DefaultMutableTreeNode) node;
                } else {
                    parentNode = (DefaultMutableTreeNode) node.getParent();
                }
            }

            //Elimin cazul in care un element se poate adauga la el insusi
            Iterator <Object> it = packObjects.iterator();
            while (it.hasNext()) {
                Object item = it.next();
                Integer nodeHash = (Integer) packObjects.getNodeHash(item);
                if (nodeHash != null && 
                        findNodeInTree(nodeHash,parentNode)) return false;
            }

            // Trimit nodul parinte si pachetul pentru a fi verifica de catre
            // data provider
            return dataProvider.canImport(packObjects, parentNode);

        } catch (UnsupportedFlavorException ex) {
            ex.printStackTrace();
            System.err.println("Unsupported Flavor");
        } catch (IOException ex) {
            ex.printStackTrace();
            System.err.println("Eroare la operatii IO (Serializabil)");
        } catch (Exception ex) {
            ex.printStackTrace();
            System.err.println("Unusual exception in DnD");
        }
        return false;
    }

    /**
     * Functie recursiva pentru a gasi hash-ul unui nod in arbore
     */
    private boolean findNodeInTree(Integer res,
            DefaultMutableTreeNode node) {
        if (node.isRoot()) return false;
        if ( res.intValue() == node.hashCode() ) return true;
        else return findNodeInTree(res, (DefaultMutableTreeNode)node.getParent());
    }

    /**~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *      Etapa de IMPORT
     *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

     /**
     * DROP / PASTE:
     *  Realizeaza efectiv transferul datelor catre obiect
     */
    @Override
    public boolean importData(TransferSupport support) {
        Treetable treetable = dataProvider.getTreetable();
        Component comp = support.getComponent();
        //Verific daca e vorba de aceiasi componenta
        if (!treetable.equals(comp))
            return false;

        //Variabile de context
        DefaultTreeModel treeModel = (DefaultTreeModel)
                treetable.getOutlineModel().getTreeModel();
        boolean isDropOn = false;

        // if we can't handle the import, say so
        if (!canImport(support)) {
            return false;
        }

        try {
            DataFlavor dataFlavor = new DataFlavor(PackObjects.class,
                    dataProvider.getFlavourName());
            //Obtin packetul transportat
            PackObjects packObjects = (PackObjects)
                support.getTransferable().
                getTransferData(dataFlavor);

            // Daca pachetul este local, iar locatia in care s-a tras este diferita
            // decat ownerul sau, atunci nu se poate continua
            if (!dataProvider.isGlobalPack())
                if (!packObjects.isOwner(treetable)) return false;

            //Daca este chiar acelasi obiect folosesc pachetul local
            if (packObjects.isOwner(treetable))
                packObjects = localPack;

            // fetch the drop location
            int row;
            if (support.isDrop()) {
                JTable.DropLocation dl = (JTable.DropLocation)support.getDropLocation();
                row = dl.getRow();
                if (!dl.isInsertRow()) isDropOn = true;
            } else row = treetable.getSelectedRow();
            if (row < 0 ) row = treetable.getRowCount();

            //Obtin locatia de drop
            DefaultMutableTreeNode node = ((DefaultMutableTreeNode) treetable.getValueAt(row, 0));

            //Decid cum trebuie sa se comporte la Paste
            if (!support.isDrop()) {
                if (dataProvider.isTreatPasteAsDropOn(packObjects, node))
                    isDropOn = true;
            }

            //Decid cum trebuie sa se comporte la dropOn
            if (isDropOn && (node != null))
            {
                if (dataProvider.isTreatDropOnAsNear(packObjects, node))
                    isDropOn = false;
            }

            //Identific parintele
            DefaultMutableTreeNode parentNode = null;
            if (node == null)
                //Inseamna ca adaug la sfarsit
                parentNode =(DefaultMutableTreeNode) treeModel.getRoot();
             else
             {
                if (isDropOn) parentNode = node;
                else parentNode = (DefaultMutableTreeNode) node.getParent();
             }

            //Indexul copilului in parinte
            int dropNodeIndex;
            if (node != null) {
                if (isDropOn) dropNodeIndex = parentNode.getChildCount();
                else dropNodeIndex= parentNode.getIndex(node);
            }
            else dropNodeIndex = parentNode.getChildCount();

            //Nodul propriu-zis al dropului
            DefaultMutableTreeNode dropNode;
            if (isDropOn) dropNode = null;
            else dropNode = node;

            //Deleg actiunea de import propriu-zis catre dataProvider
            boolean isMove = false;
            if (support.isDrop()) {
                //Doar daca actiunea este drop iau in considerare caz de mutare
                int action = support.getDropAction();
                if (action == TransferHandler.MOVE) isMove = true;

                //Daca nu este drop, la copy-paste nu exista mutare
                //intai se sterge obiectul
            }

            //La import ster selectia
            treetable.clearSelection();

            //Fac importul propriu-zis
            boolean importIsDone = dataProvider.importData(packObjects, parentNode,
                    dropNode, dropNodeIndex, isMove);
            if (!importIsDone) return false;

            return true;
        } catch (UnsupportedFlavorException ex) {
            ex.printStackTrace();
            System.err.println("Unsupported Flavor");
        } catch (IOException ex) {
            ex.printStackTrace();
            System.err.println("Eroare la operatii IO (Serializabil)");
        } catch (Exception ex) {
            ex.printStackTrace();
            System.err.println("Unusual exception in DnD");
        }
        return false;
    }
}


