/*
 *                 Sun Public License Notice
 * 
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 * 
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2004 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.util;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;

/** Otherwise uncategorized useful static methods.
*
* @author Jan Palka, Ian Formanek, Jaroslav Tulach
*/
public final class QueueUtilities {
    private QueueUtilities() {}

    /** variable holding the activeReferenceQueue */
    private static ReferenceQueue activeReferenceQueue;
    
    /** Useful queue for all parts of system that use <code>java.lang.ref.Reference</code>s
     * together with some <code>ReferenceQueue</code> and need to do some clean up
     * when the reference is enqued. Usually, in order to be notified about that, one 
     * needs to either create a dedicated thread that blocks on the queue and is 
     * <code>Object.notify</code>-ed, which is the right approach but consumes 
     * valuable system resources (threads) or one can peridically check the content
     * of the queue by <code>RequestProcessor.Task.schedule</code> which is 
     * completelly wrong, because it wakes up the system every (say) 15 seconds.
     * In order to provide useful support for this problem, this queue has been
     * provided.
     * <P>
     * If you have a reference that needs clean up, make it implement <link>Runnable</link>
     * inteface and register it with the <code>activeReferenceQueue</code>:
     * <PRE>
     * class MyReference extends WeakReference implements Runnable {
     *   private Object dataToCleanUp;
     *
     *   public MyReference (Object ref, Object data) {
     *     super (ref, Utilities.activeReferenceQueue ()); // here you specify the queue
     *     dataToCleanUp = data;
     *   }
     *
     *   public void run () {
     *     // clean up your data
     *   }
     * }
     * </PRE>
     * When the <code>ref</code> object is garbage collected, your run method
     * will be invoked by calling 
     * <code>((Runnable)reference).run ()</code>
     * and you can perform what ever cleanup is necessary. Be sure not to block
     * in such cleanup for a long time as this prevents other waiting references 
     * to cleanup themselves.
     * <P>
     * Please do not call any methods of the ReferenceQueue yourself. They
     * will throw exceptions.
     *
     * @since 3.11
     */
    public static synchronized ReferenceQueue activeReferenceQueue () {
        if (activeReferenceQueue == null) {
            activeReferenceQueue = new ActiveQueue (false);
        }
        return activeReferenceQueue;
    }
    
    /** Implementation of the active queue.
     */
    private static final class ActiveQueue extends ReferenceQueue 
    implements Runnable {
        private boolean running;
        private boolean deprecated;
        
        public ActiveQueue (boolean deprecated) {
            this.deprecated = deprecated;
            
            Thread t = new Thread (this, "Active Reference Queue Daemon"); // NOI18N
            t.setPriority(Thread.MIN_PRIORITY);
            t.setDaemon(true); // to not prevent exit of VM
            t.start ();
        }
            
        @Override
        public Reference poll() {
            throw new java.lang.UnsupportedOperationException ();
        }
        
        @Override
        public Reference remove(long timeout) throws IllegalArgumentException, InterruptedException {
            throw new java.lang.InterruptedException ();
        }
        
        @Override
        public Reference remove() throws InterruptedException {
            throw new java.lang.InterruptedException ();
        }
        
        /** Called either from Thread.run or RequestProcessor.post. In first case
         * calls scanTheQueue (only once) in the second and nexts calls cleanTheQueue
         */
        public void run () {
            synchronized (this) {
                if (running) {
                    return;
                }
                running = true;
            }
            
            for (;;) {
                try {
                    Reference ref = super.remove (0);
                    if (! (ref instanceof Runnable)) {
                        continue;
                    }
                    
                    if (deprecated) {
                    }
                        
                    
                    // do the cleanup
                    try {
                        ((Runnable)ref).run ();
                    } catch (ThreadDeath td) {
                        throw td;
                    } catch (Throwable t) {
			// Should not happen.
			// If it happens, it is a bug in client code, notify!
                    } finally {
                        // to allow GC
                        ref = null;
                    }
                } catch (InterruptedException ex) {
                }
            }
        }
    }
}
